<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Production;
use App\Models\Material;
use App\Models\Sparepart;
use Illuminate\Support\Facades\DB;

class FixProductionStock extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'production:fix-stock {--production-id= : ID produksi spesifik} {--dry-run : Simulasi tanpa update database}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Perbaiki stok untuk produksi yang sudah selesai tapi stok belum dikurangi';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $productionId = $this->option('production-id');
        $dryRun = $this->option('dry-run');

        if ($dryRun) {
            $this->info('🔍 DRY RUN MODE - Tidak akan mengupdate database');
        }

        // Ambil produksi yang sudah selesai atau dalam proses
        $query = Production::with(['productionMaterials.material', 'productionSpareparts.sparepart'])
            ->whereIn('status', ['dalam_proses', 'selesai']);

        if ($productionId) {
            $query->where('id', $productionId);
        }

        $productions = $query->get();

        if ($productions->isEmpty()) {
            $this->warn('Tidak ada produksi yang ditemukan.');
            return 0;
        }

        $this->info("Menemukan {$productions->count()} produksi untuk diperiksa...\n");

        $totalFixed = 0;
        $totalSkipped = 0;
        $totalErrors = 0;

        foreach ($productions as $production) {
            $this->line("📦 Memproses Produksi #{$production->id} (Status: {$production->status})");
            
            $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
            $startDate = $production->start_date ?? $production->actual_start_date;

            if (!$startDate) {
                $this->warn("  ⚠️  Produksi #{$production->id} tidak memiliki start_date, skip");
                $totalSkipped++;
                continue;
            }

            $fixed = 0;
            $errors = 0;

            try {
                DB::beginTransaction();

                // Process materials
                foreach ($production->productionMaterials as $pm) {
                    if (!$pm->material_id || !$pm->material) {
                        $this->warn("  ⚠️  Material ID {$pm->id} tidak memiliki material_id atau material tidak ditemukan");
                        continue;
                    }

                    $needed = $pm->quantity * $orderQty;
                    $material = $pm->material;

                    // Cek apakah stok sudah dikurangi
                    // Jika item additional dan created_at >= start_date, stok sudah dikurangi saat ditambahkan
                    $shouldReduce = true;
                    if ($pm->is_additional && $pm->created_at && $startDate) {
                        if ($pm->created_at >= $startDate) {
                            $shouldReduce = false;
                            $this->line("  ✓ Material '{$material->name}' sudah dikurangi saat ditambahkan (created_at: {$pm->created_at->format('Y-m-d H:i')} >= start_date: {$startDate->format('Y-m-d H:i')})");
                        }
                    }

                    if ($shouldReduce) {
                        // Semua item (biasa atau additional yang created_at < start_date) harus dikurangi
                        if ($dryRun) {
                            $this->info("  🔄 [DRY RUN] Akan mengurangi stok '{$material->name}' sebanyak {$needed} (stok saat ini: {$material->stock}, is_additional: " . ($pm->is_additional ? 'true' : 'false') . ", created_at: " . ($pm->created_at ? $pm->created_at->format('Y-m-d H:i') : 'null') . ")");
                            $fixed++;
                        } else {
                            // Lock material
                            $materialLocked = Material::lockForUpdate()->find($material->id);
                            
                            $oldStock = $materialLocked->stock;
                            if ($oldStock >= $needed) {
                                $materialLocked->decrement('stock', $needed);
                                $newStock = $materialLocked->fresh()->stock;
                                $this->info("  ✅ Mengurangi stok '{$material->name}' sebanyak {$needed} (stok: {$oldStock} → {$newStock})");
                                $fixed++;
                            } else {
                                $this->warn("  ⚠️  Stok '{$material->name}' tidak cukup untuk dikurangi (stok: {$oldStock}, butuh: {$needed}) - mungkin sudah dikurangi sebelumnya");
                                // Mungkin stok sudah dikurangi, jadi kita anggap sudah benar
                                $fixed++;
                            }
                        }
                    }
                }

                // Process spareparts
                foreach ($production->productionSpareparts as $ps) {
                    if (!$ps->sparepart_id || !$ps->sparepart) {
                        $this->warn("  ⚠️  Sparepart ID {$ps->id} tidak memiliki sparepart_id atau sparepart tidak ditemukan");
                        continue;
                    }

                    $needed = $ps->quantity * $orderQty;
                    $sparepart = $ps->sparepart;

                    // Cek apakah stok sudah dikurangi
                    // Jika item additional dan created_at >= start_date, stok sudah dikurangi saat ditambahkan
                    $shouldReduce = true;
                    if ($ps->is_additional && $ps->created_at && $startDate) {
                        if ($ps->created_at >= $startDate) {
                            $shouldReduce = false;
                            $this->line("  ✓ Sparepart '{$sparepart->name}' sudah dikurangi saat ditambahkan (created_at: {$ps->created_at->format('Y-m-d H:i')} >= start_date: {$startDate->format('Y-m-d H:i')})");
                        }
                    }

                    if ($shouldReduce) {
                        // Semua item (biasa atau additional yang created_at < start_date) harus dikurangi
                        if ($dryRun) {
                            $this->info("  🔄 [DRY RUN] Akan mengurangi stok '{$sparepart->name}' sebanyak {$needed} (stok saat ini: {$sparepart->stock}, is_additional: " . ($ps->is_additional ? 'true' : 'false') . ", created_at: " . ($ps->created_at ? $ps->created_at->format('Y-m-d H:i') : 'null') . ")");
                            $fixed++;
                        } else {
                            // Lock sparepart
                            $sparepartLocked = Sparepart::lockForUpdate()->find($sparepart->id);
                            
                            $oldStock = $sparepartLocked->stock;
                            if ($oldStock >= $needed) {
                                $sparepartLocked->decrement('stock', $needed);
                                $newStock = $sparepartLocked->fresh()->stock;
                                $this->info("  ✅ Mengurangi stok '{$sparepart->name}' sebanyak {$needed} (stok: {$oldStock} → {$newStock})");
                                $fixed++;
                            } else {
                                $this->warn("  ⚠️  Stok '{$sparepart->name}' tidak cukup untuk dikurangi (stok: {$oldStock}, butuh: {$needed}) - mungkin sudah dikurangi sebelumnya");
                                // Mungkin stok sudah dikurangi, jadi kita anggap sudah benar
                                $fixed++;
                            }
                        }
                    }
                }

                if (!$dryRun) {
                    DB::commit();
                }

                $totalFixed += $fixed;
                $totalErrors += $errors;

                if ($fixed > 0 || $errors > 0) {
                    $this->line("  📊 Produksi #{$production->id}: {$fixed} item diperbaiki, {$errors} error\n");
                } else {
                    $this->line("  ✓ Produksi #{$production->id}: Tidak ada yang perlu diperbaiki\n");
                    $totalSkipped++;
                }

            } catch (\Exception $e) {
                if (!$dryRun) {
                    DB::rollBack();
                }
                $this->error("  ❌ Error pada produksi #{$production->id}: " . $e->getMessage());
                $totalErrors++;
            }
        }

        $this->newLine();
        $this->info("📊 Ringkasan:");
        $this->info("  ✅ Diperbaiki: {$totalFixed} item");
        $this->info("  ⚠️  Error: {$totalErrors} item");
        $this->info("  ⏭️  Di-skip: {$totalSkipped} produksi");

        return 0;
    }
}
