<?php

namespace App\Console\Commands;

use App\Models\Order;
use App\Services\NotificationService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use App\Mail\HandoverDeliveredEmail;
use Carbon\Carbon;

class SendHandoverReminder extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'handover:send-reminder 
                            {--days=3 : Jumlah hari setelah handover delivered untuk mengirim reminder}
                            {--dry-run : Jalankan tanpa mengirim email/notifikasi}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Kirim reminder otomatis ke customer yang belum mengkonfirmasi serah terima';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $days = (int) $this->option('days');
        $dryRun = $this->option('dry-run');

        $this->info("Mencari order yang sudah delivered lebih dari {$days} hari dan belum dikonfirmasi...");

        // Cari order yang:
        // 1. Handover status = 'delivered'
        // 2. Belum dikonfirmasi customer (customer_confirmed_at = null)
        // 3. Sudah delivered lebih dari X hari
        // 4. Produksi sudah selesai
        $cutoffDate = Carbon::now()->subDays($days);

        $orders = Order::where('handover_status', 'delivered')
            ->whereNull('customer_confirmed_at')
            ->whereHas('production', function($q) {
                $q->where('status', 'selesai')
                  ->where('completion_status', 'approved');
            })
            ->where(function($q) use ($cutoffDate) {
                $q->where('handover_at', '<=', $cutoffDate)
                  ->orWhere(function($sub) use ($cutoffDate) {
                      // Jika handover_at null, gunakan created_at dari production completion
                      $sub->whereNull('handover_at')
                          ->whereHas('production', function($prod) use ($cutoffDate) {
                              $prod->where('completion_approved_at', '<=', $cutoffDate);
                          });
                  });
            })
            ->with(['product', 'production'])
            ->get();

        $this->info("Ditemukan {$orders->count()} order yang perlu di-remind.");

        if ($orders->isEmpty()) {
            $this->info('Tidak ada order yang perlu di-remind.');
            return Command::SUCCESS;
        }

        $notificationService = new NotificationService();
        $sentCount = 0;
        $failedCount = 0;

        foreach ($orders as $order) {
            try {
                if ($dryRun) {
                    $this->line("DRY RUN: Akan mengirim reminder untuk Order #{$order->id} - {$order->customer_name}");
                    $sentCount++;
                    continue;
                }

                // Kirim email reminder
                if ($order->customer_email) {
                    Mail::to($order->customer_email)->send(new HandoverDeliveredEmail($order));
                    
                    Log::info('Handover reminder email sent', [
                        'order_id' => $order->id,
                        'customer_email' => $order->customer_email,
                        'days_since_delivered' => $order->handover_at 
                            ? Carbon::parse($order->handover_at)->diffInDays(now()) 
                            : 'unknown',
                    ]);
                }

                // Buat notifikasi untuk user yang terdaftar menggunakan reflection untuk akses protected method
                if ($order->user_id) {
                    $productTitle = $order->product->product_title ?? 'N/A';
                    $message = "Produk pesanan #{$order->id} ({$productTitle}) telah diserahkan lebih dari {$days} hari yang lalu. Silakan konfirmasi penerimaan Anda.";
                    
                    $reflection = new \ReflectionClass($notificationService);
                    $method = $reflection->getMethod('createNotification');
                    $method->setAccessible(true);
                    $method->invoke($notificationService, [
                        'type' => 'order',
                        'subtype' => 'handover_reminder',
                        'title' => 'Reminder: Konfirmasi Serah Terima',
                        'message' => $message,
                        'data' => [
                            'order_id' => $order->id,
                            'product_title' => $productTitle,
                            'confirmation_link' => route('user.orders.confirm-delivery', $order),
                        ],
                        'user_id' => $order->user_id,
                        'related_order_id' => $order->id,
                    ], false);
                }

                $sentCount++;
                $this->info("✓ Reminder dikirim untuk Order #{$order->id} - {$order->customer_name}");

            } catch (\Exception $e) {
                $failedCount++;
                $this->error("✗ Gagal mengirim reminder untuk Order #{$order->id}: " . $e->getMessage());
                
                Log::error('Failed to send handover reminder', [
                    'order_id' => $order->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);
            }
        }

        $this->newLine();
        $this->info("Selesai! {$sentCount} reminder berhasil dikirim, {$failedCount} gagal.");

        return Command::SUCCESS;
    }
}

