<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\BackOfficeTransaction;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class BackOfficeExpenseController extends Controller
{
    /**
     * Display a listing of expenses
     * 
     * @param Request $request
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        // Query builder
        $query = BackOfficeTransaction::expense()
            ->with('creator')
            ->latest('transaction_date');

        // Filter by date range - perbaiki agar bisa filter hanya start_date atau end_date saja
        if ($request->filled('start_date')) {
            $query->whereDate('transaction_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('transaction_date', '<=', $request->end_date);
        }

        // Filter by category
        if ($request->filled('category') && $request->category !== '') {
            $query->byCategory($request->category);
        }

        // Search by description or reference number - perbaiki dengan trim
        if ($request->filled('search') && trim($request->search) !== '') {
            $query->search(trim($request->search));
        }

        // Pagination
        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $expenses = $query->paginate($itemsPerPage)->withQueryString();

        // Calculate total expense (with filters applied)
        $totalExpense = $query->sum('amount');

        // Get categories for filter dropdown
        $categories = BackOfficeTransaction::$expenseCategories;

        return view('admin.backoffice.expense.index', compact(
            'expenses',
            'totalExpense',
            'categories'
        ));
    }

    /**
     * Show the form for creating a new expense
     * 
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $categories = BackOfficeTransaction::$expenseCategories;

        return view('admin.backoffice.expense.create', compact('categories'));
    }

    /**
     * Store a newly created expense in database
     * 
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        // Validation
        $validated = $request->validate([
            'category' => 'required|string|max:50',
            'description' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0|max:999999999999.99',
            'transaction_date' => 'required|date',
            'reference_number' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:1000',
            'attachment' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048', // Max 2MB
        ], [
            'category.required' => 'Kategori harus dipilih',
            'description.required' => 'Deskripsi harus diisi',
            'amount.required' => 'Jumlah harus diisi',
            'amount.numeric' => 'Jumlah harus berupa angka',
            'amount.min' => 'Jumlah tidak boleh negatif',
            'transaction_date.required' => 'Tanggal transaksi harus diisi',
            'attachment.mimes' => 'File harus berformat JPG, PNG, atau PDF',
            'attachment.max' => 'Ukuran file maksimal 2MB',
        ]);

        DB::beginTransaction();
        try {
            // Set type as expense
            $validated['type'] = 'expense';
            $validated['created_by'] = Auth::id();

            // Handle file upload
            if ($request->hasFile('attachment')) {
                $file = $request->file('attachment');
                $filename = time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $file->getClientOriginalName());
                $validated['attachment'] = $file->storeAs('backoffice/expense', $filename, 'uploads');
            }

            // Create transaction
            $expense = BackOfficeTransaction::create($validated);

            DB::commit();

            return redirect()
                ->route('admin.backoffice.expense.index')
                ->with('success', 'Data pengeluaran berhasil ditambahkan!');
        } catch (\Exception $e) {
            DB::rollBack();

            // Delete uploaded file if error
            if (isset($validated['attachment'])) {
                Storage::disk('public')->delete($validated['attachment']);
            }

            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Gagal menambahkan pengeluaran: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified expense
     * 
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $expense = BackOfficeTransaction::expense()
            ->with('creator')
            ->findOrFail($id);

        return view('admin.backoffice.expense.show', compact('expense'));
    }

    /**
     * Show the form for editing the specified expense
     * 
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $expense = BackOfficeTransaction::expense()->findOrFail($id);
        $categories = BackOfficeTransaction::$expenseCategories;

        return view('admin.backoffice.expense.edit', compact('expense', 'categories'));
    }

    /**
     * Update the specified expense in database
     * 
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $id)
    {
        $expense = BackOfficeTransaction::expense()->findOrFail($id);

        // Validation
        $validated = $request->validate([
            'category' => 'required|string|max:50',
            'description' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0|max:999999999999.99',
            'transaction_date' => 'required|date',
            'reference_number' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:1000',
            'attachment' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ], [
            'category.required' => 'Kategori harus dipilih',
            'description.required' => 'Deskripsi harus diisi',
            'amount.required' => 'Jumlah harus diisi',
            'amount.numeric' => 'Jumlah harus berupa angka',
            'amount.min' => 'Jumlah tidak boleh negatif',
            'transaction_date.required' => 'Tanggal transaksi harus diisi',
            'attachment.mimes' => 'File harus berformat JPG, PNG, atau PDF',
            'attachment.max' => 'Ukuran file maksimal 2MB',
        ]);

        DB::beginTransaction();
        try {
            // Handle file upload
            if ($request->hasFile('attachment')) {
                // Delete old file
                if ($expense->attachment) {
                    Storage::disk('public')->delete($expense->attachment);
                }

                $file = $request->file('attachment');
                $filename = time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $file->getClientOriginalName());
                $validated['attachment'] = $file->storeAs('backoffice/expense', $filename, 'uploads');
            }

            // Update transaction
            $expense->update($validated);

            DB::commit();

            return redirect()
                ->route('admin.backoffice.expense.index')
                ->with('success', 'Data pengeluaran berhasil diperbarui!');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Gagal memperbarui pengeluaran: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified expense from database
     * 
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id)
    {
        $expense = BackOfficeTransaction::expense()->findOrFail($id);

        DB::beginTransaction();
        try {
            // Hapus file lampiran jika ada
            if ($expense->attachment) {
                Storage::disk('public')->delete($expense->attachment);
            }

            // Hapus data secara permanen
            $expense->forceDelete();

            DB::commit();

            return redirect()
                ->route('admin.backoffice.expense.index')
                ->with('success', 'Data pengeluaran berhasil dihapus secara permanen!');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()
                ->back()
                ->with('error', 'Gagal menghapus pengeluaran: ' . $e->getMessage());
        }
    }
}
