<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ChartOfAccount;
use App\Models\Setting;
use Illuminate\Http\Request;

class ChartOfAccountController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = ChartOfAccount::query();

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('kode_akun', 'like', "%{$search}%")
                    ->orWhere('nama_akun', 'like', "%{$search}%");
            });
        }

        // Filter by pos_laporan
        if ($request->filled('pos_laporan')) {
            $query->where('pos_laporan', $request->pos_laporan);
        }

        // Filter by pos_saldo
        if ($request->filled('pos_saldo')) {
            $query->where('pos_saldo', $request->pos_saldo);
        }

        // Filter by is_active
        if ($request->filled('is_active')) {
            $query->where('is_active', $request->is_active);
        }

        // Get items per page from settings
        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $accounts = $query->orderBy('sort_order')->orderBy('kode_akun')->paginate($itemsPerPage)->withQueryString();

        return view('admin.chart-of-accounts.index', compact('accounts'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.chart-of-accounts.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'kode_akun' => 'required|string|max:50|unique:chart_of_accounts,kode_akun',
            'nama_akun' => 'required|string|max:255',
            'tabel_bantuan' => 'nullable|string|max:50',
            'pos_saldo' => 'required|in:DEBET,KREDIT',
            'pos_laporan' => 'required|in:NERACA,LABA RUGI',
            'saldo_awal_debet' => 'nullable|numeric|min:0',
            'saldo_awal_kredit' => 'nullable|numeric|min:0',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
            'keterangan' => 'nullable|string',
        ]);

        // Set tabel_bantuan default ke kode_akun jika kosong
        if (empty($validated['tabel_bantuan'])) {
            $validated['tabel_bantuan'] = $validated['kode_akun'];
        }

        ChartOfAccount::create($validated);

        return redirect()->route('admin.chart-of-accounts.index')
            ->with('success', 'Chart of Account berhasil ditambahkan');
    }

    /**
     * Display the specified resource.
     */
    public function show(ChartOfAccount $chartOfAccount)
    {
        $chartOfAccount->loadCount('journalEntries');
        return view('admin.chart-of-accounts.show', compact('chartOfAccount'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ChartOfAccount $chartOfAccount)
    {
        return view('admin.chart-of-accounts.edit', compact('chartOfAccount'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ChartOfAccount $chartOfAccount)
    {
        $validated = $request->validate([
            'kode_akun' => 'required|string|max:50|unique:chart_of_accounts,kode_akun,' . $chartOfAccount->id,
            'nama_akun' => 'required|string|max:255',
            'tabel_bantuan' => 'nullable|string|max:50',
            'pos_saldo' => 'required|in:DEBET,KREDIT',
            'pos_laporan' => 'required|in:NERACA,LABA RUGI',
            'saldo_awal_debet' => 'nullable|numeric|min:0',
            'saldo_awal_kredit' => 'nullable|numeric|min:0',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
            'keterangan' => 'nullable|string',
        ]);

        // Set tabel_bantuan default ke kode_akun jika kosong
        if (empty($validated['tabel_bantuan'])) {
            $validated['tabel_bantuan'] = $validated['kode_akun'];
        }

        $chartOfAccount->update($validated);

        return redirect()->route('admin.chart-of-accounts.index')
            ->with('success', 'Chart of Account berhasil diperbarui');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ChartOfAccount $chartOfAccount)
    {
        // Cek apakah ada journal entries yang menggunakan akun ini
        if ($chartOfAccount->journalEntries()->count() > 0) {
            return redirect()->back()
                ->with('error', 'Tidak dapat menghapus akun yang sudah digunakan dalam jurnal');
        }

        $chartOfAccount->delete();

        return redirect()->route('admin.chart-of-accounts.index')
            ->with('success', 'Chart of Account berhasil dihapus');
    }

    /**
     * Seed akun standar yang dibutuhkan untuk jurnal otomasi proyek
     */
    public function seedStandard(Request $request)
    {
        $templates = [
            [
                'kode_akun' => '101-01',
                'nama_akun' => 'KAS Operasional Kantor',
                'pos_saldo' => 'DEBET',
                'pos_laporan' => 'NERACA',
                'sort_order' => 10101,
            ],
            [
                'kode_akun' => '201-02',
                'nama_akun' => 'Uang Muka Penjualan',
                'pos_saldo' => 'KREDIT',
                'pos_laporan' => 'NERACA',
                'sort_order' => 20102,
            ],
            [
                'kode_akun' => '401-02',
                'nama_akun' => 'Penjualan Jasa',
                'pos_saldo' => 'KREDIT',
                'pos_laporan' => 'LABA RUGI',
                'sort_order' => 40102,
            ],
            [
                'kode_akun' => '501-01',
                'nama_akun' => 'Biaya Material Proyek',
                'pos_saldo' => 'DEBET',
                'pos_laporan' => 'LABA RUGI',
                'sort_order' => 50101,
            ],
            [
                'kode_akun' => '501-02',
                'nama_akun' => 'Biaya Tenaga Kerja Langsung',
                'pos_saldo' => 'DEBET',
                'pos_laporan' => 'LABA RUGI',
                'sort_order' => 50102,
            ],
        ];

        $created = 0;
        foreach ($templates as $tpl) {
            $existing = ChartOfAccount::where('kode_akun', $tpl['kode_akun'])->first();
            if ($existing) {
                continue;
            }
            ChartOfAccount::create([
                'kode_akun' => $tpl['kode_akun'],
                'nama_akun' => $tpl['nama_akun'],
                'tabel_bantuan' => $tpl['kode_akun'],
                'pos_saldo' => $tpl['pos_saldo'],
                'pos_laporan' => $tpl['pos_laporan'],
                'saldo_awal_debet' => 0,
                'saldo_awal_kredit' => 0,
                'is_active' => true,
                'sort_order' => $tpl['sort_order'],
                'keterangan' => 'Template akun standar',
            ]);
            $created++;
        }

        return redirect()->route('admin.chart-of-accounts.index')
            ->with('success', $created > 0 ? 'Akun standar ditambahkan: ' . $created : 'Semua akun standar sudah tersedia');
    }
}
