<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Sale;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\Material;
use App\Models\Sparepart;
use App\Models\Product;
use App\Models\Production;
use App\Models\Setting;
use App\Models\BackOfficeTransaction;
use App\Models\JournalEntry;
use App\Models\ChartOfAccount;
use App\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;

class ReportController extends Controller
{
    /**
     * Laporan Penjualan
     */
    public function sales(Request $request)
    {
        $query = Sale::with(['product', 'user', 'order'])
            ->orderBy('sale_date', 'desc');

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->where('sale_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('sale_date', '<=', $request->end_date);
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Search filter
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $query->where(function ($q) use ($searchTerm) {
                $q->whereHas('product', function($q2) use ($searchTerm) {
                    $q2->where('product_title', 'like', "%{$searchTerm}%");
                })
                ->orWhereHas('order', function($q2) use ($searchTerm) {
                    $q2->where('customer_name', 'like', "%{$searchTerm}%")
                       ->orWhere('customer_email', 'like', "%{$searchTerm}%");
                    if (is_numeric($searchTerm)) {
                        $q2->orWhere('id', $searchTerm);
                    }
                })
                ->orWhereHas('user', function($q2) use ($searchTerm) {
                    $q2->where('name', 'like', "%{$searchTerm}%");
                });
                if (is_numeric($searchTerm)) {
                    $q->orWhere('id', $searchTerm);
                }
            });
        }

        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $sales = $query->paginate($itemsPerPage)->withQueryString();
        
        // Statistik (hitung ulang setelah filter search)
        $statsQuery = Sale::with(['product', 'user', 'order'])
            ->orderBy('sale_date', 'desc');
        
        if ($request->filled('start_date')) {
            $statsQuery->where('sale_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $statsQuery->where('sale_date', '<=', $request->end_date);
        }
        if ($request->filled('status')) {
            $statsQuery->where('status', $request->status);
        }
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $statsQuery->where(function ($q) use ($searchTerm) {
                $q->whereHas('product', function($q2) use ($searchTerm) {
                    $q2->where('product_title', 'like', "%{$searchTerm}%");
                })
                ->orWhereHas('order', function($q2) use ($searchTerm) {
                    $q2->where('customer_name', 'like', "%{$searchTerm}%")
                       ->orWhere('customer_email', 'like', "%{$searchTerm}%");
                    if (is_numeric($searchTerm)) {
                        $q2->orWhere('id', $searchTerm);
                    }
                })
                ->orWhereHas('user', function($q2) use ($searchTerm) {
                    $q2->where('name', 'like', "%{$searchTerm}%");
                });
                if (is_numeric($searchTerm)) {
                    $q->orWhere('id', $searchTerm);
                }
            });
        }
        
        $totalSales = $statsQuery->sum('total_price');
        $totalQuantity = $statsQuery->sum('quantity');
        $totalTransactions = $statsQuery->count();

        // Chart data untuk 12 bulan terakhir
        $chartData = Sale::select(
                DB::raw('MONTH(sale_date) as month'),
                DB::raw('YEAR(sale_date) as year'),
                DB::raw('SUM(total_price) as total'),
                DB::raw('COUNT(*) as count')
            )
            ->where('sale_date', '>=', Carbon::now()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        return view('admin.reports.sales', compact(
            'sales', 
            'totalSales', 
            'totalQuantity', 
            'totalTransactions',
            'chartData'
        ));
    }

    /**
     * Export Laporan Penjualan
     */
    public function salesExport(Request $request)
    {
        $query = Sale::with(['product', 'user', 'order'])
            ->orderBy('sale_date', 'desc');

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->where('sale_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('sale_date', '<=', $request->end_date);
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $sales = $query->get();

        // Generate HTML table dengan styling untuk Excel
        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table {
            border-collapse: collapse;
            width: 100%;
            font-family: Arial, sans-serif;
            font-size: 11px;
        }
        th {
            background-color: #4472C4;
            color: #FFFFFF;
            font-weight: bold;
            text-align: center;
            padding: 10px 8px;
            border: 1px solid #000000;
            vertical-align: middle;
        }
        td {
            padding: 8px;
            border: 1px solid #CCCCCC;
            vertical-align: middle;
        }
        tr:nth-child(even) {
            background-color: #F2F2F2;
        }
        .text-right {
            text-align: right;
        }
        .text-center {
            text-align: center;
        }
        .text-left {
            text-align: left;
        }
    </style>
</head>
<body>
    <table>
        <thead>
            <tr>
                <th>No</th>
                <th>Tanggal</th>
                <th>Kode Order</th>
                <th>Nama Produk</th>
                <th>Customer</th>
                <th class="text-center">Quantity</th>
                <th class="text-right">Harga Satuan</th>
                <th class="text-right">Total Harga</th>
                <th class="text-center">Status</th>
                <th>Metode Pembayaran</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($sales as $index => $sale) {
            $html .= '<tr>';
            $html .= '<td class="text-center">' . ($index + 1) . '</td>';
            $html .= '<td>' . htmlspecialchars($sale->sale_date->format('d/m/Y')) . '</td>';
            $html .= '<td class="text-center">' . htmlspecialchars($sale->order->id ?? '-') . '</td>';
            $html .= '<td>' . htmlspecialchars($sale->product->product_title ?? '-') . '</td>';
            $html .= '<td>' . htmlspecialchars($sale->user->name ?? $sale->order->customer_name ?? '-') . '</td>';
            $html .= '<td class="text-center">' . number_format($sale->quantity, 0, ',', '.') . '</td>';
            $html .= '<td class="text-right">' . number_format($sale->unit_price, 0, ',', '.') . '</td>';
            $html .= '<td class="text-right">' . number_format($sale->total_price, 0, ',', '.') . '</td>';
            $html .= '<td class="text-center">' . htmlspecialchars($sale->status_label ?? '-') . '</td>';
            $html .= '<td>' . htmlspecialchars($sale->payment_method ?? '-') . '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody>
    </table>
</body>
</html>';

        // Generate filename
        $filename = 'laporan_penjualan_' . now()->format('Y-m-d_H-i-s') . '.xls';

        // Return response dengan header untuk Excel
        return response($html, 200)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->header('Cache-Control', 'max-age=0');
    }

    /**
     * Export Laporan Penjualan ke PDF
     */
    public function salesPdf(Request $request)
    {
        $query = Sale::with(['product', 'user', 'order'])
            ->orderBy('sale_date', 'desc');

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->where('sale_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('sale_date', '<=', $request->end_date);
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $sales = $query->get();
        
        // Statistik
        $totalSales = $sales->sum('total_price');
        $totalQuantity = $sales->sum('quantity');
        $totalTransactions = $sales->count();

        $pdf = Pdf::loadView('admin.reports.pdf.sales', compact(
            'sales', 
            'totalSales', 
            'totalQuantity', 
            'totalTransactions'
        ));

        $filename = 'laporan_penjualan_' . date('Y-m-d_H-i-s') . '.pdf';
        
        return $pdf->download($filename);
    }

    /**
     * Laporan Pembelian
     */
    public function purchases(Request $request)
    {
        $query = Purchase::with(['items'])
            ->orderBy('purchase_date', 'desc');

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->where('purchase_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('purchase_date', '<=', $request->end_date);
        }

        // Filter berdasarkan supplier - perbaiki dengan trim
        if ($request->filled('supplier') && trim($request->supplier) !== '') {
            $query->where('supplier_name', 'like', '%' . trim($request->supplier) . '%');
        }

        // Filter berdasarkan tipe pembelian
        if ($request->filled('purchase_type') && $request->purchase_type !== '') {
            $query->where('purchase_type', $request->purchase_type);
        }

        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $purchases = $query->paginate($itemsPerPage)->withQueryString();

        // Statistik (hitung ulang setelah filter)
        $statsQuery = Purchase::with(['items'])
            ->orderBy('purchase_date', 'desc');
        
        if ($request->filled('start_date')) {
            $statsQuery->where('purchase_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $statsQuery->where('purchase_date', '<=', $request->end_date);
        }
        if ($request->filled('supplier') && trim($request->supplier) !== '') {
            $statsQuery->where('supplier_name', 'like', '%' . trim($request->supplier) . '%');
        }
        if ($request->filled('purchase_type')) {
            $statsQuery->where('purchase_type', $request->purchase_type);
        }
        
        $totalAmount = $statsQuery->sum('total_amount');
        $totalTransactions = $statsQuery->count();

        // Detail items untuk chart
        $purchaseItems = PurchaseItem::with(['purchase'])
            ->whereHas('purchase', function($q) use ($request) {
                if ($request->filled('start_date')) {
                    $q->where('purchase_date', '>=', $request->start_date);
                }
                if ($request->filled('end_date')) {
                    $q->where('purchase_date', '<=', $request->end_date);
                }
            })
            ->get();

        $totalItems = $purchaseItems->sum('quantity');
        $totalItemValue = $purchaseItems->sum('total_price');

        // Chart data untuk 12 bulan terakhir
        $chartData = Purchase::select(
                DB::raw('MONTH(purchase_date) as month'),
                DB::raw('YEAR(purchase_date) as year'),
                DB::raw('SUM(total_amount) as total'),
                DB::raw('COUNT(*) as count')
            )
            ->where('purchase_date', '>=', Carbon::now()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        return view('admin.reports.purchases', compact(
            'purchases',
            'totalAmount',
            'totalTransactions',
            'totalItems',
            'totalItemValue',
            'chartData'
        ));
    }

    /**
     * Export Laporan Pembelian
     */
    public function purchasesExport(Request $request)
    {
        $query = Purchase::with(['items'])
            ->orderBy('purchase_date', 'desc');

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->where('purchase_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('purchase_date', '<=', $request->end_date);
        }

        // Filter berdasarkan supplier
        if ($request->filled('supplier')) {
            $query->where('supplier_name', 'like', '%' . $request->supplier . '%');
        }

        // Filter berdasarkan tipe pembelian
        if ($request->filled('purchase_type')) {
            $query->where('purchase_type', $request->purchase_type);
        }

        $purchases = $query->get();

        // Generate HTML table dengan styling untuk Excel
        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table {
            border-collapse: collapse;
            width: 100%;
            font-family: Arial, sans-serif;
            font-size: 11px;
        }
        th {
            background-color: #4472C4;
            color: #FFFFFF;
            font-weight: bold;
            text-align: center;
            padding: 10px 8px;
            border: 1px solid #000000;
            vertical-align: middle;
        }
        td {
            padding: 8px;
            border: 1px solid #CCCCCC;
            vertical-align: middle;
        }
        tr:nth-child(even) {
            background-color: #F2F2F2;
        }
        .text-right {
            text-align: right;
        }
        .text-center {
            text-align: center;
        }
    </style>
</head>
<body>
    <table>
        <thead>
            <tr>
                <th>No</th>
                <th>Tanggal</th>
                <th>Kode Pembelian</th>
                <th>Supplier</th>
                <th class="text-center">Tipe</th>
                <th>Item Dibeli</th>
                <th class="text-right">Total Amount</th>
                <th>Catatan</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($purchases as $index => $purchase) {
            $itemsList = '';
            if ($purchase->items && $purchase->items->count() > 0) {
                $items = $purchase->items->take(5)->map(function($item) {
                    return ($item->item_name ?? '-') . ' (' . $item->quantity . ' ' . ($item->unit ?? 'pcs') . ')';
                })->implode(', ');
                if ($purchase->items->count() > 5) {
                    $items .= ' +' . ($purchase->items->count() - 5) . ' item lainnya';
                }
                $itemsList = $items;
            } else {
                $itemsList = '-';
            }

            $html .= '<tr>';
            $html .= '<td class="text-center">' . ($index + 1) . '</td>';
            $html .= '<td>' . htmlspecialchars($purchase->purchase_date ? $purchase->purchase_date->format('d/m/Y') : '-') . '</td>';
            $html .= '<td>' . htmlspecialchars($purchase->purchase_code) . '</td>';
            $html .= '<td>' . htmlspecialchars($purchase->supplier_name) . '</td>';
            $html .= '<td class="text-center">' . htmlspecialchars(ucfirst($purchase->purchase_type)) . '</td>';
            $html .= '<td>' . htmlspecialchars($itemsList) . '</td>';
            $html .= '<td class="text-right">' . number_format($purchase->total_amount, 0, ',', '.') . '</td>';
            $html .= '<td>' . htmlspecialchars(Str::limit($purchase->notes ?? '-', 50)) . '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody>
    </table>
</body>
</html>';

        // Generate filename
        $filename = 'laporan_pembelian_' . now()->format('Y-m-d_H-i-s') . '.xls';

        // Return response dengan header untuk Excel
        return response($html, 200)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->header('Cache-Control', 'max-age=0');
    }

    /**
     * Export Laporan Pembelian ke PDF
     */
    public function purchasesPdf(Request $request)
    {
        $query = Purchase::with(['items'])
            ->orderBy('purchase_date', 'desc');

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->where('purchase_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('purchase_date', '<=', $request->end_date);
        }

        // Filter berdasarkan supplier
        if ($request->filled('supplier')) {
            $query->where('supplier_name', 'like', '%' . $request->supplier . '%');
        }

        // Filter berdasarkan tipe pembelian
        if ($request->filled('purchase_type')) {
            $query->where('purchase_type', $request->purchase_type);
        }

        $purchases = $query->get();

        // Statistik
        $totalAmount = $purchases->sum('total_amount');
        $totalTransactions = $purchases->count();

        // Detail items untuk statistik
        $purchaseItems = PurchaseItem::with(['purchase'])
            ->whereHas('purchase', function($q) use ($request) {
                if ($request->filled('start_date')) {
                    $q->where('purchase_date', '>=', $request->start_date);
                }
                if ($request->filled('end_date')) {
                    $q->where('purchase_date', '<=', $request->end_date);
                }
            })
            ->get();

        $totalItems = $purchaseItems->sum('quantity');

        $pdf = Pdf::loadView('admin.reports.pdf.purchases', compact(
            'purchases',
            'totalAmount',
            'totalTransactions',
            'totalItems'
        ));

        $filename = 'laporan_pembelian_' . date('Y-m-d_H-i-s') . '.pdf';
        
        return $pdf->download($filename);
    }

    /**
     * Laporan Barang Limit (Stock yang hampir habis)
     */
    public function stockLimit(Request $request)
    {
        // Query untuk materials dengan stock rendah
        $materialsBaseQuery = Material::where('is_active', true)
            ->where('stock', '<=', 10); // Threshold bisa disesuaikan

        if ($request->filled('category') && $request->category !== '') {
            $materialsBaseQuery->where('category', $request->category);
        }

        // Search filter untuk materials
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $materialsBaseQuery->where(function($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('material_code', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%");
            });
        }

        // Pagination untuk materials
        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $materials = (clone $materialsBaseQuery)->orderBy('stock', 'asc')->paginate($itemsPerPage, ['*'], 'materials_page')->withQueryString();

        // Query untuk spareparts dengan stock rendah
        $sparepartsBaseQuery = Sparepart::where('is_active', true)
            ->where('stock', '<=', 5); // Threshold bisa disesuaikan

        if ($request->filled('brand') && $request->brand !== '') {
            $sparepartsBaseQuery->where('brand', $request->brand);
        }

        // Search filter untuk spareparts
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $sparepartsBaseQuery->where(function($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('part_code', 'like', "%{$searchTerm}%")
                  ->orWhere('part_number', 'like', "%{$searchTerm}%");
            });
        }

        // Pagination untuk spareparts
        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $spareparts = (clone $sparepartsBaseQuery)->orderBy('stock', 'asc')->paginate($itemsPerPage, ['*'], 'spareparts_page')->withQueryString();

        // Statistik (menggunakan query terpisah untuk akurasi - semua data, bukan hanya yang di-paginate)
        $totalMaterialsLow = (clone $materialsBaseQuery)->count();
        $totalSparepartsLow = (clone $sparepartsBaseQuery)->count();
        $totalLowStock = $totalMaterialsLow + $totalSparepartsLow;

        // Hitung nilai total stock rendah (menggunakan query terpisah untuk semua data)
        $materialsForStats = (clone $materialsBaseQuery)->get();
        $sparepartsForStats = (clone $sparepartsBaseQuery)->get();

        $totalValueMaterials = $materialsForStats->sum(function($item) {
            return $item->stock * $item->price;
        });

        $totalValueSpareparts = $sparepartsForStats->sum(function($item) {
            return $item->stock * $item->price;
        });

        $totalValueLowStock = $totalValueMaterials + $totalValueSpareparts;

        return view('admin.reports.stock-limit', compact(
            'materials',
            'spareparts',
            'totalMaterialsLow',
            'totalSparepartsLow',
            'totalLowStock',
            'totalValueMaterials',
            'totalValueSpareparts',
            'totalValueLowStock'
        ));
    }

    /**
     * Export Laporan Barang Limit
     */
    public function stockLimitExport(Request $request)
    {
        $materialsBaseQuery = Material::where('is_active', true)
            ->where('stock', '<=', 10);

        if ($request->filled('category')) {
            $materialsBaseQuery->where('category', $request->category);
        }

        $materials = $materialsBaseQuery->orderBy('stock', 'asc')->get();

        $sparepartsBaseQuery = Sparepart::where('is_active', true)
            ->where('stock', '<=', 5);

        if ($request->filled('brand')) {
            $sparepartsBaseQuery->where('brand', $request->brand);
        }

        $spareparts = $sparepartsBaseQuery->orderBy('stock', 'asc')->get();

        // Generate HTML table dengan styling untuk Excel
        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table {
            border-collapse: collapse;
            width: 100%;
            font-family: Arial, sans-serif;
            font-size: 11px;
        }
        th {
            background-color: #4472C4;
            color: #FFFFFF;
            font-weight: bold;
            text-align: center;
            padding: 10px 8px;
            border: 1px solid #000000;
            vertical-align: middle;
        }
        td {
            padding: 8px;
            border: 1px solid #CCCCCC;
            vertical-align: middle;
        }
        tr:nth-child(even) {
            background-color: #F2F2F2;
        }
        .text-right {
            text-align: right;
        }
        .text-center {
            text-align: center;
        }
    </style>
</head>
<body>
    <h2>Materials dengan Stock Rendah (≤ 10)</h2>
    <table>
        <thead>
            <tr>
                <th>No</th>
                <th>Kode</th>
                <th>Nama</th>
                <th class="text-center">Stock</th>
                <th>Unit</th>
                <th class="text-right">Harga</th>
                <th class="text-right">Total Value</th>
                <th>Kategori</th>
                <th class="text-center">Status</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($materials as $index => $material) {
            $html .= '<tr>';
            $html .= '<td class="text-center">' . ($index + 1) . '</td>';
            $html .= '<td>' . htmlspecialchars($material->material_code) . '</td>';
            $html .= '<td>' . htmlspecialchars($material->name) . '</td>';
            $html .= '<td class="text-center">' . $material->stock . '</td>';
            $html .= '<td>' . htmlspecialchars($material->unit) . '</td>';
            $html .= '<td class="text-right">' . number_format($material->price, 0, ',', '.') . '</td>';
            $html .= '<td class="text-right">' . number_format($material->stock * $material->price, 0, ',', '.') . '</td>';
            $html .= '<td>' . htmlspecialchars($material->category) . '</td>';
            $html .= '<td class="text-center">' . ($material->stock <= 5 ? 'Kritis' : 'Rendah') . '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody>
    </table>
    
    <br><br>
    
    <h2>Spareparts dengan Stock Rendah (≤ 5)</h2>
    <table>
        <thead>
            <tr>
                <th>No</th>
                <th>Kode</th>
                <th>Nama</th>
                <th class="text-center">Stock</th>
                <th>Unit</th>
                <th class="text-right">Harga</th>
                <th class="text-right">Total Value</th>
                <th>Brand</th>
                <th class="text-center">Status</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($spareparts as $index => $sparepart) {
            $html .= '<tr>';
            $html .= '<td class="text-center">' . ($index + 1) . '</td>';
            $html .= '<td>' . htmlspecialchars($sparepart->part_code) . '</td>';
            $html .= '<td>' . htmlspecialchars($sparepart->name) . '</td>';
            $html .= '<td class="text-center">' . $sparepart->stock . '</td>';
            $html .= '<td>' . htmlspecialchars($sparepart->unit) . '</td>';
            $html .= '<td class="text-right">' . number_format($sparepart->price, 0, ',', '.') . '</td>';
            $html .= '<td class="text-right">' . number_format($sparepart->stock * $sparepart->price, 0, ',', '.') . '</td>';
            $html .= '<td>' . htmlspecialchars($sparepart->brand) . '</td>';
            $html .= '<td class="text-center">' . ($sparepart->stock <= 2 ? 'Kritis' : 'Rendah') . '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody>
    </table>
</body>
</html>';

        // Generate filename
        $filename = 'laporan_barang_limit_' . now()->format('Y-m-d_H-i-s') . '.xls';

        // Return response dengan header untuk Excel
        return response($html, 200)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->header('Cache-Control', 'max-age=0');
    }

    /**
     * Laporan Barang Rusak
     */
    public function damagedItems(Request $request)
    {
        // Untuk sementara, kita akan menggunakan field 'is_active' = false sebagai indikator barang rusak
        // Atau bisa juga menggunakan tabel terpisah untuk tracking barang rusak
        
        $materialsBaseQuery = Material::hasDamage();
        $sparepartsBaseQuery = Sparepart::hasDamage();
        $productsQuery = Product::where('condition', 'damaged');

        // Filter berdasarkan kategori (untuk materials)
        if ($request->filled('category')) {
            $materialsBaseQuery->where('category', $request->category);
        }

        // Filter berdasarkan brand (untuk spareparts)
        if ($request->filled('brand')) {
            $sparepartsBaseQuery->where('brand', $request->brand);
        }

        // Search filter untuk materials
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $materialsBaseQuery->where(function($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('material_code', 'like', "%{$searchTerm}%");
            });
        }

        // Search filter untuk spareparts
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $sparepartsBaseQuery->where(function($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('part_code', 'like', "%{$searchTerm}%")
                  ->orWhere('part_number', 'like', "%{$searchTerm}%");
            });
        }

        // Pagination untuk materials
        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $materials = (clone $materialsBaseQuery)->orderBy('updated_at', 'desc')->paginate($itemsPerPage, ['*'], 'materials_page')->withQueryString();

        // Pagination untuk spareparts
        $spareparts = (clone $sparepartsBaseQuery)->orderBy('updated_at', 'desc')->paginate($itemsPerPage, ['*'], 'spareparts_page')->withQueryString();

        // Products tetap get() karena biasanya sedikit
        $products = $productsQuery->orderBy('updated_at', 'desc')->get();

        // Statistik (menggunakan query terpisah untuk akurasi - semua data, bukan hanya yang di-paginate)
        $totalDamagedMaterials = (clone $materialsBaseQuery)->count();
        $totalDamagedSpareparts = (clone $sparepartsBaseQuery)->count();
        $totalDamagedProducts = $products->count();
        $totalDamagedItems = $totalDamagedMaterials + $totalDamagedSpareparts + $totalDamagedProducts;

        // Hitung nilai total barang rusak (dari semua data, bukan hanya yang di-paginate)
        $allMaterials = (clone $materialsBaseQuery)->get();
        $allSpareparts = (clone $sparepartsBaseQuery)->get();

        $totalValueMaterials = $allMaterials->sum(function($item) {
            return ($item->damaged_stock ?? 0) * $item->price;
        });

        $totalValueSpareparts = $allSpareparts->sum(function($item) {
            return ($item->damaged_stock ?? 0) * $item->price;
        });

        $totalValueProducts = $products->sum(function($item) {
            return $item->product_prices;
        });

        $totalValueDamaged = $totalValueMaterials + $totalValueSpareparts + $totalValueProducts;

        return view('admin.reports.damaged-items', compact(
            'materials',
            'spareparts',
            'products',
            'totalDamagedMaterials',
            'totalDamagedSpareparts',
            'totalDamagedProducts',
            'totalDamagedItems',
            'totalValueMaterials',
            'totalValueSpareparts',
            'totalValueProducts',
            'totalValueDamaged'
        ));
    }

    /**
     * Export Laporan Barang Rusak
     */
    public function damagedItemsExport(Request $request)
    {
        $materialsQuery = Material::hasDamage();
        $sparepartsQuery = Sparepart::hasDamage();
        $productsQuery = Product::where('condition', 'damaged');

        // Filter berdasarkan kategori (untuk materials)
        if ($request->filled('category')) {
            $materialsQuery->where('category', $request->category);
        }

        // Filter berdasarkan brand (untuk spareparts)
        if ($request->filled('brand')) {
            $sparepartsQuery->where('brand', $request->brand);
        }

        $materials = $materialsQuery->orderBy('updated_at', 'desc')->get();
        $spareparts = $sparepartsQuery->orderBy('updated_at', 'desc')->get();
        $products = $productsQuery->orderBy('updated_at', 'desc')->get();

        // Generate HTML table dengan styling untuk Excel
        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table {
            border-collapse: collapse;
            width: 100%;
            font-family: Arial, sans-serif;
            font-size: 11px;
        }
        th {
            background-color: #4472C4;
            color: #FFFFFF;
            font-weight: bold;
            text-align: center;
            padding: 10px 8px;
            border: 1px solid #000000;
            vertical-align: middle;
        }
        td {
            padding: 8px;
            border: 1px solid #CCCCCC;
            vertical-align: middle;
        }
        tr:nth-child(even) {
            background-color: #F2F2F2;
        }
        .text-right {
            text-align: right;
        }
        .text-center {
            text-align: center;
        }
    </style>
</head>
<body>
    <h2>Materials Rusak</h2>
    <table>
        <thead>
            <tr>
                <th>No</th>
                <th>Kode</th>
                <th>Nama</th>
                <th class="text-center">Stock Total</th>
                <th class="text-center">Stock Rusak</th>
                <th class="text-center">Stock Baik</th>
                <th class="text-center">% Rusak</th>
                <th>Unit</th>
                <th class="text-right">Harga</th>
                <th class="text-right">Nilai Kerugian</th>
                <th>Kategori</th>
                <th>Tanggal Rusak</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($materials as $index => $material) {
            $damagedStock = $material->damaged_stock ?? 0;
            $goodStock = $material->good_stock ?? ($material->stock - $damagedStock);
            $damagePercentage = $material->stock > 0 ? ($damagedStock / $material->stock) * 100 : 0;

            $html .= '<tr>';
            $html .= '<td class="text-center">' . ($index + 1) . '</td>';
            $html .= '<td>' . htmlspecialchars($material->material_code) . '</td>';
            $html .= '<td>' . htmlspecialchars($material->name) . '</td>';
            $html .= '<td class="text-center">' . $material->stock . '</td>';
            $html .= '<td class="text-center">' . $damagedStock . '</td>';
            $html .= '<td class="text-center">' . $goodStock . '</td>';
            $html .= '<td class="text-center">' . number_format($damagePercentage, 1) . '%</td>';
            $html .= '<td>' . htmlspecialchars($material->unit) . '</td>';
            $html .= '<td class="text-right">' . number_format($material->price, 0, ',', '.') . '</td>';
            $html .= '<td class="text-right">' . number_format($damagedStock * $material->price, 0, ',', '.') . '</td>';
            $html .= '<td>' . htmlspecialchars($material->category) . '</td>';
            $html .= '<td>' . ($material->damage_date ? $material->damage_date->format('d/m/Y') : '-') . '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody>
    </table>
    
    <br><br>
    
    <h2>Spareparts Rusak</h2>
    <table>
        <thead>
            <tr>
                <th>No</th>
                <th>Kode</th>
                <th>Nama</th>
                <th class="text-center">Stock Total</th>
                <th class="text-center">Stock Rusak</th>
                <th class="text-center">Stock Baik</th>
                <th class="text-center">% Rusak</th>
                <th>Unit</th>
                <th class="text-right">Harga</th>
                <th class="text-right">Nilai Kerugian</th>
                <th>Brand</th>
                <th>Tanggal Rusak</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($spareparts as $index => $sparepart) {
            $damagedStock = $sparepart->damaged_stock ?? 0;
            $goodStock = $sparepart->good_stock ?? ($sparepart->stock - $damagedStock);
            $damagePercentage = $sparepart->stock > 0 ? ($damagedStock / $sparepart->stock) * 100 : 0;

            $html .= '<tr>';
            $html .= '<td class="text-center">' . ($index + 1) . '</td>';
            $html .= '<td>' . htmlspecialchars($sparepart->part_code) . '</td>';
            $html .= '<td>' . htmlspecialchars($sparepart->name) . '</td>';
            $html .= '<td class="text-center">' . $sparepart->stock . '</td>';
            $html .= '<td class="text-center">' . $damagedStock . '</td>';
            $html .= '<td class="text-center">' . $goodStock . '</td>';
            $html .= '<td class="text-center">' . number_format($damagePercentage, 1) . '%</td>';
            $html .= '<td>' . htmlspecialchars($sparepart->unit) . '</td>';
            $html .= '<td class="text-right">' . number_format($sparepart->price, 0, ',', '.') . '</td>';
            $html .= '<td class="text-right">' . number_format($damagedStock * $sparepart->price, 0, ',', '.') . '</td>';
            $html .= '<td>' . htmlspecialchars($sparepart->brand) . '</td>';
            $html .= '<td>' . ($sparepart->damage_date ? $sparepart->damage_date->format('d/m/Y') : '-') . '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody>
    </table>';

        if ($products->count() > 0) {
            $html .= '<br><br>
    
    <h2>Products Rusak</h2>
    <table>
        <thead>
            <tr>
                <th>No</th>
                <th>ID</th>
                <th>Nama</th>
                <th class="text-right">Harga</th>
                <th>Kategori</th>
                <th class="text-center">Kondisi</th>
                <th>Alasan Rusak</th>
                <th>Tanggal Rusak</th>
            </tr>
        </thead>
        <tbody>';

            foreach ($products as $index => $product) {
                $html .= '<tr>';
                $html .= '<td class="text-center">' . ($index + 1) . '</td>';
                $html .= '<td class="text-center">' . $product->id . '</td>';
                $html .= '<td>' . htmlspecialchars($product->product_title) . '</td>';
                $html .= '<td class="text-right">' . number_format($product->product_prices, 0, ',', '.') . '</td>';
                $html .= '<td>' . htmlspecialchars($product->category->name ?? '-') . '</td>';
                $html .= '<td class="text-center">' . htmlspecialchars(ucfirst($product->condition ?? 'damaged')) . '</td>';
                $html .= '<td>' . htmlspecialchars(Str::limit($product->damage_reason ?? '-', 50)) . '</td>';
                $html .= '<td>' . ($product->damage_date ? $product->damage_date->format('d/m/Y') : '-') . '</td>';
                $html .= '</tr>';
            }

            $html .= '</tbody>
    </table>';
        }

        $html .= '</body>
</html>';

        // Generate filename
        $filename = 'laporan_barang_rusak_' . now()->format('Y-m-d_H-i-s') . '.xls';

        // Return response dengan header untuk Excel
        return response($html, 200)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->header('Cache-Control', 'max-age=0');
    }

    /**
     * Export Laporan Barang Limit ke PDF
     */
    public function stockLimitPdf(Request $request)
    {
        // Ambil materials dengan stock rendah
        $materialsQuery = Material::where('is_active', true)
            ->where('stock', '<=', 10);

        if ($request->filled('category')) {
            $materialsQuery->where('category', $request->category);
        }

        $materials = $materialsQuery->orderBy('stock', 'asc')->get();

        // Ambil spareparts dengan stock rendah
        $sparepartsQuery = Sparepart::where('is_active', true)
            ->where('stock', '<=', 5);

        if ($request->filled('brand')) {
            $sparepartsQuery->where('brand', $request->brand);
        }

        $spareparts = $sparepartsQuery->orderBy('stock', 'asc')->get();

        // Statistik
        $totalMaterialsLow = $materials->count();
        $totalSparepartsLow = $spareparts->count();
        $totalLowStock = $totalMaterialsLow + $totalSparepartsLow;

        // Hitung nilai total stock rendah
        $totalValueMaterials = $materials->sum(function($item) {
            return $item->stock * $item->price;
        });

        $totalValueSpareparts = $spareparts->sum(function($item) {
            return $item->stock * $item->price;
        });

        $totalValueLowStock = $totalValueMaterials + $totalValueSpareparts;

        $pdf = Pdf::loadView('admin.reports.pdf.stock-limit', compact(
            'materials',
            'spareparts',
            'totalMaterialsLow',
            'totalSparepartsLow',
            'totalLowStock',
            'totalValueMaterials',
            'totalValueSpareparts',
            'totalValueLowStock'
        ));

        $filename = 'laporan_barang_limit_' . date('Y-m-d_H-i-s') . '.pdf';
        
        return $pdf->download($filename);
    }

    /**
     * Export Laporan Barang Rusak ke PDF
     */
    public function damagedItemsPdf(Request $request)
    {
        $materialsQuery = Material::hasDamage();
        $sparepartsQuery = Sparepart::hasDamage();
        $productsQuery = Product::where('condition', 'damaged');

        // Filter berdasarkan kategori (untuk materials)
        if ($request->filled('category')) {
            $materialsQuery->where('category', $request->category);
        }

        // Filter berdasarkan brand (untuk spareparts)
        if ($request->filled('brand')) {
            $sparepartsQuery->where('brand', $request->brand);
        }

        $materials = $materialsQuery->orderBy('updated_at', 'desc')->get();
        $spareparts = $sparepartsQuery->orderBy('updated_at', 'desc')->get();
        $products = $productsQuery->orderBy('updated_at', 'desc')->get();

        // Statistik
        $totalDamagedMaterials = $materials->count();
        $totalDamagedSpareparts = $spareparts->count();
        $totalDamagedProducts = $products->count();
        $totalDamagedItems = $totalDamagedMaterials + $totalDamagedSpareparts + $totalDamagedProducts;

        // Hitung nilai total barang rusak
        $totalValueMaterials = $materials->sum(function($item) {
            return $item->stock * $item->price;
        });

        $totalValueSpareparts = $spareparts->sum(function($item) {
            return $item->stock * $item->price;
        });

        $totalValueProducts = $products->sum(function($item) {
            return $item->product_prices;
        });

        $totalValueDamaged = $totalValueMaterials + $totalValueSpareparts + $totalValueProducts;

        $pdf = Pdf::loadView('admin.reports.pdf.damaged-items', compact(
            'materials',
            'spareparts',
            'products',
            'totalDamagedMaterials',
            'totalDamagedSpareparts',
            'totalDamagedProducts',
            'totalDamagedItems',
            'totalValueMaterials',
            'totalValueSpareparts',
            'totalValueProducts',
            'totalValueDamaged'
        ));

        $filename = 'laporan_barang_rusak_' . date('Y-m-d_H-i-s') . '.pdf';
        
        return $pdf->download($filename);
    }

    /**
     * Laporan Produksi
     */
    public function productions(Request $request)
    {
        $query = Production::with([
            'order', 
            'product', 
            'teknisi', 
            'productionMaterials.material',
            'productionSpareparts.sparepart',
            'supervisor',
            'completionApprover',
            'materialsReceiver'
        ])->orderBy('created_at', 'desc');

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->where('created_at', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('created_at', '<=', $request->end_date . ' 23:59:59');
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter berdasarkan teknisi
        if ($request->filled('teknisi_id')) {
            $query->where('teknisi_id', $request->teknisi_id);
        }

        // Filter berdasarkan product
        if ($request->filled('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        // Search filter
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $query->where(function ($q) use ($searchTerm) {
                $q->whereHas('order', function($q2) use ($searchTerm) {
                    $q2->where('customer_name', 'like', "%{$searchTerm}%")
                       ->orWhere('customer_email', 'like', "%{$searchTerm}%");
                    if (is_numeric($searchTerm)) {
                        $q2->orWhere('id', $searchTerm);
                    }
                })
                ->orWhereHas('product', function($q2) use ($searchTerm) {
                    $q2->where('product_title', 'like', "%{$searchTerm}%");
                })
                ->orWhereHas('teknisi', function($q2) use ($searchTerm) {
                    $q2->where('name', 'like', "%{$searchTerm}%");
                });
                if (is_numeric($searchTerm)) {
                    $q->orWhere('id', $searchTerm);
                }
            });
        }

        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $productions = $query->paginate($itemsPerPage)->withQueryString();

        // Statistik (hitung ulang setelah filter search)
        $statsQuery = Production::with([
            'order', 
            'product', 
            'teknisi', 
            'productionMaterials.material',
            'productionSpareparts.sparepart',
            'supervisor',
            'completionApprover',
            'materialsReceiver'
        ])->orderBy('created_at', 'desc');
        
        if ($request->filled('start_date')) {
            $statsQuery->where('created_at', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $statsQuery->where('created_at', '<=', $request->end_date . ' 23:59:59');
        }
        if ($request->filled('status')) {
            $statsQuery->where('status', $request->status);
        }
        if ($request->filled('teknisi_id')) {
            $statsQuery->where('teknisi_id', $request->teknisi_id);
        }
        if ($request->filled('product_id')) {
            $statsQuery->where('product_id', $request->product_id);
        }
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $statsQuery->where(function ($q) use ($searchTerm) {
                $q->whereHas('order', function($q2) use ($searchTerm) {
                    $q2->where('customer_name', 'like', "%{$searchTerm}%")
                       ->orWhere('customer_email', 'like', "%{$searchTerm}%");
                    if (is_numeric($searchTerm)) {
                        $q2->orWhere('id', $searchTerm);
                    }
                })
                ->orWhereHas('product', function($q2) use ($searchTerm) {
                    $q2->where('product_title', 'like', "%{$searchTerm}%");
                })
                ->orWhereHas('teknisi', function($q2) use ($searchTerm) {
                    $q2->where('name', 'like', "%{$searchTerm}%");
                });
                if (is_numeric($searchTerm)) {
                    $q->orWhere('id', $searchTerm);
                }
            });
        }
        
        $totalProductions = $statsQuery->count();
        $totalMaterialCost = $statsQuery->sum('total_material_cost') ?? 0;
        $totalSparepartCost = $statsQuery->sum('total_sparepart_cost') ?? 0;
        $totalLaborCost = $statsQuery->sum('labor_cost') ?? 0;
        $totalProductionCost = $totalMaterialCost + $totalSparepartCost + $totalLaborCost;

        // Teknisi list untuk filter
        $teknisiList = \App\Models\User::where('user_type', 'teknisi')->get();
        $productList = Product::where('is_active', true)->get();

        // Chart data untuk 12 bulan terakhir
        $chartData = Production::select(
                DB::raw('MONTH(created_at) as month'),
                DB::raw('YEAR(created_at) as year'),
                DB::raw('COUNT(*) as count'),
                DB::raw('SUM(total_production_cost) as total_cost'),
                DB::raw('SUM(labor_cost) as total_labor')
            )
            ->where('created_at', '>=', Carbon::now()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        return view('admin.reports.productions', compact(
            'productions',
            'totalProductions',
            'totalMaterialCost',
            'totalSparepartCost',
            'totalLaborCost',
            'totalProductionCost',
            'teknisiList',
            'productList',
            'chartData'
        ));
    }

    /**
     * Export Laporan Produksi
     */
    public function productionsExport(Request $request)
    {
        $query = Production::with([
            'order', 
            'product', 
            'teknisi', 
            'productionMaterials.material',
            'productionSpareparts.sparepart'
        ])->orderBy('created_at', 'desc');

        // Filter sama seperti index
        if ($request->filled('start_date')) {
            $query->where('created_at', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('created_at', '<=', $request->end_date . ' 23:59:59');
        }
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        if ($request->filled('teknisi_id')) {
            $query->where('teknisi_id', $request->teknisi_id);
        }
        if ($request->filled('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        $productions = $query->get();

        // Generate HTML table dengan styling untuk Excel
        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table {
            border-collapse: collapse;
            width: 100%;
            font-family: Arial, sans-serif;
            font-size: 11px;
        }
        th {
            background-color: #4472C4;
            color: #FFFFFF;
            font-weight: bold;
            text-align: center;
            padding: 10px 8px;
            border: 1px solid #000000;
            vertical-align: middle;
        }
        td {
            padding: 8px;
            border: 1px solid #CCCCCC;
            vertical-align: middle;
        }
        tr:nth-child(even) {
            background-color: #F2F2F2;
        }
        .text-right {
            text-align: right;
        }
        .text-center {
            text-align: center;
        }
    </style>
</head>
<body>
    <table>
        <thead>
            <tr>
                <th>No</th>
                <th>ID</th>
                <th>Order ID</th>
                <th>Customer</th>
                <th>Produk</th>
                <th class="text-center">Qty</th>
                <th>Teknisi</th>
                <th>Tanggal Mulai</th>
                <th>Durasi</th>
                <th class="text-center">Status</th>
                <th class="text-right">Biaya Bahan</th>
                <th class="text-right">Biaya Sparepart</th>
                <th class="text-right">Labor Cost</th>
                <th class="text-right">Total Biaya</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($productions as $index => $production) {
            $duration = '-';
            if ($production->start_date && $production->end_date) {
                $duration = $production->start_date->diffInDays($production->end_date) . ' hari';
            } elseif ($production->estimated_duration_days) {
                $duration = 'Estimasi: ' . $production->estimated_duration_days . ' hari';
            }

            $html .= '<tr>';
            $html .= '<td class="text-center">' . ($index + 1) . '</td>';
            $html .= '<td class="text-center">' . $production->id . '</td>';
            $html .= '<td class="text-center">' . ($production->order->id ?? '-') . '</td>';
            $html .= '<td>' . htmlspecialchars($production->order->customer_name ?? '-') . '</td>';
            $html .= '<td>' . htmlspecialchars($production->product->product_title ?? '-') . '</td>';
            $html .= '<td class="text-center">' . $production->quantity . '</td>';
            $html .= '<td>' . htmlspecialchars($production->teknisi->name ?? '-') . '</td>';
            $html .= '<td>' . ($production->actual_start_date ? $production->actual_start_date->format('d/m/Y') : ($production->start_date ? $production->start_date->format('d/m/Y') : '-')) . '</td>';
            $html .= '<td class="text-center">' . $duration . '</td>';
            $html .= '<td class="text-center">' . htmlspecialchars(ucfirst(str_replace('_', ' ', $production->status))) . '</td>';
            $html .= '<td class="text-right">' . number_format($production->total_material_cost ?? 0, 0, ',', '.') . '</td>';
            $html .= '<td class="text-right">' . number_format($production->total_sparepart_cost ?? 0, 0, ',', '.') . '</td>';
            $html .= '<td class="text-right">' . number_format($production->labor_cost ?? 0, 0, ',', '.') . '</td>';
            $html .= '<td class="text-right">' . number_format($production->total_actual_cost ?? 0, 0, ',', '.') . '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody>
    </table>
</body>
</html>';

        // Generate filename
        $filename = 'laporan_produksi_' . now()->format('Y-m-d_H-i-s') . '.xls';

        // Return response dengan header untuk Excel
        return response($html, 200)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->header('Cache-Control', 'max-age=0');
    }

    /**
     * Export Laporan Produksi ke PDF
     */
    public function productionsPdf(Request $request)
    {
        $query = Production::with([
            'order', 
            'product', 
            'teknisi', 
            'productionMaterials.material',
            'productionSpareparts.sparepart'
        ])->orderBy('created_at', 'desc');

        // Filter sama seperti index
        if ($request->filled('start_date')) {
            $query->where('created_at', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('created_at', '<=', $request->end_date . ' 23:59:59');
        }
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        if ($request->filled('teknisi_id')) {
            $query->where('teknisi_id', $request->teknisi_id);
        }
        if ($request->filled('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        $productions = $query->get();

        // Statistik
        $totalProductions = $productions->count();
        $totalMaterialCost = $productions->sum('total_material_cost') ?? 0;
        $totalSparepartCost = $productions->sum('total_sparepart_cost') ?? 0;
        $totalLaborCost = $productions->sum('labor_cost') ?? 0;
        $totalProductionCost = $totalMaterialCost + $totalSparepartCost + $totalLaborCost;

        $pdf = Pdf::loadView('admin.reports.pdf.productions', compact(
            'productions',
            'totalProductions',
            'totalMaterialCost',
            'totalSparepartCost',
            'totalLaborCost',
            'totalProductionCost'
        ));

        $filename = 'laporan_produksi_' . date('Y-m-d_H-i-s') . '.pdf';
        
        return $pdf->download($filename);
    }

    /**
     * Laporan Keuangan - Jurnal Umum
     */
    public function financial(Request $request)
    {
        // For AJAX requests to handle journal data
        // Check if it's an AJAX request or POST request with action parameter
        if ($request->ajax() || ($request->isMethod('post') && $request->has('action'))) {
            return $this->handleJournalAjax($request);
        }

        // Get master data for dropdowns
        $chartOfAccounts = \App\Models\ChartOfAccount::active()->orderBy('kode_akun')->get();
        $projects = \App\Models\Project::active()->orderBy('kode_bantu')->get();

        // For regular page load, show the journal interface
        return view('admin.reports.general-journal', compact('chartOfAccounts', 'projects'));
    }

    /**
     * Handle AJAX requests for journal operations
     */
    private function handleJournalAjax(Request $request)
    {
        $action = $request->get('action');

        switch ($action) {
            case 'save':
                return $this->saveJournalEntries($request);
            case 'load':
                return $this->loadJournalEntries($request);
            case 'delete':
                return $this->deleteJournalEntry($request);
            default:
                return response()->json(['error' => 'Invalid action'], 400);
        }
    }

    private function resolvePairAccountCode($kdPerk, $keterangan, $namaPerkiraan)
    {
        $kd = $kdPerk ?? '';
        $ket = trim((string)$keterangan);
        $nama = (string)$namaPerkiraan;

        if (preg_match('/^dp\s/i', $ket)) {
            if ($kd === '101-01') return '201-02';
            if ($kd === '201-02') return '101-01';
        }

        if (stripos($ket, 'upah pekerja') !== false || stripos($nama, 'Biaya Tenaga Kerja Langsung') !== false || $kd === '501-02') {
            if ($kd === '501-02') return '101-01';
            if ($kd === '101-01') return '501-02';
        }

        if (stripos($ket, 'biaya material') !== false || stripos($nama, 'Biaya Material Proyek') !== false || $kd === '501-01') {
            if ($kd === '501-01') return '101-01';
            if ($kd === '101-01') return '501-01';
        }

        if (stripos($ket, 'biaya pengangkutan') !== false || stripos($nama, 'Biaya Pengangkutan & Pengiriman') !== false || $kd === '501-04') {
            if ($kd === '501-04') return '101-01';
            if ($kd === '101-01') return '501-04';
        }

        return null;
    }

    private function resolvePrimaryAccountCode($side, $keterangan, $namaPerkiraan)
    {
        $ket = trim((string)$keterangan);
        $nama = (string)$namaPerkiraan;

        if (preg_match('/^dp\s/i', $ket)) {
            return $side === 'debit' ? '101-01' : '201-02';
        }

        if (stripos($ket, 'upah pekerja') !== false || stripos($nama, 'Biaya Tenaga Kerja Langsung') !== false) {
            return '501-02';
        }

        if (stripos($ket, 'biaya material') !== false || stripos($nama, 'Biaya Material Proyek') !== false) {
            return '501-01';
        }

        return null;
    }

    /**
     * Save journal entries
     */
    private function saveJournalEntries(Request $request)
    {
        $entries = $request->get('entries', []);

        // Get all chart of accounts and projects for validation and name lookup
        $chartAccounts = ChartOfAccount::active()->get()->keyBy('kode_akun');
        $projects = Project::active()->get()->keyBy('kode_bantu');

        // Validate entries
        foreach ($entries as $entry) {
            if (empty($entry['tanggal_transaksi']) || empty($entry['keterangan'])) {
                continue; // Skip empty entries
            }

            // Validate kode akun exists in Chart of Accounts
            if (!empty($entry['kd_perk'])) {
                if (!$chartAccounts->has($entry['kd_perk'])) {
                    return response()->json([
                        'error' => 'Kode akun "' . $entry['kd_perk'] . '" tidak ditemukan di Chart of Accounts. Silakan pilih dari dropdown.'
                    ], 400);
                }
            }

            // Validate kode proyek exists in Projects
            if (!empty($entry['kd_project'])) {
                if (!$projects->has($entry['kd_project'])) {
                    return response()->json([
                        'error' => 'Kode proyek "' . $entry['kd_project'] . '" tidak ditemukan di Master Data Projects. Silakan pilih dari dropdown.'
                    ], 400);
                }
            }

            // Convert to float and treat 0 as empty
            $debit = isset($entry['debit']) ? (float) $entry['debit'] : 0;
            $kredit = isset($entry['kredit']) ? (float) $entry['kredit'] : 0;

            // Validate that debit and kredit are not both filled (> 0)
            if ($debit > 0 && $kredit > 0) {
                return response()->json(['error' => 'Debit dan Kredit tidak boleh diisi bersamaan'], 400);
            }

            // Ensure at least one of debit or kredit is filled (> 0)
            if ($debit <= 0 && $kredit <= 0) {
                return response()->json(['error' => 'Minimal salah satu Debit atau Kredit harus diisi'], 400);
            }
        }

        try {
            $savedCount = 0;
            $updatedCount = 0;
            $createdCount = 0;

            foreach ($entries as $entry) {
                if (empty($entry['tanggal_transaksi']) || empty($entry['keterangan'])) {
                    continue; // Skip empty entries
                }

                $debitVal = isset($entry['debit']) && $entry['debit'] > 0 ? $entry['debit'] : null;
                $kreditVal = isset($entry['kredit']) && $entry['kredit'] > 0 ? $entry['kredit'] : null;
                $side = $debitVal ? 'debit' : 'kredit';

                if (empty($entry['kd_perk'])) {
                    $autoKd = $this->resolvePrimaryAccountCode($side, $entry['keterangan'] ?? '', $entry['nama_perkiraan'] ?? '');
                    if ($autoKd) {
                        $entry['kd_perk'] = $autoKd;
                    }
                }

                $namaPerkiraan = null;
                if (!empty($entry['kd_perk']) && $chartAccounts->has($entry['kd_perk'])) {
                    $namaPerkiraan = $chartAccounts->get($entry['kd_perk'])->nama_akun;
                }

                $namaProyek = null;
                if (!empty($entry['kd_project']) && $projects->has($entry['kd_project'])) {
                    $namaProyek = $projects->get($entry['kd_project'])->nama_proyek;
                }

                // Check if entry has ID (update existing) or create new
                if (!empty($entry['id'])) {
                    // Update existing entry
                    $journalEntry = JournalEntry::find($entry['id']);
                    if ($journalEntry) {
                        $debit = $debitVal;
                        $kredit = $kreditVal;

                        $journalEntry->update([
                            'tanggal_transaksi' => $entry['tanggal_transaksi'] ?? null,
                            'bukti_transaksi' => $entry['bukti_transaksi'] ?? null,
                            'keterangan' => $entry['keterangan'] ?? null,
                            'nama_perkiraan' => $namaPerkiraan, // Use from master data
                            'kd_perk' => $entry['kd_perk'] ?? null,
                            'customer' => $entry['customer'] ?? null,
                            'kode' => $entry['kode'] ?? null,
                            'nama_proyek' => $namaProyek, // Use from master data
                            'kd_project' => $entry['kd_project'] ?? null,
                            'debit' => $debit,
                            'kredit' => $kredit,
                        ]);
                        $updatedCount++;
                        $savedCount++;

                        $pairKd = $this->resolvePairAccountCode($entry['kd_perk'] ?? null, $entry['keterangan'] ?? '', $namaPerkiraan ?? '');
                        if ($pairKd && $chartAccounts->has($pairKd)) {
                            $pairNama = $chartAccounts->get($pairKd)->nama_akun;
                            $pairExists = JournalEntry::whereDate('tanggal_transaksi', $entry['tanggal_transaksi'] ?? null)
                                ->where('keterangan', $entry['keterangan'] ?? null)
                                ->where('kd_perk', $pairKd)
                                ->where('kd_project', $entry['kd_project'] ?? null)
                                ->where(function($q) use ($debit, $kredit) {
                                    if ($debit) { $q->where('debit', null)->where('kredit', $debit); }
                                    if ($kredit) { $q->where('debit', $kredit)->where('kredit', null); }
                                })->first();
                            if (!$pairExists) {
                                JournalEntry::create([
                                    'tanggal_transaksi' => $entry['tanggal_transaksi'] ?? null,
                                    'bukti_transaksi' => $entry['bukti_transaksi'] ?? null,
                                    'keterangan' => $entry['keterangan'] ?? null,
                                    'nama_perkiraan' => $pairNama,
                                    'kd_perk' => $pairKd,
                                    'customer' => $entry['customer'] ?? null,
                                    'kode' => $entry['kode'] ?? null,
                                    'nama_proyek' => $namaProyek,
                                    'kd_project' => $entry['kd_project'] ?? null,
                                    'debit' => $kredit ? $kredit : null,
                                    'kredit' => $debit ? $debit : null,
                                    'created_by' => auth()->id(),
                                ]);
                                $createdCount++;
                                $savedCount++;
                            }
                        }
                    }
                } else {
                    // Create new entry
                    $debit = $debitVal;
                    $kredit = $kreditVal;

                JournalEntry::create([
                    'tanggal_transaksi' => $entry['tanggal_transaksi'] ?? null,
                    'bukti_transaksi' => $entry['bukti_transaksi'] ?? null,
                    'keterangan' => $entry['keterangan'] ?? null,
                    'nama_perkiraan' => $namaPerkiraan, // Use from master data
                    'kd_perk' => $entry['kd_perk'] ?? null,
                    'customer' => $entry['customer'] ?? null,
                    'kode' => $entry['kode'] ?? null,
                    'nama_proyek' => $namaProyek, // Use from master data
                    'kd_project' => $entry['kd_project'] ?? null,
                        'debit' => $debit,
                        'kredit' => $kredit,
                    'created_by' => auth()->id(),
                ]);
                    $createdCount++;
                $savedCount++;

                $pairKd = $this->resolvePairAccountCode($entry['kd_perk'] ?? null, $entry['keterangan'] ?? '', $namaPerkiraan ?? '');
                if ($pairKd && $chartAccounts->has($pairKd)) {
                    $pairNama = $chartAccounts->get($pairKd)->nama_akun;
                    $pairExists = JournalEntry::whereDate('tanggal_transaksi', $entry['tanggal_transaksi'] ?? null)
                        ->where('keterangan', $entry['keterangan'] ?? null)
                        ->where('kd_perk', $pairKd)
                        ->where('kd_project', $entry['kd_project'] ?? null)
                        ->where(function($q) use ($debit, $kredit) {
                            if ($debit) { $q->where('debit', null)->where('kredit', $debit); }
                            if ($kredit) { $q->where('debit', $kredit)->where('kredit', null); }
                        })->first();
                    if (!$pairExists) {
                        JournalEntry::create([
                            'tanggal_transaksi' => $entry['tanggal_transaksi'] ?? null,
                            'bukti_transaksi' => $entry['bukti_transaksi'] ?? null,
                            'keterangan' => $entry['keterangan'] ?? null,
                            'nama_perkiraan' => $pairNama,
                            'kd_perk' => $pairKd,
                            'customer' => $entry['customer'] ?? null,
                            'kode' => $entry['kode'] ?? null,
                            'nama_proyek' => $namaProyek,
                            'kd_project' => $entry['kd_project'] ?? null,
                            'debit' => $kredit ? $kredit : null,
                            'kredit' => $debit ? $debit : null,
                            'created_by' => auth()->id(),
                        ]);
                        $createdCount++;
                        $savedCount++;
                    }
                }
                }
            }

            $message = 'Data jurnal berhasil disimpan';
            if ($createdCount > 0 && $updatedCount > 0) {
                $message .= ' (' . $createdCount . ' baru, ' . $updatedCount . ' diperbarui)';
            } elseif ($createdCount > 0) {
                $message .= ' (' . $createdCount . ' entries baru)';
            } elseif ($updatedCount > 0) {
                $message .= ' (' . $updatedCount . ' entries diperbarui)';
            }

            return response()->json([
                'success' => true,
                'message' => $message,
                'count' => $savedCount,
                'created' => $createdCount,
                'updated' => $updatedCount
            ]);

        } catch (\Exception $e) {
            \Log::error('Journal save error: ' . $e->getMessage());
            return response()->json([
                'error' => 'Gagal menyimpan data jurnal: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Load journal entries
     */
    private function loadJournalEntries(Request $request)
    {
        try {
            // Get all chart of accounts and projects for name lookup
            $chartAccounts = ChartOfAccount::active()->get()->keyBy('kode_akun');
            $projects = Project::active()->get()->keyBy('kode_bantu');

            // Load all entries for report
            $query = JournalEntry::query();
            
            // Show all entries (no filter by user)
            $entries = $query
                ->orderBy('tanggal_transaksi', 'asc')
                ->orderBy('created_at', 'asc')
                ->get()
                ->map(function ($entry) use ($chartAccounts, $projects) {
                    // Get nama from master data (prioritize master data over stored value)
                    $namaPerkiraan = '';
                    if (!empty($entry->kd_perk) && $chartAccounts->has($entry->kd_perk)) {
                        $namaPerkiraan = $chartAccounts->get($entry->kd_perk)->nama_akun;
                    } elseif (!empty($entry->nama_perkiraan)) {
                        // Fallback to stored value if not found in master data
                        $namaPerkiraan = $entry->nama_perkiraan;
                    }

                    $namaProyek = '';
                    if (!empty($entry->kd_project) && $projects->has($entry->kd_project)) {
                        $namaProyek = $projects->get($entry->kd_project)->nama_proyek;
                    } elseif (!empty($entry->nama_proyek)) {
                        // Fallback to stored value if not found in master data
                        $namaProyek = $entry->nama_proyek;
                    }

                    return [
                        'id' => $entry->id,
                        'tanggal_transaksi' => $entry->tanggal_transaksi ? $entry->tanggal_transaksi->format('Y-m-d') : null,
                        'bukti_transaksi' => $entry->bukti_transaksi ?? '',
                        'keterangan' => $entry->keterangan ?? '',
                        'nama_perkiraan' => $namaPerkiraan, // From master data
                        'kd_perk' => $entry->kd_perk ?? '',
                        'customer' => $entry->customer ?? '',
                        'kode' => $entry->kode ?? '',
                        'nama_proyek' => $namaProyek, // From master data
                        'kd_project' => $entry->kd_project ?? '',
                        'debit' => $entry->debit ? (float) $entry->debit : 0,
                        'kredit' => $entry->kredit ? (float) $entry->kredit : 0,
                    ];
                });

            \Log::info('Journal entries loaded: ' . $entries->count() . ' entries');

            return response()->json([
                'success' => true,
                'entries' => $entries,
                'count' => $entries->count()
            ]);

        } catch (\Exception $e) {
            \Log::error('Journal load error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'error' => 'Gagal memuat data jurnal: ' . $e->getMessage(),
                'entries' => []
            ], 500);
        }
    }

    /**
     * Delete journal entry
     */
    private function deleteJournalEntry(Request $request)
    {
        $id = $request->get('id');

        try {
            // Validate ID
            if (!$id) {
                return response()->json([
                    'success' => false,
                    'error' => 'ID tidak valid'
                ], 400);
            }

            // Find entry - remove created_by filter to allow admin to delete any entry
            $entry = JournalEntry::find($id);

            if ($entry) {
                $entry->delete(); // Soft delete
                return response()->json([
                    'success' => true,
                    'message' => 'Entry berhasil dihapus'
                ], 200);
            }

            return response()->json([
                'success' => false,
                'error' => 'Entry tidak ditemukan'
            ], 404);

        } catch (\Exception $e) {
            \Log::error('Journal delete error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Gagal menghapus entry: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export Laporan Keuangan - Jurnal Umum (Excel)
     */
    public function financialExport(Request $request)
    {
        $query = JournalEntry::with(['creator'])
            ->orderBy('tanggal_transaksi', 'asc')
            ->orderBy('created_at', 'asc');

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->whereDate('tanggal_transaksi', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('tanggal_transaksi', '<=', $request->end_date);
        }

        // Search filter
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $query->where(function ($q) use ($searchTerm) {
                $q->where('keterangan', 'like', "%{$searchTerm}%")
                  ->orWhere('bukti_transaksi', 'like', "%{$searchTerm}%")
                  ->orWhere('nama_perkiraan', 'like', "%{$searchTerm}%")
                  ->orWhere('kd_perk', 'like', "%{$searchTerm}%");
            });
        }

        $entries = $query->get();

        // Calculate totals and running balance
        $totalDebit = $entries->sum('debit') ?? 0;
        $totalKredit = $entries->sum('kredit') ?? 0;
        $totalEntries = $entries->count();
        $runningBalance = 0;

        // Determine journal type based on account code
        $getJournalType = function($kdPerk) {
            if (preg_match('/^4/', $kdPerk ?? '')) return 'SAJ'; // Sales Journal
            if (preg_match('/^5/', $kdPerk ?? '')) return 'PCH'; // Purchase Journal
            if (preg_match('/^1[0-2]/', $kdPerk ?? '')) return 'BNK'; // Bank
            return 'GL'; // General Ledger
        };

        // Generate HTML table dengan styling untuk Excel sesuai format contoh
        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table {
            border-collapse: collapse;
            width: 100%;
            font-family: Arial, sans-serif;
            font-size: 11px;
        }
        th {
            background-color: #90EE90;
            color: #000000;
            font-weight: bold;
            text-align: center;
            padding: 8px 6px;
            border: 1px solid #000000;
            vertical-align: middle;
        }
        td {
            padding: 6px;
            border: 1px solid #CCCCCC;
            vertical-align: middle;
        }
        .text-right {
            text-align: right;
        }
        .text-center {
            text-align: center;
        }
        .summary {
            background-color: #4472C4;
            color: #FFFFFF;
            font-weight: bold;
        }
        .header-info {
            text-align: center;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <div class="header-info">
        <h2>JURNAL UMUM</h2>
        <p>' . ($request->start_date ? date('d/m/Y', strtotime($request->start_date)) : '01/01/2020') . ' - ' . ($request->end_date ? date('d/m/Y', strtotime($request->end_date)) : '31/12/2020') . '</p>
        <p>Page 1 of 1 | Print by: ' . (auth()->user()->name ?? 'Admin') . '</p>
    </div>
    <table>
        <thead>
            <tr>
                <th>Date</th>
                <th>Post</th>
                <th>Journal</th>
                <th>Account</th>
                <th>Description</th>
                <th>Name</th>
                <th>P. ID</th>
                <th>Source</th>
                <th>Ref.</th>
                <th>Debit</th>
                <th>Credit</th>
                <th>Balance</th>
            </tr>
        </thead>
        <tbody>';

        $no = 1;
        foreach ($entries as $entry) {
            $debit = $entry->debit ?? 0;
            $kredit = $entry->kredit ?? 0;
            $runningBalance = $runningBalance + $debit - $kredit;
            
            $accountDisplay = ($entry->kd_perk ?? '') . ' ' . ($entry->nama_perkiraan ?? '');
            $journalType = $getJournalType($entry->kd_perk);
            
            $html .= '<tr>
                <td class="text-center">' . ($entry->tanggal_transaksi ? $entry->tanggal_transaksi->format('d M') : '-') . '</td>
                <td class="text-center">' . $no++ . '</td>
                <td class="text-center">' . $journalType . '</td>
                <td>' . htmlspecialchars($accountDisplay) . '</td>
                <td>' . htmlspecialchars($entry->keterangan ?? '-') . '</td>
                <td>' . htmlspecialchars($entry->customer ?? '-') . '</td>
                <td class="text-center">' . ($entry->kode ?? '-') . '</td>
                <td class="text-center">' . htmlspecialchars($entry->bukti_transaksi ?? '-') . '</td>
                <td class="text-center">-</td>
                <td class="text-right">' . ($debit > 0 ? number_format($debit, 2, '.', ',') : '') . '</td>
                <td class="text-right">' . ($kredit > 0 ? number_format($kredit, 2, '.', ',') : '') . '</td>
                <td class="text-right">' . ($runningBalance < 0 ? '(' . number_format(abs($runningBalance), 2, '.', ',') . ')' : number_format($runningBalance, 2, '.', ',')) . '</td>
            </tr>';
        }

        $html .= '</tbody>
        <tfoot>
            <tr class="summary">
                <td colspan="9" class="text-right"><strong>Total</strong></td>
                <td class="text-right"><strong>' . number_format($totalDebit, 2, '.', ',') . '</strong></td>
                <td class="text-right"><strong>' . number_format($totalKredit, 2, '.', ',') . '</strong></td>
                <td class="text-right"><strong>' . ($runningBalance < 0 ? '(' . number_format(abs($runningBalance), 2, '.', ',') . ')' : number_format($runningBalance, 2, '.', ',')) . '</strong></td>
            </tr>
        </tfoot>
    </table>
</body>
</html>';

        return response($html)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="Jurnal_Umum_' . date('Y-m-d_H-i-s') . '.xls"');
    }

    /**
     * Export Laporan Keuangan - Jurnal Umum (PDF)
     */
    public function financialPdf(Request $request)
    {
        $query = JournalEntry::with(['creator'])
            ->orderBy('tanggal_transaksi', 'asc')
            ->orderBy('created_at', 'asc');

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->whereDate('tanggal_transaksi', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('tanggal_transaksi', '<=', $request->end_date);
        }

        // Search filter
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $query->where(function ($q) use ($searchTerm) {
                $q->where('keterangan', 'like', "%{$searchTerm}%")
                  ->orWhere('bukti_transaksi', 'like', "%{$searchTerm}%")
                  ->orWhere('nama_perkiraan', 'like', "%{$searchTerm}%")
                  ->orWhere('kd_perk', 'like', "%{$searchTerm}%");
            });
        }

        $entries = $query->get();

        // Calculate totals
        $totalDebit = $entries->sum('debit') ?? 0;
        $totalKredit = $entries->sum('kredit') ?? 0;
        $totalEntries = $entries->count();

        $pdf = Pdf::loadView('admin.reports.pdf.financial', compact(
            'entries',
            'totalDebit',
            'totalKredit',
            'totalEntries'
        ));

        $filename = 'laporan_jurnal_umum_' . date('Y-m-d_H-i-s') . '.pdf';
        
        return $pdf->download($filename);
    }

    /**
     * Buku Besar (General Ledger) - Per Akun
     */
    public function generalLedger(Request $request)
    {
        $accountCode = $request->get('kd_perk');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');

        // Get all accounts from Chart of Accounts
        $accounts = ChartOfAccount::active()
            ->orderBy('kode_akun')
            ->get()
            ->map(function($account) {
                return (object)[
                    'kd_perk' => $account->kode_akun,
                    'nama_perkiraan' => $account->nama_akun
                ];
            });

        // If account code is selected, show ledger for that account
        if ($accountCode) {
            // Get account info from Chart of Accounts
            $chartAccount = ChartOfAccount::where('kode_akun', $accountCode)->first();
            
            if (!$chartAccount) {
                return redirect()->back()->with('error', 'Kode akun tidak ditemukan di Chart of Accounts');
            }

            $entries = JournalEntry::where('kd_perk', $accountCode)
                ->when($startDate, function($q) use ($startDate) {
                    return $q->whereDate('tanggal_transaksi', '>=', $startDate);
                })
                ->when($endDate, function($q) use ($endDate) {
                    return $q->whereDate('tanggal_transaksi', '<=', $endDate);
                })
                ->orderBy('tanggal_transaksi', 'asc')
                ->orderBy('created_at', 'asc')
                ->get();

            // Get balance position from Chart of Accounts
            $balancePosition = $chartAccount->pos_saldo ?? 'DEBIT';

            // Calculate opening balance (saldo awal) - include saldo awal from Chart of Accounts
            $openingBalance = 0;
            
            // Add initial balance from Chart of Accounts
            if ($balancePosition === 'KREDIT') {
                $openingBalance = ($chartAccount->saldo_awal_kredit ?? 0) - ($chartAccount->saldo_awal_debet ?? 0);
            } else {
                $openingBalance = ($chartAccount->saldo_awal_debet ?? 0) - ($chartAccount->saldo_awal_kredit ?? 0);
            }
            
            // Add transactions before start date
            if ($startDate) {
                $openingEntries = JournalEntry::where('kd_perk', $accountCode)
                    ->whereDate('tanggal_transaksi', '<', $startDate)
                    ->get();
                
                foreach ($openingEntries as $entry) {
                    $debit = $entry->debit ?? 0;
                    $kredit = $entry->kredit ?? 0;
                    
                    if ($balancePosition === 'KREDIT') {
                        $openingBalance += ($kredit - $debit);
                    } else {
                        $openingBalance += ($debit - $kredit);
                    }
                }
            } else {
                // If no start date, calculate from all previous entries
                $openingEntries = JournalEntry::where('kd_perk', $accountCode)
                    ->where('id', '<', $entries->first()->id ?? 0)
                    ->get();
                
                foreach ($openingEntries as $entry) {
                    $debit = $entry->debit ?? 0;
                    $kredit = $entry->kredit ?? 0;
                    
                    if ($balancePosition === 'KREDIT') {
                        $openingBalance += ($kredit - $debit);
                    } else {
                        $openingBalance += ($debit - $kredit);
                    }
                }
            }

            // Calculate running balance
            $runningBalance = $openingBalance;
            $ledgerData = [];
            
            foreach ($entries as $entry) {
                $debit = $entry->debit ?? 0;
                $kredit = $entry->kredit ?? 0;
                
                if ($balancePosition === 'KREDIT') {
                    $runningBalance += ($kredit - $debit);
                } else {
                    $runningBalance += ($debit - $kredit);
                }
                
                $ledgerData[] = [
                    'id' => $entry->id,
                    'tanggal' => $entry->tanggal_transaksi,
                    'bukti_transaksi' => $entry->bukti_transaksi,
                    'keterangan' => $entry->keterangan,
                    'debit' => $debit,
                    'kredit' => $kredit,
                    'saldo' => $runningBalance,
                ];
            }

            $accountName = $chartAccount->nama_akun ?? '';

            return view('admin.reports.general-ledger', compact(
                'accounts',
                'accountCode',
                'accountName',
                'balancePosition',
                'openingBalance',
                'ledgerData',
                'startDate',
                'endDate'
            ));
        }

        return view('admin.reports.general-ledger', compact('accounts', 'accountCode', 'startDate', 'endDate'));
    }

    /**
     * Laporan Laba Rugi Per Proyek
     */
    public function profitLossByProject(Request $request)
    {
        $projectCode = $request->get('kd_project');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');

        // Get all projects from Projects table
        $projects = Project::active()
            ->orderBy('kode_bantu')
            ->get()
            ->map(function($project) {
                return (object)[
                    'kd_project' => $project->kode_bantu,
                    'nama_proyek' => $project->nama_proyek
                ];
            });

        if ($projectCode) {
            $entries = JournalEntry::where('kd_project', $projectCode)
                ->when($startDate, function($q) use ($startDate) {
                    return $q->whereDate('tanggal_transaksi', '>=', $startDate);
                })
                ->when($endDate, function($q) use ($endDate) {
                    return $q->whereDate('tanggal_transaksi', '<=', $endDate);
                })
                ->get();

            // Calculate revenue (Pendapatan) - accounts starting with 4
            $revenue = $entries->filter(function($entry) {
                return preg_match('/^4/', $entry->kd_perk ?? '');
            })->sum('kredit');

            // Calculate COGS (Harga Pokok Penjualan) - accounts starting with 5
            $cogs = $entries->filter(function($entry) {
                return preg_match('/^5/', $entry->kd_perk ?? '');
            })->sum('debit');

            // Calculate operating expenses (Beban Administrasi & Umum) - accounts starting with 6
            $operatingExpenses = $entries->filter(function($entry) {
                return preg_match('/^6/', $entry->kd_perk ?? '');
            })->sum('debit');

            // Calculate other income/expenses - accounts starting with 7-8
            $otherIncome = $entries->filter(function($entry) {
                return preg_match('/^[78]/', $entry->kd_perk ?? '');
            })->sum(function($entry) {
                return ($entry->kredit ?? 0) - ($entry->debit ?? 0);
            });

            // Calculate tax - accounts starting with 9
            $tax = $entries->filter(function($entry) {
                return preg_match('/^9/', $entry->kd_perk ?? '');
            })->sum('debit');

            $grossProfit = $revenue - $cogs;
            $operatingProfit = $grossProfit - $operatingExpenses;
            $profitBeforeTax = $operatingProfit + $otherIncome;
            $netProfit = $profitBeforeTax - $tax;

            // Get project name from Projects table
            $project = Project::where('kode_bantu', $projectCode)->first();
            $projectName = $project->nama_proyek ?? '';

            // Group by account for detailed view
            $accountGroups = $entries->groupBy('kd_perk')->map(function($group) {
                return [
                    'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                    'kd_perk' => $group->first()->kd_perk ?? '',
                    'debit' => $group->sum('debit'),
                    'kredit' => $group->sum('kredit'),
                ];
            });

            // Group by category for report display
            $revenueAccounts = $entries->filter(function($entry) {
                return preg_match('/^4/', $entry->kd_perk ?? '');
            })->groupBy('kd_perk')->map(function($group) {
                return [
                    'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                    'amount' => $group->sum('kredit'),
                ];
            });

            $cogsAccounts = $entries->filter(function($entry) {
                return preg_match('/^5/', $entry->kd_perk ?? '');
            })->groupBy('kd_perk')->map(function($group) {
                return [
                    'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                    'amount' => $group->sum('debit'),
                ];
            });

            $expenseAccounts = $entries->filter(function($entry) {
                return preg_match('/^6/', $entry->kd_perk ?? '');
            })->groupBy('kd_perk')->map(function($group) {
                return [
                    'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                    'amount' => $group->sum('debit'),
                ];
            });

            $otherIncomeAccounts = $entries->filter(function($entry) {
                return preg_match('/^[78]/', $entry->kd_perk ?? '');
            })->groupBy('kd_perk')->map(function($group) {
                $debit = $group->sum('debit');
                $kredit = $group->sum('kredit');
                return [
                    'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                    'amount' => $kredit - $debit,
                    'debit' => $debit,
                    'kredit' => $kredit,
                ];
            });

            $taxAccounts = $entries->filter(function($entry) {
                return preg_match('/^9/', $entry->kd_perk ?? '');
            })->groupBy('kd_perk')->map(function($group) {
                return [
                    'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                    'amount' => $group->sum('debit'),
                ];
            });

            return view('admin.reports.profit-loss-project', compact(
                'projects',
                'projectCode',
                'projectName',
                'revenue',
                'cogs',
                'operatingExpenses',
                'otherIncome',
                'tax',
                'grossProfit',
                'operatingProfit',
                'profitBeforeTax',
                'netProfit',
                'accountGroups',
                'revenueAccounts',
                'cogsAccounts',
                'expenseAccounts',
                'otherIncomeAccounts',
                'taxAccounts',
                'startDate',
                'endDate'
            ));
        }

        return view('admin.reports.profit-loss-project', compact('projects', 'projectCode', 'startDate', 'endDate'));
    }

    /**
     * Laporan Laba Rugi Per Bulan
     */
    public function profitLossByMonth(Request $request)
    {
        $month = $request->get('month', date('Y-m'));
        $startDate = $month . '-01';
        $endDate = date('Y-m-t', strtotime($startDate));

        // Get all chart of accounts
        $chartAccounts = ChartOfAccount::active()
            ->orderBy('kode_akun')
            ->get()
            ->keyBy('kode_akun');

        $entries = JournalEntry::whereBetween('tanggal_transaksi', [$startDate, $endDate])->get();

        // Calculate revenue - accounts with pos_laporan = 'LABA RUGI' and kode starting with 4
        $revenue = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^4/', $entry->kd_perk ?? '');
        })->sum('kredit');

        // Calculate COGS - accounts with pos_laporan = 'LABA RUGI' and kode starting with 5
        $cogs = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^5/', $entry->kd_perk ?? '');
        })->sum('debit');

        // Calculate operating expenses - accounts with pos_laporan = 'LABA RUGI' and kode starting with 6
        $operatingExpenses = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^6/', $entry->kd_perk ?? '');
        })->sum('debit');

        // Calculate other income/expenses - accounts with pos_laporan = 'LABA RUGI' and kode starting with 7-8
        $otherIncome = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^[78]/', $entry->kd_perk ?? '');
        })->sum(function($entry) {
            return ($entry->kredit ?? 0) - ($entry->debit ?? 0);
        });

        // Calculate tax - accounts with pos_laporan = 'LABA RUGI' and kode starting with 9
        $tax = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^9/', $entry->kd_perk ?? '');
        })->sum('debit');

        $grossProfit = $revenue - $cogs;
        $operatingProfit = $grossProfit - $operatingExpenses;
        $profitBeforeTax = $operatingProfit + $otherIncome;
        $netProfit = $profitBeforeTax - $tax;

        // Group revenue by account - use Chart of Accounts for nama_akun
        $revenueAccounts = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^4/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $kdPerk = $group->first()->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            return [
                'nama_perkiraan' => $chartAccount ? $chartAccount->nama_akun : ($group->first()->nama_perkiraan ?? ''),
                'amount' => $group->sum('kredit'),
            ];
        });

        // Group COGS by account
        $cogsAccounts = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^5/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $kdPerk = $group->first()->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            return [
                'nama_perkiraan' => $chartAccount ? $chartAccount->nama_akun : ($group->first()->nama_perkiraan ?? ''),
                'amount' => $group->sum('debit'),
            ];
        });

        // Group operating expenses by account
        $expenseAccounts = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^6/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $kdPerk = $group->first()->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            return [
                'nama_perkiraan' => $chartAccount ? $chartAccount->nama_akun : ($group->first()->nama_perkiraan ?? ''),
                'amount' => $group->sum('debit'),
            ];
        });

        // Group other income/expenses by account
        $otherIncomeAccounts = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^[78]/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $kdPerk = $group->first()->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            $debit = $group->sum('debit');
            $kredit = $group->sum('kredit');
            return [
                'nama_perkiraan' => $chartAccount ? $chartAccount->nama_akun : ($group->first()->nama_perkiraan ?? ''),
                'amount' => $kredit - $debit, // Positive = income, Negative = expense
                'debit' => $debit,
                'kredit' => $kredit,
            ];
        });

        // Group tax by account
        $taxAccounts = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^9/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $kdPerk = $group->first()->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            return [
                'nama_perkiraan' => $chartAccount ? $chartAccount->nama_akun : ($group->first()->nama_perkiraan ?? ''),
                'amount' => $group->sum('debit'),
            ];
        });

        $monthName = \Carbon\Carbon::parse($startDate)->locale('id')->isoFormat('MMMM YYYY');

        return view('admin.reports.profit-loss-month', compact(
            'month',
            'monthName',
            'revenue',
            'cogs',
            'operatingExpenses',
            'otherIncome',
            'tax',
            'grossProfit',
            'operatingProfit',
            'profitBeforeTax',
            'netProfit',
            'revenueAccounts',
            'cogsAccounts',
            'expenseAccounts',
            'otherIncomeAccounts',
            'taxAccounts',
            'startDate',
            'endDate'
        ));
    }

    /**
     * Laporan Laba Rugi Per Tahun
     */
    public function profitLossByYear(Request $request)
    {
        $year = $request->get('year', date('Y'));
        $startDate = $year . '-01-01';
        $endDate = $year . '-12-31';

        // Get all chart of accounts
        $chartAccounts = ChartOfAccount::active()
            ->orderBy('kode_akun')
            ->get()
            ->keyBy('kode_akun');

        $entries = JournalEntry::whereBetween('tanggal_transaksi', [$startDate, $endDate])->get();

        // Calculate revenue - accounts with pos_laporan = 'LABA RUGI' and kode starting with 4
        $revenue = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^4/', $entry->kd_perk ?? '');
        })->sum('kredit');

        // Calculate COGS - accounts with pos_laporan = 'LABA RUGI' and kode starting with 5
        $cogs = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^5/', $entry->kd_perk ?? '');
        })->sum('debit');

        // Calculate operating expenses - accounts with pos_laporan = 'LABA RUGI' and kode starting with 6
        $operatingExpenses = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^6/', $entry->kd_perk ?? '');
        })->sum('debit');

        // Calculate other income/expenses - accounts with pos_laporan = 'LABA RUGI' and kode starting with 7-8
        $otherIncome = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^[78]/', $entry->kd_perk ?? '');
        })->sum(function($entry) {
            return ($entry->kredit ?? 0) - ($entry->debit ?? 0);
        });

        // Calculate tax - accounts with pos_laporan = 'LABA RUGI' and kode starting with 9
        $tax = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^9/', $entry->kd_perk ?? '');
        })->sum('debit');

        $grossProfit = $revenue - $cogs;
        $operatingProfit = $grossProfit - $operatingExpenses;
        $profitBeforeTax = $operatingProfit + $otherIncome;
        $netProfit = $profitBeforeTax - $tax;

        // Group revenue by account - use Chart of Accounts for nama_akun
        $revenueAccounts = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^4/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $kdPerk = $group->first()->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            return [
                'nama_perkiraan' => $chartAccount ? $chartAccount->nama_akun : ($group->first()->nama_perkiraan ?? ''),
                'amount' => $group->sum('kredit'),
            ];
        });

        // Group COGS by account
        $cogsAccounts = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^5/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $kdPerk = $group->first()->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            return [
                'nama_perkiraan' => $chartAccount ? $chartAccount->nama_akun : ($group->first()->nama_perkiraan ?? ''),
                'amount' => $group->sum('debit'),
            ];
        });

        // Group operating expenses by account
        $expenseAccounts = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^6/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $kdPerk = $group->first()->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            return [
                'nama_perkiraan' => $chartAccount ? $chartAccount->nama_akun : ($group->first()->nama_perkiraan ?? ''),
                'amount' => $group->sum('debit'),
            ];
        });

        // Group other income/expenses by account
        $otherIncomeAccounts = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^[78]/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $kdPerk = $group->first()->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            $debit = $group->sum('debit');
            $kredit = $group->sum('kredit');
            return [
                'nama_perkiraan' => $chartAccount ? $chartAccount->nama_akun : ($group->first()->nama_perkiraan ?? ''),
                'amount' => $kredit - $debit, // Positive = income, Negative = expense
                'debit' => $debit,
                'kredit' => $kredit,
            ];
        });

        // Group tax by account
        $taxAccounts = $entries->filter(function($entry) use ($chartAccounts) {
            $chartAccount = $chartAccounts->get($entry->kd_perk ?? '');
            return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^9/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $kdPerk = $group->first()->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            return [
                'nama_perkiraan' => $chartAccount ? $chartAccount->nama_akun : ($group->first()->nama_perkiraan ?? ''),
                'amount' => $group->sum('debit'),
            ];
        });

        return view('admin.reports.profit-loss-year', compact(
            'year',
            'revenue',
            'cogs',
            'operatingExpenses',
            'otherIncome',
            'tax',
            'grossProfit',
            'operatingProfit',
            'profitBeforeTax',
            'netProfit',
            'revenueAccounts',
            'cogsAccounts',
            'expenseAccounts',
            'otherIncomeAccounts',
            'taxAccounts',
            'startDate',
            'endDate'
        ));
    }

    /**
     * Export Buku Besar (Excel)
     */
    public function generalLedgerExport(Request $request)
    {
        $accountCode = $request->get('kd_perk');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');

        if (!$accountCode) {
            return redirect()->back()->with('error', 'Pilih kode akun terlebih dahulu');
        }

        // Get account info from Chart of Accounts
        $chartAccount = ChartOfAccount::where('kode_akun', $accountCode)->first();
        
        if (!$chartAccount) {
            return redirect()->back()->with('error', 'Kode akun tidak ditemukan di Chart of Accounts');
        }

        $entries = JournalEntry::where('kd_perk', $accountCode)
            ->when($startDate, function($q) use ($startDate) {
                return $q->whereDate('tanggal_transaksi', '>=', $startDate);
            })
            ->when($endDate, function($q) use ($endDate) {
                return $q->whereDate('tanggal_transaksi', '<=', $endDate);
            })
            ->orderBy('tanggal_transaksi', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();

        // Get balance position from Chart of Accounts
        $balancePosition = $chartAccount->pos_saldo ?? 'DEBIT';

        // Calculate opening balance - include saldo awal from Chart of Accounts
        $openingBalance = 0;
        
        // Add initial balance from Chart of Accounts
        if ($balancePosition === 'KREDIT') {
            $openingBalance = ($chartAccount->saldo_awal_kredit ?? 0) - ($chartAccount->saldo_awal_debet ?? 0);
        } else {
            $openingBalance = ($chartAccount->saldo_awal_debet ?? 0) - ($chartAccount->saldo_awal_kredit ?? 0);
        }
        
        // Add transactions before start date
        if ($startDate) {
            $openingEntries = JournalEntry::where('kd_perk', $accountCode)
                ->whereDate('tanggal_transaksi', '<', $startDate)
                ->get();
            
            foreach ($openingEntries as $entry) {
                $debit = $entry->debit ?? 0;
                $kredit = $entry->kredit ?? 0;
                
                if ($balancePosition === 'KREDIT') {
                    $openingBalance += ($kredit - $debit);
                } else {
                    $openingBalance += ($debit - $kredit);
                }
            }
        }

        // Calculate running balance
        $runningBalance = $openingBalance;
        $accountName = $chartAccount->nama_akun ?? '';

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table { border-collapse: collapse; width: 100%; font-family: Arial, sans-serif; font-size: 11px; }
        th { background-color: #90EE90; color: #000000; font-weight: bold; text-align: center; padding: 8px 6px; border: 1px solid #000000; }
        td { padding: 6px; border: 1px solid #CCCCCC; }
        .text-right { text-align: right; }
        .text-center { text-align: center; }
        .header-info { text-align: center; margin-bottom: 10px; }
    </style>
</head>
<body>
    <div class="header-info">
        <h2>POLJAMTECH</h2>
        <p>Jl. Lkr. Barat 3 No.1, Bagan Pete, Kec. Kota Baru, Kota Jambi, Jambi 36361</p>
        <h3>Buku Besar</h3>
        <p><strong>NAMA AKUN:</strong> ' . htmlspecialchars($accountName) . ' | <strong>KODE AKUN:</strong> ' . htmlspecialchars($accountCode) . ' | <strong>POS SALDO:</strong> ' . $balancePosition . '</p>
    </div>
    <table>
        <thead>
            <tr>
                <th>NO</th>
                <th>TANGGAL</th>
                <th>BUKTI TRANSAKSI</th>
                <th>KETERANGAN</th>
                <th>DEBET</th>
                <th>KREDIT</th>
                <th>SALDO</th>
            </tr>
        </thead>
        <tbody>
            <tr>
                <td class="text-center">-</td>
                <td class="text-center">-</td>
                <td class="text-center">-</td>
                <td>Saldo awal</td>
                <td class="text-right">-</td>
                <td class="text-right">-</td>
                <td class="text-right">' . ($openingBalance != 0 ? number_format($openingBalance, 0, ',', '.') : '-') . '</td>
            </tr>';

        $no = 1;
        foreach ($entries as $entry) {
            $debit = $entry->debit ?? 0;
            $kredit = $entry->kredit ?? 0;
            
            if ($balancePosition === 'KREDIT') {
                $runningBalance += ($kredit - $debit);
            } else {
                $runningBalance += ($debit - $kredit);
            }
            
            $html .= '<tr>
                <td class="text-center">' . $no++ . '</td>
                <td class="text-center">' . ($entry->tanggal_transaksi ? $entry->tanggal_transaksi->format('d-M-y') : '-') . '</td>
                <td class="text-center">' . htmlspecialchars($entry->bukti_transaksi ?: '-') . '</td>
                <td>' . htmlspecialchars($entry->keterangan ?: '-') . '</td>
                <td class="text-right">' . ($debit > 0 ? number_format($debit, 0, ',', '.') : '-') . '</td>
                <td class="text-right">' . ($kredit > 0 ? number_format($kredit, 0, ',', '.') : '-') . '</td>
                <td class="text-right">' . (abs($runningBalance) > 0.01 ? number_format($runningBalance, 0, ',', '.') : '-') . '</td>
            </tr>';
        }

        $html .= '</tbody>
    </table>
</body>
</html>';

        return response($html)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="Buku_Besar_' . $accountCode . '_' . date('Y-m-d_H-i-s') . '.xls"');
    }

    /**
     * Export Laporan Laba Rugi Per Proyek (Excel)
     */
    public function profitLossProjectExport(Request $request)
    {
        $projectCode = $request->get('kd_project');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');

        if (!$projectCode) {
            return redirect()->back()->with('error', 'Pilih proyek terlebih dahulu');
        }

        // Get project info from Projects table
        $project = Project::where('kode_bantu', $projectCode)->first();
        
        if (!$project) {
            return redirect()->back()->with('error', 'Kode proyek tidak ditemukan di Master Data Projects');
        }

        $entries = JournalEntry::where('kd_project', $projectCode)
            ->when($startDate, function($q) use ($startDate) {
                return $q->whereDate('tanggal_transaksi', '>=', $startDate);
            })
            ->when($endDate, function($q) use ($endDate) {
                return $q->whereDate('tanggal_transaksi', '<=', $endDate);
            })
            ->get();

        // Calculate values
        $revenue = $entries->filter(function($entry) {
            return preg_match('/^4/', $entry->kd_perk ?? '');
        })->sum('kredit');

        $cogs = $entries->filter(function($entry) {
            return preg_match('/^5/', $entry->kd_perk ?? '');
        })->sum('debit');

        $operatingExpenses = $entries->filter(function($entry) {
            return preg_match('/^6/', $entry->kd_perk ?? '');
        })->sum('debit');

        $otherIncome = $entries->filter(function($entry) {
            return preg_match('/^[78]/', $entry->kd_perk ?? '');
        })->sum(function($entry) {
            return ($entry->kredit ?? 0) - ($entry->debit ?? 0);
        });

        $tax = $entries->filter(function($entry) {
            return preg_match('/^9/', $entry->kd_perk ?? '');
        })->sum('debit');

        $grossProfit = $revenue - $cogs;
        $operatingProfit = $grossProfit - $operatingExpenses;
        $profitBeforeTax = $operatingProfit + $otherIncome;
        $netProfit = $profitBeforeTax - $tax;

        // Group by account
        $revenueAccounts = $entries->filter(function($entry) {
            return preg_match('/^4/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('kredit'),
            ];
        });

        $cogsAccounts = $entries->filter(function($entry) {
            return preg_match('/^5/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('debit'),
            ];
        });

        $expenseAccounts = $entries->filter(function($entry) {
            return preg_match('/^6/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('debit'),
            ];
        });

        $otherIncomeAccounts = $entries->filter(function($entry) {
            return preg_match('/^[78]/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            $debit = $group->sum('debit');
            $kredit = $group->sum('kredit');
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $kredit - $debit,
            ];
        });

        $taxAccounts = $entries->filter(function($entry) {
            return preg_match('/^9/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('debit'),
            ];
        });

        $projectName = $project->nama_proyek ?? '';

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table { border-collapse: collapse; width: 100%; font-family: Arial, sans-serif; font-size: 11px; }
        th { background-color: #90EE90; color: #000000; font-weight: bold; text-align: center; padding: 8px 6px; border: 1px solid #000000; }
        td { padding: 6px; border: 1px solid #CCCCCC; }
        .text-right { text-align: right; }
        .text-left { text-align: left; }
        .header-info { text-align: center; margin-bottom: 10px; }
        .section-header { background-color: #E7E6E6; font-weight: bold; }
        .total-row { background-color: #E7E6E6; font-weight: bold; }
    </style>
</head>
<body>
    <div class="header-info">
        <h2>POLJAMTECH WORKSHOP TECHNOLOGY</h2>
        <p>Jl. Lkr. Barat 3 No.1, Bagan Pete, Kec. Kota Baru, Kota Jambi, Jambi 36361</p>
        <h3>LAPORAN LABA RUGI PER PROJECT</h3>
        <p style="background-color: #ADD8E6; padding: 5px;"><strong>' . htmlspecialchars($projectName) . '</strong></p>
    </div>
    <table>
        <thead>
            <tr>
                <th class="text-left">Keterangan</th>
                <th class="text-right">Jumlah</th>
                <th class="text-right">TOTAL</th>
            </tr>
        </thead>
        <tbody>
            <tr class="section-header">
                <td colspan="3">Pendapatan</td>
            </tr>';

        foreach ($revenueAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH PENDAPATAN</td>
                <td class="text-right">' . ($revenue > 0 ? number_format($revenue, 0, ',', '.') : '-') . '</td>
                <td class="text-right">' . ($revenue > 0 ? number_format($revenue, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="section-header">
                <td colspan="3">HARGA POKOK PENJUALAN</td>
            </tr>';

        foreach ($cogsAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">HARGA POKOK PENJUALAN</td>
                <td class="text-right">' . ($cogs > 0 ? number_format($cogs, 0, ',', '.') : '-') . '</td>
                <td class="text-right">' . ($cogs > 0 ? number_format($cogs, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row">
                <td class="text-left">LABA BRUTO</td>
                <td class="text-right">' . ($grossProfit < 0 ? '(' . number_format(abs($grossProfit), 0, ',', '.') . ')' : number_format($grossProfit, 0, ',', '.')) . '</td>
                <td class="text-right">' . ($grossProfit < 0 ? '(' . number_format(abs($grossProfit), 0, ',', '.') . ')' : number_format($grossProfit, 0, ',', '.')) . '</td>
            </tr>
            <tr class="section-header">
                <td colspan="3">BEBAN ADMINISTRASI & UMUM</td>
            </tr>';

        foreach ($expenseAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">Jumlah Beban Administrasi Dan Umum</td>
                <td class="text-right">' . ($operatingExpenses > 0 ? number_format($operatingExpenses, 0, ',', '.') : '-') . '</td>
                <td class="text-right">' . ($operatingExpenses > 0 ? number_format($operatingExpenses, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row">
                <td class="text-left">LABA (RUGI) Usaha</td>
                <td class="text-right">' . ($operatingProfit < 0 ? '(' . number_format(abs($operatingProfit), 0, ',', '.') . ')' : number_format($operatingProfit, 0, ',', '.')) . '</td>
                <td class="text-right">' . ($operatingProfit < 0 ? '(' . number_format(abs($operatingProfit), 0, ',', '.') . ')' : number_format($operatingProfit, 0, ',', '.')) . '</td>
            </tr>
            <tr class="section-header">
                <td colspan="3">PENDAPATAN (BEBAN) LAIN - LAIN</td>
            </tr>';

        foreach ($otherIncomeAccounts as $account) {
            if ($account['amount'] != 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . ($account['amount'] != 0 ? number_format($account['amount'], 0, ',', '.') : '-') . '</td>
                    <td class="text-right">' . ($account['amount'] != 0 ? number_format($account['amount'], 0, ',', '.') : '-') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH PENDAPATAN (BEBAN) LAIN - LAIN</td>
                <td class="text-right">' . ($otherIncome != 0 ? number_format($otherIncome, 0, ',', '.') : '-') . '</td>
                <td class="text-right">' . ($otherIncome != 0 ? number_format($otherIncome, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row">
                <td class="text-left">LABA (RUGI) SEBELUM PAJAK</td>
                <td class="text-right">' . ($profitBeforeTax < 0 ? '(' . number_format(abs($profitBeforeTax), 0, ',', '.') . ')' : number_format($profitBeforeTax, 0, ',', '.')) . '</td>
                <td class="text-right">' . ($profitBeforeTax < 0 ? '(' . number_format(abs($profitBeforeTax), 0, ',', '.') . ')' : number_format($profitBeforeTax, 0, ',', '.')) . '</td>
            </tr>
            <tr class="section-header">
                <td colspan="3">PANJAK PENGHASILAN</td>
            </tr>';

        foreach ($taxAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH PPH</td>
                <td class="text-right">' . ($tax > 0 ? number_format($tax, 0, ',', '.') : '-') . '</td>
                <td class="text-right">' . ($tax > 0 ? number_format($tax, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row" style="background-color: #ADD8E6; font-weight: bold;">
                <td class="text-left">LABA SETELAH PAJAK</td>
                <td class="text-right">' . ($netProfit < 0 ? '(' . number_format(abs($netProfit), 0, ',', '.') . ')' : number_format($netProfit, 0, ',', '.')) . '</td>
                <td class="text-right">' . ($netProfit < 0 ? '(' . number_format(abs($netProfit), 0, ',', '.') . ')' : number_format($netProfit, 0, ',', '.')) . '</td>
            </tr>
        </tbody>
    </table>
</body>
</html>';

        return response($html)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="Laporan_Laba_Rugi_Proyek_' . $projectCode . '_' . date('Y-m-d_H-i-s') . '.xls"');
    }

    /**
     * Export Laporan Laba Rugi Per Bulan (Excel)
     */
    public function profitLossMonthExport(Request $request)
    {
        $month = $request->get('month', date('Y-m'));
        $startDate = $month . '-01';
        $endDate = date('Y-m-t', strtotime($startDate));

        $entries = JournalEntry::whereBetween('tanggal_transaksi', [$startDate, $endDate])->get();

        // Calculate values (same logic as profitLossByMonth)
        $revenue = $entries->filter(function($entry) {
            return preg_match('/^4/', $entry->kd_perk ?? '');
        })->sum('kredit');

        $cogs = $entries->filter(function($entry) {
            return preg_match('/^5/', $entry->kd_perk ?? '');
        })->sum('debit');

        $operatingExpenses = $entries->filter(function($entry) {
            return preg_match('/^6/', $entry->kd_perk ?? '');
        })->sum('debit');

        $otherIncome = $entries->filter(function($entry) {
            return preg_match('/^[78]/', $entry->kd_perk ?? '');
        })->sum(function($entry) {
            return ($entry->kredit ?? 0) - ($entry->debit ?? 0);
        });

        $tax = $entries->filter(function($entry) {
            return preg_match('/^9/', $entry->kd_perk ?? '');
        })->sum('debit');

        $grossProfit = $revenue - $cogs;
        $operatingProfit = $grossProfit - $operatingExpenses;
        $profitBeforeTax = $operatingProfit + $otherIncome;
        $netProfit = $profitBeforeTax - $tax;

        // Group by account
        $revenueAccounts = $entries->filter(function($entry) {
            return preg_match('/^4/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('kredit'),
            ];
        });

        $cogsAccounts = $entries->filter(function($entry) {
            return preg_match('/^5/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('debit'),
            ];
        });

        $expenseAccounts = $entries->filter(function($entry) {
            return preg_match('/^6/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('debit'),
            ];
        });

        $otherIncomeAccounts = $entries->filter(function($entry) {
            return preg_match('/^[78]/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            $debit = $group->sum('debit');
            $kredit = $group->sum('kredit');
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $kredit - $debit,
            ];
        });

        $taxAccounts = $entries->filter(function($entry) {
            return preg_match('/^9/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('debit'),
            ];
        });

        $monthName = \Carbon\Carbon::parse($startDate)->locale('id')->isoFormat('MMMM YYYY');
        $monthNumber = \Carbon\Carbon::parse($startDate)->format('n');

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table { border-collapse: collapse; width: 100%; font-family: Arial, sans-serif; font-size: 11px; }
        th { background-color: #90EE90; color: #000000; font-weight: bold; text-align: center; padding: 8px 6px; border: 1px solid #000000; }
        td { padding: 6px; border: 1px solid #CCCCCC; }
        .text-right { text-align: right; }
        .text-left { text-align: left; }
        .header-info { text-align: center; margin-bottom: 10px; }
        .section-header { background-color: #E7E6E6; font-weight: bold; }
        .total-row { background-color: #E7E6E6; font-weight: bold; }
    </style>
</head>
<body>
    <div class="header-info">
        <h2>LAPORAN LABA RUGI</h2>
        <p style="font-size: 14px;"><strong>' . $monthNumber . '</strong></p>
        <p>' . $monthName . '</p>
    </div>
    <table>
        <thead>
            <tr>
                <th class="text-left">Keterangan</th>
                <th class="text-right">Jumlah</th>
            </tr>
        </thead>
        <tbody>
            <tr class="section-header">
                <td>Pendapatan :</td>
                <td></td>
            </tr>';

        foreach ($revenueAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH PENDAPATAN</td>
                <td class="text-right">' . ($revenue > 0 ? number_format($revenue, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="section-header">
                <td>HARGA POKOK PENJUALAN</td>
                <td></td>
            </tr>';

        foreach ($cogsAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH HARGA POKOK</td>
                <td class="text-right">' . ($cogs > 0 ? number_format($cogs, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row">
                <td class="text-left">LABA BRUTO</td>
                <td class="text-right">' . ($grossProfit < 0 ? '(' . number_format(abs($grossProfit), 0, ',', '.') . ')' : number_format($grossProfit, 0, ',', '.')) . '</td>
            </tr>
            <tr class="section-header">
                <td>BEBAN ADMINISTRASI & UMUM</td>
                <td></td>
            </tr>';

        foreach ($expenseAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">Jumlah Beban Administrasi Dan Umum</td>
                <td class="text-right">' . ($operatingExpenses > 0 ? number_format($operatingExpenses, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row">
                <td class="text-left">LABA (RUGI) Usaha</td>
                <td class="text-right">' . ($operatingProfit < 0 ? '(' . number_format(abs($operatingProfit), 0, ',', '.') . ')' : number_format($operatingProfit, 0, ',', '.')) . '</td>
            </tr>
            <tr class="section-header">
                <td>PENDAPATAN (BEBAN) LAIN - LAIN</td>
                <td></td>
            </tr>';

        foreach ($otherIncomeAccounts as $account) {
            if ($account['amount'] != 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . ($account['amount'] != 0 ? number_format($account['amount'], 0, ',', '.') : '-') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH PENDAPATAN (BEBAN) LAIN - LAIN</td>
                <td class="text-right">' . ($otherIncome != 0 ? number_format($otherIncome, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row">
                <td class="text-left">LABA (RUGI)</td>
                <td class="text-right">' . ($profitBeforeTax < 0 ? '(' . number_format(abs($profitBeforeTax), 0, ',', '.') . ')' : number_format($profitBeforeTax, 0, ',', '.')) . '</td>
            </tr>
            <tr class="section-header">
                <td>PAJAK PENGHASILAN</td>
                <td></td>
            </tr>';

        foreach ($taxAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH PPH</td>
                <td class="text-right">' . ($tax > 0 ? number_format($tax, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row" style="background-color: #ADD8E6; font-weight: bold;">
                <td class="text-left">LABA RUGI SETELAH PAJAK</td>
                <td class="text-right">' . ($netProfit < 0 ? '(' . number_format(abs($netProfit), 0, ',', '.') . ')' : number_format($netProfit, 0, ',', '.')) . '</td>
            </tr>
        </tbody>
    </table>
</body>
</html>';

        return response($html)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="Laporan_Laba_Rugi_Bulan_' . $month . '_' . date('Y-m-d_H-i-s') . '.xls"');
    }

    /**
     * Export Laporan Laba Rugi Per Tahun (Excel)
     */
    public function profitLossYearExport(Request $request)
    {
        $year = $request->get('year', date('Y'));
        $startDate = $year . '-01-01';
        $endDate = $year . '-12-31';

        $entries = JournalEntry::whereBetween('tanggal_transaksi', [$startDate, $endDate])->get();

        // Calculate values (same logic as profitLossByYear)
        $revenue = $entries->filter(function($entry) {
            return preg_match('/^4/', $entry->kd_perk ?? '');
        })->sum('kredit');

        $cogs = $entries->filter(function($entry) {
            return preg_match('/^5/', $entry->kd_perk ?? '');
        })->sum('debit');

        $operatingExpenses = $entries->filter(function($entry) {
            return preg_match('/^6/', $entry->kd_perk ?? '');
        })->sum('debit');

        $otherIncome = $entries->filter(function($entry) {
            return preg_match('/^[78]/', $entry->kd_perk ?? '');
        })->sum(function($entry) {
            return ($entry->kredit ?? 0) - ($entry->debit ?? 0);
        });

        $tax = $entries->filter(function($entry) {
            return preg_match('/^9/', $entry->kd_perk ?? '');
        })->sum('debit');

        $grossProfit = $revenue - $cogs;
        $operatingProfit = $grossProfit - $operatingExpenses;
        $profitBeforeTax = $operatingProfit + $otherIncome;
        $netProfit = $profitBeforeTax - $tax;

        // Group by account
        $revenueAccounts = $entries->filter(function($entry) {
            return preg_match('/^4/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('kredit'),
            ];
        });

        $cogsAccounts = $entries->filter(function($entry) {
            return preg_match('/^5/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('debit'),
            ];
        });

        $expenseAccounts = $entries->filter(function($entry) {
            return preg_match('/^6/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('debit'),
            ];
        });

        $otherIncomeAccounts = $entries->filter(function($entry) {
            return preg_match('/^[78]/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            $debit = $group->sum('debit');
            $kredit = $group->sum('kredit');
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $kredit - $debit,
            ];
        });

        $taxAccounts = $entries->filter(function($entry) {
            return preg_match('/^9/', $entry->kd_perk ?? '');
        })->groupBy('kd_perk')->map(function($group) {
            return [
                'nama_perkiraan' => $group->first()->nama_perkiraan ?? '',
                'amount' => $group->sum('debit'),
            ];
        });

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table { border-collapse: collapse; width: 100%; font-family: Arial, sans-serif; font-size: 11px; }
        th { background-color: #90EE90; color: #000000; font-weight: bold; text-align: center; padding: 8px 6px; border: 1px solid #000000; }
        td { padding: 6px; border: 1px solid #CCCCCC; }
        .text-right { text-align: right; }
        .text-left { text-align: left; }
        .header-info { text-align: center; margin-bottom: 10px; }
        .section-header { background-color: #E7E6E6; font-weight: bold; }
        .total-row { background-color: #E7E6E6; font-weight: bold; }
    </style>
</head>
<body>
    <div class="header-info">
        <h2>POLJAMTECH WORKSHOP TECHNOLOGY</h2>
        <p>Jl. Lkr. Barat 3 No.1, Bagan Pete, Kec. Kota Baru, Kota Jambi, Jambi 3636</p>
        <h3>LAPORAN LABA RUGI</h3>
        <p style="background-color: #E7E6E6; padding: 5px;"><strong>TAHUN ' . $year . '</strong></p>
    </div>
    <table>
        <thead>
            <tr>
                <th class="text-left">Keterangan</th>
                <th class="text-right">Jumlah</th>
            </tr>
        </thead>
        <tbody>
            <tr class="section-header">
                <td>Pendapatan :</td>
                <td></td>
            </tr>';

        foreach ($revenueAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH PENDAPATAN</td>
                <td class="text-right">' . ($revenue > 0 ? number_format($revenue, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="section-header">
                <td>HARGA POKOK PENJUALAN</td>
                <td></td>
            </tr>';

        foreach ($cogsAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH HARGA POKOK PENJUALAN</td>
                <td class="text-right">' . ($cogs > 0 ? number_format($cogs, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row">
                <td class="text-left">LABA BRUTO</td>
                <td class="text-right">' . ($grossProfit < 0 ? '(' . number_format(abs($grossProfit), 0, ',', '.') . ')' : number_format($grossProfit, 0, ',', '.')) . '</td>
            </tr>
            <tr class="section-header">
                <td>BEBAN ADMINISTRASI & UMUM</td>
                <td></td>
            </tr>';

        foreach ($expenseAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">Jumlah Beban Administrasi Dan Umum</td>
                <td class="text-right">' . ($operatingExpenses > 0 ? number_format($operatingExpenses, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row">
                <td class="text-left">LABA (RUGI) Usaha</td>
                <td class="text-right">' . ($operatingProfit < 0 ? '(' . number_format(abs($operatingProfit), 0, ',', '.') . ')' : number_format($operatingProfit, 0, ',', '.')) . '</td>
            </tr>
            <tr class="section-header">
                <td>PENDAPATAN (BEBAN) LAIN - LAIN</td>
                <td></td>
            </tr>';

        foreach ($otherIncomeAccounts as $account) {
            if ($account['amount'] != 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . ($account['amount'] != 0 ? number_format($account['amount'], 0, ',', '.') : '-') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH PENDAPATAN (BEBAN) LAIN - LAIN</td>
                <td class="text-right">' . ($otherIncome != 0 ? number_format($otherIncome, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row">
                <td class="text-left">LABA (RUGI) SEBELUM PAJAK</td>
                <td class="text-right">' . ($profitBeforeTax < 0 ? '(' . number_format(abs($profitBeforeTax), 0, ',', '.') . ')' : number_format($profitBeforeTax, 0, ',', '.')) . '</td>
            </tr>
            <tr class="section-header">
                <td>PPH</td>
                <td></td>
            </tr>';

        foreach ($taxAccounts as $account) {
            if ($account['amount'] > 0) {
                $html .= '<tr>
                    <td class="text-left" style="padding-left: 20px;">' . strtoupper(htmlspecialchars($account['nama_perkiraan'])) . '</td>
                    <td class="text-right">' . number_format($account['amount'], 0, ',', '.') . '</td>
                </tr>';
            }
        }

        $html .= '<tr class="total-row">
                <td class="text-left">JUMLAH PPH</td>
                <td class="text-right">' . ($tax > 0 ? number_format($tax, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr class="total-row" style="background-color: #ADD8E6; font-weight: bold;">
                <td class="text-left">LABA RUGI SETELAH PAJAK</td>
                <td class="text-right">' . ($netProfit < 0 ? '(' . number_format(abs($netProfit), 0, ',', '.') . ')' : number_format($netProfit, 0, ',', '.')) . '</td>
            </tr>
        </tbody>
    </table>
</body>
</html>';

        return response($html)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="Laporan_Laba_Rugi_Tahun_' . $year . '_' . date('Y-m-d_H-i-s') . '.xls"');
    }

    /**
     * Neraca Saldo (Trial Balance)
     */
    public function trialBalance(Request $request)
    {
        try {
            $endDate = $request->get('end_date', date('Y-m-d'));
            $showOnlyWithBalance = $request->get('show_balance_only', false);

            // Initialize default values
            $accounts = collect();
            $groupedAccounts = collect();
            $totalAktiva = 0;
            $totalKewajiban = 0;
            $totalEkuitas = 0;

        // Get all chart of accounts
        $chartAccounts = ChartOfAccount::active()
            ->orderBy('kode_akun')
            ->get()
            ->keyBy('kode_akun');

        // Get all journal entries up to end date
        $entries = JournalEntry::whereDate('tanggal_transaksi', '<=', $endDate)
            ->whereNotNull('kd_perk')
            ->orderBy('kd_perk')
            ->get();

        // Group by account code and calculate balances
        if ($entries->count() > 0) {
            $accounts = $entries->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
                $firstEntry = $group->first();
                $kdPerk = $firstEntry->kd_perk ?? '';
                
                // Get account info from Chart of Accounts
                $chartAccount = $chartAccounts->get($kdPerk);
                
                if (!$chartAccount) {
                    $namaPerkiraan = $firstEntry->nama_perkiraan ?? '';
                    $balancePosition = 'DEBIT';
                    if (preg_match('/^4|^[78]|^2|^3/', $kdPerk)) {
                        $balancePosition = 'KREDIT';
                    }
                } else {
                    $namaPerkiraan = $chartAccount->nama_akun;
                    $balancePosition = $chartAccount->pos_saldo;
                }
                
                $totalDebit = $group->sum('debit') ?? 0;
                $totalKredit = $group->sum('kredit') ?? 0;
                
                // Calculate balance based on position
                $balance = 0;
                if ($balancePosition === 'KREDIT') {
                    $balance = $totalKredit - $totalDebit;
                } else {
                    $balance = $totalDebit - $totalKredit;
                }
                
                // Determine category for grouping
                $category = 'Lainnya';
                if (preg_match('/^1/', $kdPerk)) {
                    $category = 'Aktiva Lancar';
                } elseif (preg_match('/^12/', $kdPerk)) {
                    $category = 'Aktiva Tetap';
                } elseif (preg_match('/^2/', $kdPerk)) {
                    $category = 'Kewajiban';
                } elseif (preg_match('/^3/', $kdPerk)) {
                    $category = 'Ekuitas';
                }
                
                return [
                    'kd_perk' => $kdPerk,
                    'nama_perkiraan' => $namaPerkiraan,
                    'balance_position' => $balancePosition,
                    'debit' => $totalDebit,
                    'kredit' => $totalKredit,
                    'balance' => $balance,
                    'category' => $category,
                ];
            })->values();

            // Hitung laba rugi bersih dari akun laba rugi
            $profitLossAccounts = $entries->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
                $firstEntry = $group->first();
                $kdPerk = $firstEntry->kd_perk ?? '';

                $chartAccount = $chartAccounts->get($kdPerk);

                if (!$chartAccount) {
                    $balancePosition = 'DEBIT';
                    if (preg_match('/^4|^[78]|^2|^3/', $kdPerk)) {
                        $balancePosition = 'KREDIT';
                    }
                    $reportPosition = 'NERACA';
                    if (preg_match('/^[4-9]/', $kdPerk)) {
                        $reportPosition = 'LABA RUGI';
                    }
                } else {
                    $balancePosition = $chartAccount->pos_saldo;
                    $reportPosition = $chartAccount->pos_laporan;
                }

                $totalDebit = $group->sum('debit') ?? 0;
                $totalKredit = $group->sum('kredit') ?? 0;

                $saldoAwalDebet = $chartAccount ? ($chartAccount->saldo_awal_debet ?? 0) : 0;
                $saldoAwalKredit = $chartAccount ? ($chartAccount->saldo_awal_kredit ?? 0) : 0;

                $totalDebit += $saldoAwalDebet;
                $totalKredit += $saldoAwalKredit;

                $profitLossDebit = 0;
                $profitLossKredit = 0;
                if ($reportPosition === 'LABA RUGI') {
                    if (preg_match('/^4/', $kdPerk)) {
                        $profitLossKredit = $totalKredit;
                    } elseif (preg_match('/^[5-9]/', $kdPerk)) {
                        $profitLossDebit = $totalDebit;
                    }
                }

                return [
                    'profit_loss_debit' => $profitLossDebit,
                    'profit_loss_kredit' => $profitLossKredit,
                ];
            })->values();

            $totalProfitLossDebit = $profitLossAccounts->sum('profit_loss_debit');
            $totalProfitLossKredit = $profitLossAccounts->sum('profit_loss_kredit');
            $netProfitLoss = $totalProfitLossKredit - $totalProfitLossDebit;

            if (abs($netProfitLoss) > 0.01) {
                $existing31004 = $accounts->firstWhere('kd_perk', '310-04');
                if (!$existing31004) {
                    $account31004 = $chartAccounts->get('310-04');
                    $nama31004 = $account31004 ? $account31004->nama_akun : 'LABA (RUGI) TAHUN BERJALAN';
                    $balancePosition31004 = $account31004 ? $account31004->pos_saldo : 'KREDIT';

                    $accounts->push([
                        'kd_perk' => '310-04',
                        'nama_perkiraan' => $nama31004,
                        'balance_position' => $balancePosition31004,
                        'debit' => 0,
                        'kredit' => 0,
                        'balance' => $netProfitLoss,
                        'category' => 'Ekuitas',
                    ]);
                }
            }

            // Filter only accounts with balance if requested
            if ($showOnlyWithBalance) {
                $accounts = $accounts->filter(function($account) {
                    return abs($account['balance']) > 0.01;
                })->values();
            }

            // Group by category
            $groupedAccounts = $accounts->groupBy('category');

            // Calculate totals
            $totalAktiva = $accounts->filter(function($acc) {
                return in_array($acc['category'], ['Aktiva Lancar', 'Aktiva Tetap']);
            })->sum('balance');

            $totalKewajiban = $accounts->filter(function($acc) {
                return $acc['category'] === 'Kewajiban';
            })->sum('balance');

            $totalEkuitas = $accounts->filter(function($acc) {
                return $acc['category'] === 'Ekuitas';
            })->sum('balance');
        }

            return view('admin.reports.trial-balance', compact(
                'accounts',
                'groupedAccounts',
                'totalAktiva',
                'totalKewajiban',
                'totalEkuitas',
                'endDate',
                'showOnlyWithBalance'
            ));
        } catch (\Exception $e) {
            \Log::error('Trial Balance Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return view('admin.reports.trial-balance', [
                'accounts' => collect(),
                'groupedAccounts' => collect(),
                'totalAktiva' => 0,
                'totalKewajiban' => 0,
                'totalEkuitas' => 0,
                'endDate' => $request->get('end_date', date('Y-m-d')),
                'showOnlyWithBalance' => $request->get('show_balance_only', false),
                'error' => 'Terjadi kesalahan: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Neraca Lajur (Worksheet)
     */
    public function worksheet(Request $request)
    {
        try {
            $endDate = $request->get('end_date', date('Y-m-d'));

            // Initialize default values
            $accounts = collect();
            $totalTrialBalanceDebit = 0;
            $totalTrialBalanceKredit = 0;
            $totalProfitLossDebit = 0;
            $totalProfitLossKredit = 0;
            $totalBalanceSheetDebit = 0;
            $totalBalanceSheetKredit = 0;
            $netProfitLoss = 0;

        // Get all chart of accounts
        $chartAccounts = ChartOfAccount::active()
            ->orderBy('kode_akun')
            ->get()
            ->keyBy('kode_akun');

        // Get all journal entries up to end date
        $entries = JournalEntry::whereDate('tanggal_transaksi', '<=', $endDate)
            ->whereNotNull('kd_perk')
            ->orderBy('kd_perk')
            ->get();

        // Group by account code
        if ($entries->count() > 0) {
            $accounts = $entries->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
                $firstEntry = $group->first();
                $kdPerk = $firstEntry->kd_perk ?? '';
                
                // Get account info from Chart of Accounts
                $chartAccount = $chartAccounts->get($kdPerk);
                
                if (!$chartAccount) {
                    // Fallback to entry data if not found in Chart of Accounts
                    $namaPerkiraan = $firstEntry->nama_perkiraan ?? '';
                    $balancePosition = 'DEBIT';
                    if (preg_match('/^4|^[78]|^2|^3/', $kdPerk)) {
                        $balancePosition = 'KREDIT';
                    }
                    $reportPosition = 'NERACA';
                    if (preg_match('/^[4-9]/', $kdPerk)) {
                        $reportPosition = 'LABA RUGI';
                    }
                } else {
                    $namaPerkiraan = $chartAccount->nama_akun;
                    $balancePosition = $chartAccount->pos_saldo;
                    $reportPosition = $chartAccount->pos_laporan;
                }
                
                $totalDebit = $group->sum('debit') ?? 0;
                $totalKredit = $group->sum('kredit') ?? 0;
                
                // Add saldo awal from Chart of Accounts
                $saldoAwalDebet = $chartAccount ? ($chartAccount->saldo_awal_debet ?? 0) : 0;
                $saldoAwalKredit = $chartAccount ? ($chartAccount->saldo_awal_kredit ?? 0) : 0;
                
                $totalDebit += $saldoAwalDebet;
                $totalKredit += $saldoAwalKredit;
                
                // Calculate trial balance
                $trialBalanceDebit = 0;
                $trialBalanceKredit = 0;
                if ($balancePosition === 'DEBIT') {
                    $trialBalanceDebit = max(0, $totalDebit - $totalKredit);
                } else {
                    $trialBalanceKredit = max(0, $totalKredit - $totalDebit);
                }
                
                // For profit/loss accounts, extend to LABA RUGI
                $profitLossDebit = 0;
                $profitLossKredit = 0;
                if ($reportPosition === 'LABA RUGI') {
                    if (preg_match('/^4/', $kdPerk)) {
                        // Revenue - goes to KREDIT
                        $profitLossKredit = $totalKredit;
                    } elseif (preg_match('/^[5-9]/', $kdPerk)) {
                        // Expenses/COGS - goes to DEBIT
                        $profitLossDebit = $totalDebit;
                    }
                }
                
                // For balance sheet accounts, extend to NERACA
                $balanceSheetDebit = 0;
                $balanceSheetKredit = 0;
                if ($reportPosition === 'NERACA') {
                    if ($balancePosition === 'DEBIT') {
                        $balanceSheetDebit = $trialBalanceDebit;
                    } else {
                        $balanceSheetKredit = $trialBalanceKredit;
                    }
                }
                
                // Special handling: Laba (Rugi) Tahun Berjalan (310-04) should appear in LABA RUGI KREDIT
                if ($kdPerk === '310-04' && $reportPosition === 'NERACA') {
                    // This will be handled separately after calculating net profit/loss
                }
                
                return [
                    'kd_perk' => $kdPerk,
                    'nama_perkiraan' => $namaPerkiraan,
                    'balance_position' => $balancePosition,
                    'report_position' => $reportPosition,
                    'trial_balance_debit' => $trialBalanceDebit,
                    'trial_balance_kredit' => $trialBalanceKredit,
                    'profit_loss_debit' => $profitLossDebit,
                    'profit_loss_kredit' => $profitLossKredit,
                    'balance_sheet_debit' => $balanceSheetDebit,
                    'balance_sheet_kredit' => $balanceSheetKredit,
                ];
            })->values();

            // Calculate totals
            $totalTrialBalanceDebit = $accounts->sum('trial_balance_debit');
            $totalTrialBalanceKredit = $accounts->sum('trial_balance_kredit');
            
            $totalProfitLossDebit = $accounts->sum('profit_loss_debit');
            $totalProfitLossKredit = $accounts->sum('profit_loss_kredit');
            
            $totalBalanceSheetDebit = $accounts->sum('balance_sheet_debit');
            $totalBalanceSheetKredit = $accounts->sum('balance_sheet_kredit');
            
            // Calculate net profit/loss
            $netProfitLoss = $totalProfitLossKredit - $totalProfitLossDebit;
            
            // Add net profit/loss to LABA RUGI KREDIT (for 310-04 account)
            // This will be displayed in the view as a separate row
            // Then it will be transferred to NERACA DEBET
        }

            return view('admin.reports.worksheet', compact(
                'accounts',
                'totalTrialBalanceDebit',
                'totalTrialBalanceKredit',
                'totalProfitLossDebit',
                'totalProfitLossKredit',
                'totalBalanceSheetDebit',
                'totalBalanceSheetKredit',
                'netProfitLoss',
                'endDate'
            ));
        } catch (\Exception $e) {
            \Log::error('Worksheet Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return view('admin.reports.worksheet', [
                'accounts' => collect(),
                'totalTrialBalanceDebit' => 0,
                'totalTrialBalanceKredit' => 0,
                'totalProfitLossDebit' => 0,
                'totalProfitLossKredit' => 0,
                'totalBalanceSheetDebit' => 0,
                'totalBalanceSheetKredit' => 0,
                'netProfitLoss' => 0,
                'endDate' => $request->get('end_date', date('Y-m-d')),
                'error' => 'Terjadi kesalahan: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Laporan Perubahan Modal (Statement of Changes in Equity)
     */
    public function equityChanges(Request $request)
    {
        $endDate = $request->get('end_date', date('Y-m-d'));
        $startDate = $request->get('start_date', date('Y-01-01')); // Default to start of year

        // Get all chart of accounts
        $chartAccounts = ChartOfAccount::active()
            ->orderBy('kode_akun')
            ->get()
            ->keyBy('kode_akun');

        // Get equity accounts (starting with 3) - use Chart of Accounts to filter
        $equityAccountCodes = $chartAccounts->filter(function($account) {
            return preg_match('/^3/', $account->kode_akun) && $account->pos_laporan === 'NERACA';
        })->pluck('kode_akun')->toArray();

        $equityEntries = JournalEntry::whereIn('kd_perk', $equityAccountCodes)
            ->whereDate('tanggal_transaksi', '<=', $endDate)
            ->get();

        // Get beginning equity (before start date) - include saldo awal from Chart of Accounts
        $beginningEquity = 0;
        
        // Add saldo awal from Chart of Accounts for equity accounts
        foreach ($chartAccounts as $account) {
            if (preg_match('/^3/', $account->kode_akun) && $account->pos_laporan === 'NERACA') {
                $beginningEquity += ($account->saldo_awal_kredit ?? 0) - ($account->saldo_awal_debet ?? 0);
            }
        }
        
        $beginningEntries = JournalEntry::whereIn('kd_perk', $equityAccountCodes)
            ->whereDate('tanggal_transaksi', '<', $startDate)
            ->get();
        
        foreach ($beginningEntries as $entry) {
            $debit = $entry->debit ?? 0;
            $kredit = $entry->kredit ?? 0;
            $beginningEquity += ($kredit - $debit);
        }

        // Get equity changes during period
        $periodEntries = JournalEntry::whereIn('kd_perk', $equityAccountCodes)
            ->whereDate('tanggal_transaksi', '>=', $startDate)
            ->whereDate('tanggal_transaksi', '<=', $endDate)
            ->get();

        // Group by account - use Chart of Accounts for nama_akun
        $equityAccounts = $periodEntries->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $firstEntry = $group->first();
            $kdPerk = $firstEntry->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            $namaPerkiraan = $chartAccount ? $chartAccount->nama_akun : ($firstEntry->nama_perkiraan ?? '');
            
            $totalDebit = $group->sum('debit') ?? 0;
            $totalKredit = $group->sum('kredit') ?? 0;
            
            // Equity increases with KREDIT, decreases with DEBIT
            $change = $totalKredit - $totalDebit;
            
            return [
                'kd_perk' => $kdPerk,
                'nama_perkiraan' => $namaPerkiraan,
                'debit' => $totalDebit,
                'kredit' => $totalKredit,
                'change' => $change,
            ];
        })->values();

        // Get current year profit/loss (310-04)
        $currentYearProfitLoss = 0;
        $profitLossEntries = JournalEntry::where('kd_perk', '310-04')
            ->whereDate('tanggal_transaksi', '>=', $startDate)
            ->whereDate('tanggal_transaksi', '<=', $endDate)
            ->get();
        
        foreach ($profitLossEntries as $entry) {
            $debit = $entry->debit ?? 0;
            $kredit = $entry->kredit ?? 0;
            $currentYearProfitLoss += ($kredit - $debit);
        }

        // If no direct entry in 310-04, calculate from profit/loss accounts
        if (abs($currentYearProfitLoss) < 0.01) {
            // Get all chart of accounts
            $profitLossChartAccounts = ChartOfAccount::active()
                ->orderBy('kode_akun')
                ->get()
                ->keyBy('kode_akun');
            
            $profitLossAccounts = JournalEntry::whereDate('tanggal_transaksi', '>=', $startDate)
                ->whereDate('tanggal_transaksi', '<=', $endDate)
                ->get();
            
            // Revenue (4) - accounts with pos_laporan = 'LABA RUGI'
            $revenue = $profitLossAccounts->filter(function($entry) use ($profitLossChartAccounts) {
                $chartAccount = $profitLossChartAccounts->get($entry->kd_perk ?? '');
                return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^4/', $entry->kd_perk ?? '');
            })->sum('kredit');
            
            // COGS (5) - accounts with pos_laporan = 'LABA RUGI'
            $cogs = $profitLossAccounts->filter(function($entry) use ($profitLossChartAccounts) {
                $chartAccount = $profitLossChartAccounts->get($entry->kd_perk ?? '');
                return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^5/', $entry->kd_perk ?? '');
            })->sum('debit');
            
            // Expenses (6) - accounts with pos_laporan = 'LABA RUGI'
            $expenses = $profitLossAccounts->filter(function($entry) use ($profitLossChartAccounts) {
                $chartAccount = $profitLossChartAccounts->get($entry->kd_perk ?? '');
                return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^6/', $entry->kd_perk ?? '');
            })->sum('debit');
            
            // Other income/expenses (7-8) - accounts with pos_laporan = 'LABA RUGI'
            $otherIncome = $profitLossAccounts->filter(function($entry) use ($profitLossChartAccounts) {
                $chartAccount = $profitLossChartAccounts->get($entry->kd_perk ?? '');
                return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^[78]/', $entry->kd_perk ?? '');
            })->sum(function($entry) {
                return ($entry->kredit ?? 0) - ($entry->debit ?? 0);
            });
            
            // Tax (9) - accounts with pos_laporan = 'LABA RUGI'
            $tax = $profitLossAccounts->filter(function($entry) use ($profitLossChartAccounts) {
                $chartAccount = $profitLossChartAccounts->get($entry->kd_perk ?? '');
                return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^9/', $entry->kd_perk ?? '');
            })->sum('debit');
            
            $currentYearProfitLoss = $revenue - $cogs - $expenses + $otherIncome - $tax;
        }

        // Calculate total equity additions
        $totalAdditions = $equityAccounts->sum('change') + $currentYearProfitLoss;

        // Calculate ending equity
        $endingEquity = $beginningEquity + $totalAdditions;

        return view('admin.reports.equity-changes', compact(
            'beginningEquity',
            'equityAccounts',
            'currentYearProfitLoss',
            'totalAdditions',
            'endingEquity',
            'startDate',
            'endDate'
        ));
    }

    /**
     * Export Neraca Saldo (Excel)
     */
    public function trialBalanceExport(Request $request)
    {
        $endDate = $request->get('end_date', date('Y-m-d'));
        $showOnlyWithBalance = $request->get('show_balance_only', false);

        // Get all chart of accounts
        $chartAccounts = ChartOfAccount::active()
            ->orderBy('kode_akun')
            ->get()
            ->keyBy('kode_akun');

        // Get all journal entries up to end date
        $entries = JournalEntry::whereDate('tanggal_transaksi', '<=', $endDate)
            ->whereNotNull('kd_perk')
            ->orderBy('kd_perk')
            ->get();

        $accounts = collect();
        $groupedAccounts = collect();
        $totalAktiva = 0;
        $totalKewajiban = 0;
        $totalEkuitas = 0;

        // Group by account code and calculate balances
        if ($entries->count() > 0) {
            $accounts = $entries->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
                $firstEntry = $group->first();
                $kdPerk = $firstEntry->kd_perk ?? '';
                
                // Get account info from Chart of Accounts
                $chartAccount = $chartAccounts->get($kdPerk);
                
                if (!$chartAccount) {
                    $namaPerkiraan = $firstEntry->nama_perkiraan ?? '';
                    $balancePosition = 'DEBIT';
                    if (preg_match('/^4|^[78]|^2|^3/', $kdPerk)) {
                        $balancePosition = 'KREDIT';
                    }
                } else {
                    $namaPerkiraan = $chartAccount->nama_akun;
                    $balancePosition = $chartAccount->pos_saldo;
                }
                
                $totalDebit = $group->sum('debit') ?? 0;
                $totalKredit = $group->sum('kredit') ?? 0;
                
                // Add saldo awal from Chart of Accounts
                $saldoAwalDebet = $chartAccount ? ($chartAccount->saldo_awal_debet ?? 0) : 0;
                $saldoAwalKredit = $chartAccount ? ($chartAccount->saldo_awal_kredit ?? 0) : 0;
                
                $totalDebit += $saldoAwalDebet;
                $totalKredit += $saldoAwalKredit;
                
                // Calculate balance based on position
                $balance = 0;
                if ($balancePosition === 'KREDIT') {
                    $balance = $totalKredit - $totalDebit;
                } else {
                    $balance = $totalDebit - $totalKredit;
                }
                
                // Determine category for grouping based on kode akun pattern
                $category = 'Lainnya';
                if (preg_match('/^1[0-1]/', $kdPerk)) {
                    $category = 'Aktiva Lancar';
                } elseif (preg_match('/^12/', $kdPerk)) {
                    $category = 'Aktiva Tetap';
                } elseif (preg_match('/^2/', $kdPerk)) {
                    $category = 'Kewajiban';
                } elseif (preg_match('/^3/', $kdPerk)) {
                    $category = 'Ekuitas';
                }
                
                return [
                    'kd_perk' => $kdPerk,
                    'nama_perkiraan' => $namaPerkiraan,
                    'balance_position' => $balancePosition,
                    'debit' => $totalDebit,
                    'kredit' => $totalKredit,
                    'balance' => $balance,
                    'category' => $category,
                ];
            })->values();

            // Hitung laba rugi bersih dari akun laba rugi
            $profitLossAccounts = $entries->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
                $firstEntry = $group->first();
                $kdPerk = $firstEntry->kd_perk ?? '';

                $chartAccount = $chartAccounts->get($kdPerk);

                if (!$chartAccount) {
                    $balancePosition = 'DEBIT';
                    if (preg_match('/^4|^[78]|^2|^3/', $kdPerk)) {
                        $balancePosition = 'KREDIT';
                    }
                    $reportPosition = 'NERACA';
                    if (preg_match('/^[4-9]/', $kdPerk)) {
                        $reportPosition = 'LABA RUGI';
                    }
                } else {
                    $balancePosition = $chartAccount->pos_saldo;
                    $reportPosition = $chartAccount->pos_laporan;
                }

                $totalDebit = $group->sum('debit') ?? 0;
                $totalKredit = $group->sum('kredit') ?? 0;

                $saldoAwalDebet = $chartAccount ? ($chartAccount->saldo_awal_debet ?? 0) : 0;
                $saldoAwalKredit = $chartAccount ? ($chartAccount->saldo_awal_kredit ?? 0) : 0;

                $totalDebit += $saldoAwalDebet;
                $totalKredit += $saldoAwalKredit;

                $profitLossDebit = 0;
                $profitLossKredit = 0;
                if ($reportPosition === 'LABA RUGI') {
                    if (preg_match('/^4/', $kdPerk)) {
                        $profitLossKredit = $totalKredit;
                    } elseif (preg_match('/^[5-9]/', $kdPerk)) {
                        $profitLossDebit = $totalDebit;
                    }
                }

                return [
                    'profit_loss_debit' => $profitLossDebit,
                    'profit_loss_kredit' => $profitLossKredit,
                ];
            })->values();

            $totalProfitLossDebit = $profitLossAccounts->sum('profit_loss_debit');
            $totalProfitLossKredit = $profitLossAccounts->sum('profit_loss_kredit');
            $netProfitLoss = $totalProfitLossKredit - $totalProfitLossDebit;

            if (abs($netProfitLoss) > 0.01) {
                $existing31004 = $accounts->firstWhere('kd_perk', '310-04');
                if (!$existing31004) {
                    $account31004 = $chartAccounts->get('310-04');
                    $nama31004 = $account31004 ? $account31004->nama_akun : 'LABA (RUGI) TAHUN BERJALAN';
                    $balancePosition31004 = $account31004 ? $account31004->pos_saldo : 'KREDIT';

                    $accounts->push([
                        'kd_perk' => '310-04',
                        'nama_perkiraan' => $nama31004,
                        'balance_position' => $balancePosition31004,
                        'debit' => 0,
                        'kredit' => 0,
                        'balance' => $netProfitLoss,
                        'category' => 'Ekuitas',
                    ]);
                }
            }

            // Filter only accounts with balance if requested
            if ($showOnlyWithBalance) {
                $accounts = $accounts->filter(function($account) {
                    return abs($account['balance']) > 0.01;
                })->values();
            }

            // Group by category
            $groupedAccounts = $accounts->groupBy('category');

            // Calculate totals
            $totalAktiva = $accounts->filter(function($acc) {
                return in_array($acc['category'], ['Aktiva Lancar', 'Aktiva Tetap']);
            })->sum('balance');

            $totalKewajiban = $accounts->filter(function($acc) {
                return $acc['category'] === 'Kewajiban';
            })->sum('balance');

            $totalEkuitas = $accounts->filter(function($acc) {
                return $acc['category'] === 'Ekuitas';
            })->sum('balance');
        }

        $endDateFormatted = \Carbon\Carbon::parse($endDate)->format('d F Y');
        $printDate = \Carbon\Carbon::now()->format('d F Y');
        $printTime = \Carbon\Carbon::now()->format('H:i');

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table { border-collapse: collapse; width: 100%; font-family: Arial, sans-serif; font-size: 11px; }
        th { background-color: #90EE90; color: #000000; font-weight: bold; text-align: center; padding: 8px 6px; border: 1px solid #000000; }
        td { padding: 6px; border: 1px solid #CCCCCC; }
        .text-right { text-align: right; }
        .text-center { text-align: center; }
        .text-left { text-align: left; }
        .header-info { text-align: center; margin-bottom: 10px; }
        .total-row { font-weight: bold; }
    </style>
</head>
<body>
    <div class="header-info">
        <h2>POLJAMTECH WORKSHOP TECHNOLOGY</h2>
        <p>Jl. Lkr. Barat 3 No.1, Bagan Pete, Kec. Kota Baru, Kota Jambi, Jambi 36361</p>
        <h3>NERACA SALDO</h3>
        <p>Per ' . $endDateFormatted . '</p>
        <p>' . $printDate . ' Jam : ' . $printTime . '</p>
    </div>
    <table>
        <thead>
            <tr>
                <th class="text-left">KODE AKUN</th>
                <th class="text-left">NAMA AKUN</th>
                <th class="text-right">JUMLAH</th>
                <th class="text-right">TOTAL</th>
            </tr>
        </thead>
        <tbody>';

        // Aktiva Lancar
        $aktivaLancar = $accounts->filter(function($acc) {
            return $acc['category'] === 'Aktiva Lancar' && abs($acc['balance']) > 0.01;
        });
        
        if ($aktivaLancar->count() > 0) {
            foreach ($aktivaLancar as $account) {
                $html .= '<tr>
                    <td class="text-left">' . htmlspecialchars($account['kd_perk']) . '</td>
                    <td class="text-left">' . htmlspecialchars($account['nama_perkiraan']) . '</td>
                    <td class="text-right">' . number_format($account['balance'], 0, ',', '.') . '</td>
                    <td class="text-right"></td>
                </tr>';
            }
            
            $html .= '<tr class="total-row">
                <td class="text-left"></td>
                <td class="text-left">Jumlah Aktiva Lancar</td>
                <td class="text-right"></td>
                <td class="text-right">' . number_format($aktivaLancar->sum('balance'), 0, ',', '.') . '</td>
            </tr>';
        }

        // Aktiva Tetap
        $aktivaTetap = $accounts->filter(function($acc) {
            return $acc['category'] === 'Aktiva Tetap' && abs($acc['balance']) > 0.01;
        });
        
        if ($aktivaTetap->count() > 0) {
            foreach ($aktivaTetap as $account) {
                $html .= '<tr>
                    <td class="text-left">' . htmlspecialchars($account['kd_perk']) . '</td>
                    <td class="text-left">' . htmlspecialchars($account['nama_perkiraan']) . '</td>
                    <td class="text-right">' . number_format($account['balance'], 0, ',', '.') . '</td>
                    <td class="text-right"></td>
                </tr>';
            }
        }

        // Total Aktiva
        if ($totalAktiva > 0) {
            $html .= '<tr class="total-row">
                <td class="text-left"></td>
                <td class="text-left">TOTAL AKTIVA</td>
                <td class="text-right"></td>
                <td class="text-right">' . number_format($totalAktiva, 0, ',', '.') . '</td>
            </tr>';
        }

        // Ekuitas
        $ekuitas = $accounts->filter(function($acc) {
            return $acc['category'] === 'Ekuitas' && abs($acc['balance']) > 0.01;
        });
        
        if ($ekuitas->count() > 0) {
            foreach ($ekuitas as $account) {
                $html .= '<tr>
                    <td class="text-left">' . htmlspecialchars($account['kd_perk']) . '</td>
                    <td class="text-left">' . htmlspecialchars($account['nama_perkiraan']) . '</td>
                    <td class="text-right">' . number_format($account['balance'], 0, ',', '.') . '</td>
                    <td class="text-right"></td>
                </tr>';
            }
            
            $html .= '<tr class="total-row">
                <td class="text-left"></td>
                <td class="text-left">Jumlah Ekuitas</td>
                <td class="text-right"></td>
                <td class="text-right">' . number_format($ekuitas->sum('balance'), 0, ',', '.') . '</td>
            </tr>';
        }

        // Total Kewajiban dan Ekuitas
        $totalKewajibanEkuitas = $totalKewajiban + $totalEkuitas;
        if ($totalKewajibanEkuitas > 0) {
            $html .= '<tr class="total-row">
                <td class="text-left"></td>
                <td class="text-left">TOTAL KEWAJIBAN DAN EKUITAS</td>
                <td class="text-right"></td>
                <td class="text-right">' . number_format($totalKewajibanEkuitas, 0, ',', '.') . '</td>
            </tr>';
        }

        $html .= '</tbody>
    </table>
</body>
</html>';

        return response($html)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="Neraca_Saldo_' . date('Y-m-d_H-i-s') . '.xls"');
    }

    /**
     * Export Neraca Lajur (Excel)
     */
    public function worksheetExport(Request $request)
    {
        $endDate = $request->get('end_date', date('Y-m-d'));

        // Get all chart of accounts
        $chartAccounts = ChartOfAccount::active()
            ->orderBy('kode_akun')
            ->get()
            ->keyBy('kode_akun');

        // Get all journal entries up to end date
        $entries = JournalEntry::whereDate('tanggal_transaksi', '<=', $endDate)
            ->whereNotNull('kd_perk')
            ->orderBy('kd_perk')
            ->get();

        $accounts = collect();
        $totalTrialBalanceDebit = 0;
        $totalTrialBalanceKredit = 0;
        $totalProfitLossDebit = 0;
        $totalProfitLossKredit = 0;
        $totalBalanceSheetDebit = 0;
        $totalBalanceSheetKredit = 0;
        $netProfitLoss = 0;

        // Group by account code
        if ($entries->count() > 0) {
            $accounts = $entries->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
                $firstEntry = $group->first();
                $kdPerk = $firstEntry->kd_perk ?? '';
                
                // Get account info from Chart of Accounts
                $chartAccount = $chartAccounts->get($kdPerk);
                
                if (!$chartAccount) {
                    // Fallback to entry data if not found in Chart of Accounts
                    $namaPerkiraan = $firstEntry->nama_perkiraan ?? '';
                    $balancePosition = 'DEBIT';
                    if (preg_match('/^4|^[78]|^2|^3/', $kdPerk)) {
                        $balancePosition = 'KREDIT';
                    }
                    $reportPosition = 'NERACA';
                    if (preg_match('/^[4-9]/', $kdPerk)) {
                        $reportPosition = 'LABA RUGI';
                    }
                } else {
                    $namaPerkiraan = $chartAccount->nama_akun;
                    $balancePosition = $chartAccount->pos_saldo;
                    $reportPosition = $chartAccount->pos_laporan;
                }
                
                $totalDebit = $group->sum('debit') ?? 0;
                $totalKredit = $group->sum('kredit') ?? 0;
                
                // Add saldo awal from Chart of Accounts
                $saldoAwalDebet = $chartAccount ? ($chartAccount->saldo_awal_debet ?? 0) : 0;
                $saldoAwalKredit = $chartAccount ? ($chartAccount->saldo_awal_kredit ?? 0) : 0;
                
                $totalDebit += $saldoAwalDebet;
                $totalKredit += $saldoAwalKredit;
                
                // Calculate trial balance
                $trialBalanceDebit = 0;
                $trialBalanceKredit = 0;
                if ($balancePosition === 'DEBIT') {
                    $trialBalanceDebit = max(0, $totalDebit - $totalKredit);
                } else {
                    $trialBalanceKredit = max(0, $totalKredit - $totalDebit);
                }
                
                // For profit/loss accounts, extend to LABA RUGI
                $profitLossDebit = 0;
                $profitLossKredit = 0;
                if ($reportPosition === 'LABA RUGI') {
                    if (preg_match('/^4/', $kdPerk)) {
                        // Revenue - goes to KREDIT
                        $profitLossKredit = $totalKredit;
                    } elseif (preg_match('/^[5-9]/', $kdPerk)) {
                        // Expenses/COGS - goes to DEBIT
                        $profitLossDebit = $totalDebit;
                    }
                }
                
                // For balance sheet accounts, extend to NERACA
                $balanceSheetDebit = 0;
                $balanceSheetKredit = 0;
                if ($reportPosition === 'NERACA') {
                    if ($balancePosition === 'DEBIT') {
                        $balanceSheetDebit = $trialBalanceDebit;
                    } else {
                        $balanceSheetKredit = $trialBalanceKredit;
                    }
                }
                
                return [
                    'kd_perk' => $kdPerk,
                    'nama_perkiraan' => $namaPerkiraan,
                    'balance_position' => $balancePosition,
                    'report_position' => $reportPosition,
                    'trial_balance_debit' => $trialBalanceDebit,
                    'trial_balance_kredit' => $trialBalanceKredit,
                    'profit_loss_debit' => $profitLossDebit,
                    'profit_loss_kredit' => $profitLossKredit,
                    'balance_sheet_debit' => $balanceSheetDebit,
                    'balance_sheet_kredit' => $balanceSheetKredit,
                ];
            })->values();

            // Calculate totals
            $totalTrialBalanceDebit = $accounts->sum('trial_balance_debit');
            $totalTrialBalanceKredit = $accounts->sum('trial_balance_kredit');
            
            $totalProfitLossDebit = $accounts->sum('profit_loss_debit');
            $totalProfitLossKredit = $accounts->sum('profit_loss_kredit');
            
            $totalBalanceSheetDebit = $accounts->sum('balance_sheet_debit');
            $totalBalanceSheetKredit = $accounts->sum('balance_sheet_kredit');
            
            // Calculate net profit/loss
            $netProfitLoss = $totalProfitLossKredit - $totalProfitLossDebit;
        }

        $endDateFormatted = \Carbon\Carbon::parse($endDate)->format('d F Y');
        $printDate = \Carbon\Carbon::now()->format('d F Y');
        $printTime = \Carbon\Carbon::now()->format('H:i');

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table { border-collapse: collapse; width: 100%; font-family: Arial, sans-serif; font-size: 10px; }
        th { background-color: #90EE90; color: #000000; font-weight: bold; text-align: center; padding: 6px 4px; border: 1px solid #000000; }
        td { padding: 4px; border: 1px solid #CCCCCC; }
        .text-right { text-align: right; }
        .text-center { text-align: center; }
        .text-left { text-align: left; }
        .header-info { text-align: center; margin-bottom: 10px; }
        .total-row { background-color: #4472C4; color: #FFFFFF; font-weight: bold; }
    </style>
</head>
<body>
    <div class="header-info">
        <h2>NERACA LAJUR</h2>
        <p>POLJAMTECH</p>
        <p>Jl. Lkr. Barat 3 No.1, Bagan Pete, Kec. Kota Baru, Kota Jambi, Jambi 36361</p>
        <p>Per ' . $endDateFormatted . '</p>
        <p>' . $printDate . ' Jam : ' . $printTime . '</p>
    </div>
    <table>
        <thead>
            <tr>
                <th rowspan="2">KODE AKUN</th>
                <th rowspan="2">NAMA AKUN</th>
                <th rowspan="2">POS SALDO</th>
                <th colspan="2">NERACA SALDO</th>
                <th rowspan="2">POS LAPORAN</th>
                <th colspan="2">LABA RUGI</th>
                <th colspan="2">NERACA</th>
            </tr>
            <tr>
                <th>DEBET</th>
                <th>KREDIT</th>
                <th>DEBET</th>
                <th>KREDIT</th>
                <th>DEBET</th>
                <th>KREDIT</th>
            </tr>
        </thead>
        <tbody>';

        foreach ($accounts as $account) {
            $html .= '<tr>
                <td class="text-center">' . htmlspecialchars($account['kd_perk']) . '</td>
                <td class="text-left">' . htmlspecialchars($account['nama_perkiraan']) . '</td>
                <td class="text-center">' . $account['balance_position'] . '</td>
                <td class="text-right">' . (abs($account['trial_balance_debit']) > 0.01 ? number_format($account['trial_balance_debit'], 0, ',', '.') : '') . '</td>
                <td class="text-right">' . (abs($account['trial_balance_kredit']) > 0.01 ? number_format($account['trial_balance_kredit'], 0, ',', '.') : '') . '</td>
                <td class="text-center">' . $account['report_position'] . '</td>
                <td class="text-right">' . (abs($account['profit_loss_debit']) > 0.01 ? number_format($account['profit_loss_debit'], 0, ',', '.') : '') . '</td>
                <td class="text-right">' . (abs($account['profit_loss_kredit']) > 0.01 ? number_format($account['profit_loss_kredit'], 0, ',', '.') : '') . '</td>
                <td class="text-right">' . (abs($account['balance_sheet_debit']) > 0.01 ? number_format($account['balance_sheet_debit'], 0, ',', '.') : '') . '</td>
                <td class="text-right">' . (abs($account['balance_sheet_kredit']) > 0.01 ? number_format($account['balance_sheet_kredit'], 0, ',', '.') : '') . '</td>
            </tr>';
        }

        // Total row
        $html .= '<tr class="total-row">
            <td colspan="3" class="text-right">JUMLAH</td>
            <td class="text-right">' . number_format($totalTrialBalanceDebit, 0, ',', '.') . '</td>
            <td class="text-right">' . number_format($totalTrialBalanceKredit, 0, ',', '.') . '</td>
            <td></td>
            <td class="text-right">' . number_format($totalProfitLossDebit, 0, ',', '.') . '</td>
            <td class="text-right">' . number_format($totalProfitLossKredit, 0, ',', '.') . '</td>
            <td class="text-right">' . number_format($totalBalanceSheetDebit, 0, ',', '.') . '</td>
            <td class="text-right">' . number_format($totalBalanceSheetKredit, 0, ',', '.') . '</td>
        </tr>';

        // Laba (Rugi) Tahun Berjalan row
        $html .= '<tr>
            <td class="text-center">310-04</td>
            <td class="text-left">LABA (RUGI) TAHUN BERJALAN</td>
            <td class="text-center">KREDIT</td>
            <td class="text-right"></td>
            <td class="text-right"></td>
            <td class="text-center">NERACA</td>
            <td class="text-right"></td>
            <td class="text-right">' . ($netProfitLoss > 0 ? number_format($netProfitLoss, 0, ',', '.') : '') . '</td>
            <td class="text-right">' . ($netProfitLoss > 0 ? number_format($netProfitLoss, 0, ',', '.') : '') . '</td>
            <td class="text-right"></td>
        </tr>';

        // Neraca Saldo Setelah Laba Rugi
        $finalBalanceSheetDebit = $totalBalanceSheetDebit + $netProfitLoss;
        $finalBalanceSheetKredit = $totalBalanceSheetKredit + $netProfitLoss;

        $html .= '<tr class="total-row">
            <td colspan="3" class="text-right">Neraca Saldo Setelah Laba Rugi</td>
            <td class="text-right"></td>
            <td class="text-right"></td>
            <td></td>
            <td class="text-right"></td>
            <td class="text-right"></td>
            <td class="text-right">' . number_format($finalBalanceSheetDebit, 0, ',', '.') . '</td>
            <td class="text-right">' . number_format($finalBalanceSheetKredit, 0, ',', '.') . '</td>
        </tr>';

        $html .= '</tbody>
    </table>
</body>
</html>';

        return response($html)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="Neraca_Lajur_' . date('Y-m-d_H-i-s') . '.xls"');
    }

    /**
     * Export Laporan Perubahan Modal (Excel)
     */
    public function equityChangesExport(Request $request)
    {
        $endDate = $request->get('end_date', date('Y-m-d'));
        $startDate = $request->get('start_date', date('Y-01-01'));

        // Get all chart of accounts
        $chartAccounts = ChartOfAccount::active()
            ->orderBy('kode_akun')
            ->get()
            ->keyBy('kode_akun');

        // Get equity accounts (starting with 3) - use Chart of Accounts to filter
        $equityAccountCodes = $chartAccounts->filter(function($account) {
            return preg_match('/^3/', $account->kode_akun) && $account->pos_laporan === 'NERACA';
        })->pluck('kode_akun')->toArray();

        $equityEntries = JournalEntry::whereIn('kd_perk', $equityAccountCodes)
            ->whereDate('tanggal_transaksi', '<=', $endDate)
            ->get();

        // Get beginning equity (before start date) - include saldo awal from Chart of Accounts
        $beginningEquity = 0;
        
        // Add saldo awal from Chart of Accounts for equity accounts
        foreach ($chartAccounts as $account) {
            if (preg_match('/^3/', $account->kode_akun) && $account->pos_laporan === 'NERACA') {
                $beginningEquity += ($account->saldo_awal_kredit ?? 0) - ($account->saldo_awal_debet ?? 0);
            }
        }
        
        $beginningEntries = JournalEntry::whereIn('kd_perk', $equityAccountCodes)
            ->whereDate('tanggal_transaksi', '<', $startDate)
            ->get();
        
        foreach ($beginningEntries as $entry) {
            $debit = $entry->debit ?? 0;
            $kredit = $entry->kredit ?? 0;
            $beginningEquity += ($kredit - $debit);
        }

        // Get equity changes during period
        $periodEntries = JournalEntry::whereIn('kd_perk', $equityAccountCodes)
            ->whereDate('tanggal_transaksi', '>=', $startDate)
            ->whereDate('tanggal_transaksi', '<=', $endDate)
            ->get();

        // Group by account - use Chart of Accounts for nama_akun
        $equityAccounts = $periodEntries->groupBy('kd_perk')->map(function($group) use ($chartAccounts) {
            $firstEntry = $group->first();
            $kdPerk = $firstEntry->kd_perk ?? '';
            $chartAccount = $chartAccounts->get($kdPerk);
            $namaPerkiraan = $chartAccount ? $chartAccount->nama_akun : ($firstEntry->nama_perkiraan ?? '');
            
            $totalDebit = $group->sum('debit') ?? 0;
            $totalKredit = $group->sum('kredit') ?? 0;
            
            // Equity increases with KREDIT, decreases with DEBIT
            $change = $totalKredit - $totalDebit;
            
            return [
                'kd_perk' => $kdPerk,
                'nama_perkiraan' => $namaPerkiraan,
                'debit' => $totalDebit,
                'kredit' => $totalKredit,
                'change' => $change,
            ];
        })->values();

        // Get current year profit/loss (310-04)
        $currentYearProfitLoss = 0;
        $profitLossEntries = JournalEntry::where('kd_perk', '310-04')
            ->whereDate('tanggal_transaksi', '>=', $startDate)
            ->whereDate('tanggal_transaksi', '<=', $endDate)
            ->get();
        
        foreach ($profitLossEntries as $entry) {
            $debit = $entry->debit ?? 0;
            $kredit = $entry->kredit ?? 0;
            $currentYearProfitLoss += ($kredit - $debit);
        }

        // If no direct entry in 310-04, calculate from profit/loss accounts
        if (abs($currentYearProfitLoss) < 0.01) {
            // Get all chart of accounts
            $profitLossChartAccounts = ChartOfAccount::active()
                ->orderBy('kode_akun')
                ->get()
                ->keyBy('kode_akun');
            
            $profitLossAccounts = JournalEntry::whereDate('tanggal_transaksi', '>=', $startDate)
                ->whereDate('tanggal_transaksi', '<=', $endDate)
                ->get();
            
            // Revenue (4) - accounts with pos_laporan = 'LABA RUGI'
            $revenue = $profitLossAccounts->filter(function($entry) use ($profitLossChartAccounts) {
                $chartAccount = $profitLossChartAccounts->get($entry->kd_perk ?? '');
                return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^4/', $entry->kd_perk ?? '');
            })->sum('kredit');
            
            // COGS (5) - accounts with pos_laporan = 'LABA RUGI'
            $cogs = $profitLossAccounts->filter(function($entry) use ($profitLossChartAccounts) {
                $chartAccount = $profitLossChartAccounts->get($entry->kd_perk ?? '');
                return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^5/', $entry->kd_perk ?? '');
            })->sum('debit');
            
            // Expenses (6) - accounts with pos_laporan = 'LABA RUGI'
            $expenses = $profitLossAccounts->filter(function($entry) use ($profitLossChartAccounts) {
                $chartAccount = $profitLossChartAccounts->get($entry->kd_perk ?? '');
                return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^6/', $entry->kd_perk ?? '');
            })->sum('debit');
            
            // Other income/expenses (7-8) - accounts with pos_laporan = 'LABA RUGI'
            $otherIncome = $profitLossAccounts->filter(function($entry) use ($profitLossChartAccounts) {
                $chartAccount = $profitLossChartAccounts->get($entry->kd_perk ?? '');
                return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^[78]/', $entry->kd_perk ?? '');
            })->sum(function($entry) {
                return ($entry->kredit ?? 0) - ($entry->debit ?? 0);
            });
            
            // Tax (9) - accounts with pos_laporan = 'LABA RUGI'
            $tax = $profitLossAccounts->filter(function($entry) use ($profitLossChartAccounts) {
                $chartAccount = $profitLossChartAccounts->get($entry->kd_perk ?? '');
                return $chartAccount && $chartAccount->pos_laporan === 'LABA RUGI' && preg_match('/^9/', $entry->kd_perk ?? '');
            })->sum('debit');
            
            $currentYearProfitLoss = $revenue - $cogs - $expenses + $otherIncome - $tax;
        }

        // Calculate total equity additions
        $totalAdditions = $equityAccounts->sum('change') + $currentYearProfitLoss;

        // Calculate ending equity
        $endingEquity = $beginningEquity + $totalAdditions;

        $endDateFormatted = \Carbon\Carbon::parse($endDate)->format('d F Y');

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        table { border-collapse: collapse; width: 100%; font-family: Arial, sans-serif; font-size: 11px; }
        th { background-color: #90EE90; color: #000000; font-weight: bold; text-align: center; padding: 8px 6px; border: 1px solid #000000; }
        td { padding: 6px; border: 1px solid #CCCCCC; }
        .text-right { text-align: right; }
        .text-left { text-align: left; }
        .header-info { text-align: center; margin-bottom: 10px; }
        .total-row { font-weight: bold; }
        .indent { padding-left: 20px; }
    </style>
</head>
<body>
    <div class="header-info">
        <h2>POLJAMTECH</h2>
        <h3>LAPORAN PEROBAHAN MODAL</h3>
        <p>Per ' . $endDateFormatted . '</p>
    </div>
    <table>
        <thead>
            <tr>
                <th class="text-left">Keterangan</th>
                <th class="text-right">Jumlah</th>
            </tr>
        </thead>
        <tbody>
            <tr>
                <td class="text-left">Ekuitas Awal Pemegang Saham periode</td>
                <td class="text-right">' . ($beginningEquity != 0 ? number_format($beginningEquity, 0, ',', '.') : '-') . '</td>
            </tr>
            <tr>
                <td class="text-left">Penambahan Ekuitas Pemegang Saham :</td>
                <td class="text-right"></td>
            </tr>';

        // MODAL (310-01)
        $modalAccount = $equityAccounts->firstWhere('kd_perk', '310-01');
        $modalChange = $modalAccount ? $modalAccount['change'] : 0;
        $html .= '<tr>
            <td class="text-left indent">310-01 MODAL</td>
            <td class="text-right">' . ($modalChange != 0 ? number_format($modalChange, 0, ',', '.') : '-') . '</td>
        </tr>';

        // DEVIDEN (310-03)
        $devidenAccount = $equityAccounts->firstWhere('kd_perk', '310-03');
        $devidenChange = $devidenAccount ? $devidenAccount['change'] : 0;
        $html .= '<tr>
            <td class="text-left indent">310-03 DEVIDEN</td>
            <td class="text-right">' . ($devidenChange != 0 ? number_format($devidenChange, 0, ',', '.') : '-') . '</td>
        </tr>';

        // LABA (RUGI) TAHUN BERJALAN (310-04)
        $html .= '<tr>
            <td class="text-left indent">310-04 LABA (RUGI) TAHUN BERJALAN</td>
            <td class="text-right">' . ($currentYearProfitLoss != 0 ? number_format($currentYearProfitLoss, 0, ',', '.') : '-') . '</td>
        </tr>';

        $html .= '<tr class="total-row">
            <td class="text-left">Total Penambahan Ekuitas Pemegang Saham</td>
            <td class="text-right">' . ($totalAdditions != 0 ? number_format($totalAdditions, 0, ',', '.') : '-') . '</td>
        </tr>
        <tr class="total-row">
            <td class="text-left">Ekuitas Pemegang Saham Akhir Periode</td>
            <td class="text-right">' . ($endingEquity != 0 ? number_format($endingEquity, 0, ',', '.') : '-') . '</td>
        </tr>
        </tbody>
    </table>
</body>
</html>';

        return response($html)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="Laporan_Perubahan_Modal_' . date('Y-m-d_H-i-s') . '.xls"');
    }
}
