<?php

namespace App\Http\Controllers;

use App\Models\Material;
use App\Models\Sparepart;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Pagination\Paginator;

class MaterialController extends Controller
{
    public function index(Request $request)
    {
        // Query untuk materials
        $materialsQuery = Material::query();
        
        // Filter type - hanya ambil materials jika type tidak diisi atau type = 'material'
        $shouldIncludeMaterials = !$request->filled('type') || $request->type === 'material';
        
        if ($shouldIncludeMaterials) {
            // Search filter untuk materials dengan grouping yang benar
            if ($request->filled('search') && trim($request->search) !== '') {
                $search = trim($request->search);
                $materialsQuery->where(function($q) use ($search) {
                    $q->where('name', 'like', '%' . $search . '%')
                        ->orWhere('description', 'like', '%' . $search . '%')
                        ->orWhere('category', 'like', '%' . $search . '%')
                        ->orWhere('supplier', 'like', '%' . $search . '%')
                        ->orWhere('material_code', 'like', '%' . $search . '%');
                });
            }
            
            // Status filter untuk materials
            if ($request->filled('status') && in_array($request->status, ['1', '0'])) {
                $materialsQuery->where('is_active', (bool)$request->status);
            }
        }

        // Query untuk spareparts
        $sparepartsQuery = Sparepart::query();
        
        // Filter type - hanya ambil spareparts jika type tidak diisi atau type = 'sparepart'
        $shouldIncludeSpareparts = !$request->filled('type') || $request->type === 'sparepart';
        
        if ($shouldIncludeSpareparts) {
            // Search filter untuk spareparts dengan grouping yang benar
            if ($request->filled('search') && trim($request->search) !== '') {
                $search = trim($request->search);
                $sparepartsQuery->where(function($q) use ($search) {
                    $q->where('name', 'like', '%' . $search . '%')
                        ->orWhere('unit', 'like', '%' . $search . '%')
                        ->orWhere('brand', 'like', '%' . $search . '%')
                        ->orWhere('type', 'like', '%' . $search . '%')
                        ->orWhere('part_code', 'like', '%' . $search . '%')
                        ->orWhere('part_number', 'like', '%' . $search . '%');
                });
            }
            
            // Status filter untuk spareparts
            if ($request->filled('status') && in_array($request->status, ['1', '0'])) {
                $sparepartsQuery->where('is_active', (bool)$request->status);
            }
        }

        // Ambil data berdasarkan filter
        $materialsAll = $shouldIncludeMaterials ? $materialsQuery->latest()->get() : collect();
        $sparepartsAll = $shouldIncludeSpareparts ? $sparepartsQuery->latest()->get() : collect();

            $materialsMapped = $materialsAll->map(function ($m) {
                return [
                    'id' => $m->id,
                    'type' => 'material',
                    'code' => $m->material_code,
                    'name' => $m->name,
                    'brand' => '-',
                    'part_number' => '-',
                    'unit' => $m->unit,
                    'stock' => $m->stock,
                    'damaged_stock' => $m->damaged_stock ?? 0,
                    'price' => $m->price,
                    'is_active' => (bool) $m->is_active,
                    'image_path' => $m->image ? asset('uploads/materials/' . $m->image) : null,
                    'created_at' => $m->created_at,
                ];
            });

            $sparepartsMapped = $sparepartsAll->map(function ($s) {
                return [
                    'id' => $s->id,
                    'type' => 'sparepart',
                    'code' => $s->part_code,
                    'name' => $s->name,
                    'brand' => $s->brand ?? '-',
                    'part_number' => $s->part_number ?? '-',
                    'unit' => $s->unit,
                    'stock' => $s->stock,
                    'damaged_stock' => $s->damaged_stock ?? 0,
                    'price' => $s->price,
                    'is_active' => (bool) $s->is_active,
                    'image_path' => $s->image ? asset('uploads/spareparts/' . $s->image) : null,
                    'created_at' => $s->created_at,
                ];
            });

        // Gabungkan dan urutkan berdasarkan created_at
        $combinedItems = $materialsMapped->merge($sparepartsMapped)->sortByDesc('created_at')->values();
        
        // Filter type dan status sudah diterapkan di query database, jadi tidak perlu filter lagi di collection

        // Simpan total untuk statistik sebelum pagination
        $totalItems = $combinedItems->count();
        $activeItems = $combinedItems->where('is_active', true)->count();
        $lowStockItems = $combinedItems->where('stock', '<=', 5)->count();
        $inactiveItems = $combinedItems->where('is_active', false)->count();

        // Manual pagination untuk collection
        $currentPage = Paginator::resolveCurrentPage() ?: 1;
        $perPage = (int) Setting::get('items_per_page', 10);
        $items = $combinedItems->slice(($currentPage - 1) * $perPage, $perPage)->values();
        $total = $totalItems;

        $combinedItems = new LengthAwarePaginator(
            $items,
            $total,
            $perPage,
            $currentPage,
            [
                'path' => Paginator::resolveCurrentPath(),
                'pageName' => 'page',
            ]
        );

        // Append query string untuk search, type, dan status
        $combinedItems->appends($request->only(['search', 'type', 'status']));

        return view('admin.master.materials.index', compact('combinedItems', 'totalItems', 'activeItems', 'lowStockItems', 'inactiveItems'));
    }

    public function create(Request $request)
    {
        $type = $request->get('type', 'material'); // Default material
        return view('admin.master.materials.create', compact('type'));
    }

    public function store(Request $request)
    {
        $type = $request->input('type', 'material');
        
        if ($type === 'sparepart') {
            $request->validate([
                'spareparts.*.part_code' => 'nullable|string|max:50|distinct',
                'spareparts.*.name' => 'required|string|max:255',
                'spareparts.*.brand' => 'nullable|string|max:100',
                'spareparts.*.type' => 'nullable|string|max:100',
                'spareparts.*.part_number' => 'nullable|string|max:100',
                'spareparts.*.stock' => 'required|integer|min:0',
                'spareparts.*.unit' => 'nullable|string|max:50',
                'spareparts.*.price' => 'nullable|numeric|min:0',
                'spareparts.*.description' => 'nullable|string',
                'spareparts.*.is_active' => 'nullable|boolean',
                'spareparts.*.image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            ]);

            foreach ($request->spareparts as $index => $sparepartInput) {
                $data = $sparepartInput;
                $data['is_active'] = $request->has("spareparts.$index.is_active") ? 1 : 0;

                if ($request->hasFile("spareparts.$index.image")) {
                    $file = $request->file("spareparts.$index.image");
                    if ($file->isValid()) {
                        $filename = Str::uuid() . '.' . $file->getClientOriginalExtension();
                        $file->storeAs('spareparts', $filename, 'uploads');
                        $data['image'] = $filename;
                    }
                }

                Sparepart::create($data);
            }

            return redirect()->route('admin.master.materials.index')
                ->with('add_message', 'Semua sparepart berhasil ditambahkan!');
        } else {
            $request->validate([
                'materials.*.material_code' => 'nullable|string|max:50|distinct',
                'materials.*.name' => 'required|string|max:255',
                'materials.*.stock' => 'required|integer|min:0',
                'materials.*.unit' => 'nullable|string|max:50',
                'materials.*.price' => 'nullable|numeric|min:0',
                'materials.*.category' => 'nullable|string|max:100',
                'materials.*.supplier' => 'nullable|string|max:100',
                'materials.*.description' => 'nullable|string',
                'materials.*.is_active' => 'nullable|boolean',
                'materials.*.image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            ]);

            foreach ($request->materials as $index => $materialInput) {
                $data = $materialInput;
                $data['is_active'] = isset($materialInput['is_active']) && $materialInput['is_active'] ? 1 : 0;
                $data['material_code'] = trim($materialInput['material_code'] ?? '') !== '' ? $materialInput['material_code'] : null;
                $data['supplier'] = trim($materialInput['supplier'] ?? '') !== '' ? $materialInput['supplier'] : null;
                $data['category'] = trim($materialInput['category'] ?? '') !== '' ? $materialInput['category'] : null;

                if ($request->hasFile("materials.$index.image")) {
                    $file = $request->file("materials.$index.image");
                    if ($file->isValid()) {
                        $filename = Str::uuid() . '.' . $file->getClientOriginalExtension();
                        $file->storeAs('materials', $filename, 'uploads');
                        $data['image'] = $filename;
                    }
                }

                Material::create($data);
            }

            return redirect()->route('admin.master.materials.index')
                ->with('add_message', 'Semua material berhasil ditambahkan!');
        }
    }

    public function edit(Request $request, $id)
    {
        $type = $request->get('type', 'material');
        $queryParams = array_filter($request->only(['page', 'search', 'type']), function($value) {
            return $value !== null && $value !== '';
        });

        if ($type === 'sparepart') {
            $sparepart = Sparepart::findOrFail($id);
            return view('admin.master.materials.edit', compact('sparepart', 'type', 'queryParams'));
        } else {
            $material = Material::findOrFail($id);
            return view('admin.master.materials.edit', compact('material', 'type', 'queryParams'));
        }
    }

    public function update(Request $request, $id)
    {
        $type = $request->input('type', 'material');
        $queryParams = array_filter($request->only(['page', 'search', 'type']), function($value) {
            return $value !== null && $value !== '';
        });

        if ($type === 'sparepart') {
            $sparepart = Sparepart::findOrFail($id);
            
            $request->validate([
                'part_code' => 'nullable|string|unique:spareparts,part_code,' . $sparepart->id . '|max:50',
                'name' => 'required|string|max:255',
                'brand' => 'nullable|string|max:100',
                'type' => 'nullable|string|max:100',
                'part_number' => 'nullable|string|max:100',
                'stock' => 'required|integer|min:0',
                'damaged_stock' => 'nullable|integer|min:0|max:' . $sparepart->stock,
                'unit' => 'nullable|string|max:50',
                'price' => 'nullable|numeric|min:0',
                'description' => 'nullable|string',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
                'damage_notes' => 'nullable|string',
                'damage_date' => 'nullable|date',
            ]);

            $data = $request->only([
                'part_code', 'name', 'brand', 'type', 'part_number',
                'stock', 'damaged_stock', 'unit', 'price', 'description',
                'damage_notes', 'damage_date'
            ]);
            $data['is_active'] = $request->has('is_active') ? 1 : 0;

            if ($request->has('remove_image') && $request->remove_image == '1') {
                if ($sparepart->image && Storage::disk('uploads')->exists('spareparts/' . $sparepart->image)) {
                    Storage::disk('uploads')->delete('spareparts/' . $sparepart->image);
                }
                $data['image'] = null;
            } elseif ($request->hasFile('image')) {
                if ($sparepart->image && Storage::disk('uploads')->exists('spareparts/' . $sparepart->image)) {
                    Storage::disk('uploads')->delete('spareparts/' . $sparepart->image);
                }
                $file = $request->file('image');
                $filename = Str::uuid() . '.' . $file->getClientOriginalExtension();
                $file->storeAs('spareparts', $filename, 'uploads');
                $data['image'] = $filename;
            }

            $sparepart->update($data);

            return redirect()->route('admin.master.materials.index', $queryParams)
                ->with('add_message', 'Data sparepart berhasil diperbarui!');
        } else {
            $material = Material::findOrFail($id);
            
            $request->validate([
                'material_code' => 'nullable|string|unique:materials,material_code,' . $material->id . '|max:50',
                'name' => 'required|string|max:255',
                'stock' => 'required|integer|min:0',
                'damaged_stock' => 'nullable|integer|min:0|max:' . $material->stock,
                'unit' => 'nullable|string|max:50',
                'price' => 'nullable|numeric|min:0',
                'category' => 'nullable|string|max:100',
                'supplier' => 'nullable|string|max:100',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
                'description' => 'nullable|string',
                'is_active' => 'nullable|boolean',
                'damage_notes' => 'nullable|string',
                'damage_date' => 'nullable|date',
            ]);

            $data = $request->only([
                'material_code', 'name', 'stock', 'damaged_stock', 'unit', 'price',
                'category', 'supplier', 'description', 'damage_notes', 'damage_date'
            ]);
            $data['is_active'] = $request->has('is_active') ? 1 : 0;

            if ($request->hasFile('image')) {
                if ($material->image && Storage::disk('uploads')->exists('materials/' . $material->image)) {
                    Storage::disk('uploads')->delete('materials/' . $material->image);
                }
                $file = $request->file('image');
                $filename = Str::uuid() . '.' . $file->getClientOriginalExtension();
                $file->storeAs('materials', $filename, 'uploads');
                $data['image'] = $filename;
            }

            $material->update($data);

            return redirect()->route('admin.master.materials.index', $queryParams)
                ->with('add_message', 'Data bahan berhasil diperbarui!');
        }
    }

    public function show(Request $request, $id)
    {
        $type = $request->get('type', 'material');
        
        if ($type === 'sparepart') {
            $sparepart = Sparepart::findOrFail($id);
            return view('admin.master.materials.show', compact('sparepart', 'type'));
        } else {
            $material = Material::findOrFail($id);
            return view('admin.master.materials.show', compact('material', 'type'));
        }
    }

    public function destroy(Request $request, $id)
    {
        $type = $request->get('type', 'material');
        
        if ($type === 'sparepart') {
            $sparepart = Sparepart::findOrFail($id);
            if ($sparepart->image && Storage::disk('uploads')->exists('spareparts/' . $sparepart->image)) {
                Storage::disk('uploads')->delete('spareparts/' . $sparepart->image);
            }
            $sparepart->delete();
            return redirect()->route('admin.master.materials.index')
                ->with('add_message', 'Sparepart berhasil dihapus!');
        } else {
            $material = Material::findOrFail($id);
            if ($material->image && Storage::disk('uploads')->exists('materials/' . $material->image)) {
                Storage::disk('uploads')->delete('materials/' . $material->image);
            }
            $material->delete();
            return redirect()->route('admin.master.materials.index')
                ->with('add_message', 'Bahan berhasil dihapus!');
        }
    }

    /**
     * Hapus banyak bahan dan sparepart sekaligus
     */
    public function bulkDestroy(Request $request)
    {
        // Handle JSON string from form
        $items = $request->items;
        if (is_string($items)) {
            $items = json_decode($items, true);
        }

        $request->merge(['items' => $items]);

        $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|integer',
            'items.*.type' => 'required|in:material,sparepart',
        ]);

        DB::beginTransaction();
        try {
            $deletedCount = 0;
            $materialsDeleted = 0;
            $sparepartsDeleted = 0;

            foreach ($request->items as $item) {
                if ($item['type'] === 'material') {
                    $material = Material::find($item['id']);
                    if ($material) {
                        // Hapus gambar jika ada
                        if ($material->image && Storage::disk('uploads')->exists('materials/' . $material->image)) {
                            Storage::disk('uploads')->delete('materials/' . $material->image);
                        }
                        $material->delete();
                        $materialsDeleted++;
                        $deletedCount++;
                    }
                } elseif ($item['type'] === 'sparepart') {
                    $sparepart = Sparepart::find($item['id']);
                    if ($sparepart) {
                        // Hapus gambar jika ada
                        if ($sparepart->image && Storage::disk('uploads')->exists('spareparts/' . $sparepart->image)) {
                            Storage::disk('uploads')->delete('spareparts/' . $sparepart->image);
                        }
                        $sparepart->delete();
                        $sparepartsDeleted++;
                        $deletedCount++;
                    }
                }
            }

            DB::commit();

            $message = '';
            if ($materialsDeleted > 0 && $sparepartsDeleted > 0) {
                $message = "{$materialsDeleted} bahan dan {$sparepartsDeleted} sparepart berhasil dihapus!";
            } elseif ($materialsDeleted > 0) {
                $message = "{$materialsDeleted} bahan berhasil dihapus!";
            } elseif ($sparepartsDeleted > 0) {
                $message = "{$sparepartsDeleted} sparepart berhasil dihapus!";
            } else {
                $message = "{$deletedCount} item berhasil dihapus!";
            }

            return redirect()->route('admin.master.materials.index')
                ->with('add_message', $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['msg' => 'Gagal menghapus item: ' . $e->getMessage()]);
        }
    }
}
