<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Product;
use App\Models\Category;
use App\Models\Setting;
use Illuminate\Support\Facades\Storage;
use Barryvdh\DomPDF\Facade\Pdf;

class ProductController extends Controller
{
    /**
     * Helper method untuk menyimpan gambar produk
     */
    private function saveProductImage($file, $product, $isMain = false)
    {
        $filename = Str::uuid() . '.' . $file->getClientOriginalExtension();
        $file->storeAs('products', $filename, 'uploads');

        if ($isMain) {
            $product->update(['product_image' => $filename]);
        } else {
            $product->images()->create(['image_path' => $filename]);
        }

        return $filename;
    }

    /**
     * Helper method untuk menghapus gambar produk
     */
    private function deleteProductImage($imagePath)
    {
        if ($imagePath && Storage::disk('uploads')->exists('products/' . $imagePath)) {
            Storage::disk('uploads')->delete('products/' . $imagePath);
        }
    }

    /**
     * Helper method untuk menyimpan relasi materials/spareparts (sekarang input bebas)
     */
    private function saveProductRelations($product, $request, $type)
    {
        $relationName = $type === 'material' ? 'materials' : 'spareparts';
        $requestKey = $type === 'material' ? 'product_materials' : 'product_spareparts';

        if ($request->filled($requestKey)) {
            foreach ($request->$requestKey as $data) {
                if (!empty($data['name']) && !empty($data['quantity'])) {
                    // Insert langsung ke tabel pivot dengan data bebas
                    DB::table($type === 'material' ? 'product_material' : 'product_sparepart')->insert([
                        'product_id' => $product->id,
                        'name' => $data['name'],
                        'type' => $data['type'] ?? ($type === 'material' ? 'material' : 'sparepart'),
                        'quantity' => $data['quantity'],
                        'unit' => $data['unit'] ?? 'pcs',
                        'usage_cost' => null, // Tidak ada biaya karena input bebas
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }
        }
    }
    /**
     * Tampilkan semua produk (halaman admin)
     */
    public function index(Request $request)
    {
        // Ambil parameter dengan filter untuk menghindari duplikasi
        $search = $request->input('search');
        $categoryFilter = $request->input('category');
        $statusFilter = $request->input('status');

        $query = Product::with(['category', 'specifications', 'images', 'materials', 'spareparts', 'productions']);

        // Search filter - perbaiki dengan grouping yang benar
        if ($search && trim($search) !== '') {
            $query->where(function ($q) use ($search) {
                $q->where('product_title', 'like', "%{$search}%")
                    ->orWhereHas('category', function ($q2) use ($search) {
                        $q2->where('name', 'like', "%{$search}%");
                    });
            });
        }

        // Category filter - jika kategori besar dipilih, filter semua produk dengan sub kategori di bawahnya
        // Juga include produk yang langsung menggunakan kategori besar tersebut
        if ($categoryFilter && $categoryFilter !== '' && $categoryFilter !== '0') {
            // Pastikan categoryFilter adalah integer
            $categoryFilter = (int) $categoryFilter;

            $selectedCategory = Category::where('type', 'service')->find($categoryFilter);
            if ($selectedCategory) {
                // Jika kategori besar (parent), ambil semua sub kategori dan filter produk
                if ($selectedCategory->isParent()) {
                    $childCategoryIds = $selectedCategory->children()
                        ->where('type', 'service')
                        ->pluck('id')
                        ->toArray();

                    // Gabungkan parent category ID dengan child category IDs
                    // Ini memastikan produk yang langsung menggunakan parent category juga muncul
                    $allCategoryIds = array_merge([$categoryFilter], $childCategoryIds);

                    if (!empty($allCategoryIds)) {
                        // Filter produk yang memiliki category_id di dalam array (parent + child category IDs)
                        $query->whereIn('category_id', $allCategoryIds);
                    } else {
                        // Jika tidak ada kategori (tidak mungkin, tapi untuk safety)
                        $query->whereRaw('1 = 0');
                    }
                } else {
                    // Jika sub kategori, filter langsung berdasarkan category_id
                    // Pastikan hanya produk dengan category_id yang sesuai
                    $query->where('category_id', $categoryFilter);
                }
            } else {
                // Jika kategori tidak ditemukan, tidak ada produk
                $query->whereRaw('1 = 0');
            }
        }

        // Status filter
        if ($statusFilter !== null && $statusFilter !== '') {
            // Konversi string '1' atau '0' ke boolean
            // '1' = true (aktif), '0' = false (nonaktif)
            $query->where('is_active', (bool)$statusFilter);
        }

        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $products = $query->orderBy('id', 'desc')->paginate($itemsPerPage)->withQueryString();

        // Get semua kategori (parent dan child) untuk filter dropdown
        $parentCategories = Category::where('type', 'service')
            ->whereNull('parent_id')
            ->orderBy('name')
            ->get();

        $childCategories = Category::where('type', 'service')
            ->whereNotNull('parent_id')
            ->with('parent')
            ->orderBy('name')
            ->get();

        return view('admin.master.products.index', compact('products', 'parentCategories', 'childCategories'));
    }

    /**
     * Form tambah produk
     */
    public function create()
    {
        $categories = Category::where('type', 'service')
            ->whereNotNull('parent_id')
            ->with('parent')
            ->orderBy('name')
            ->get();
        $materials = \App\Models\Material::where('is_active', true)->orderBy('name')->get();
        $spareparts = \App\Models\Sparepart::where('is_active', true)->orderBy('name')->get();
        return view('admin.master.products.create', compact('categories', 'materials', 'spareparts'));
    }

    /**
     * Simpan produk baru
     */
    public function store(Request $request)
    {
        $request->validate([
            'product_title'         => 'required|string|max:255',
            'product_image'         => 'nullable|image|mimes:jpeg,jpg,png,gif,webp,bmp,svg|max:2048',
            'product_images.*'      => 'nullable|image|mimes:jpeg,jpg,png,gif,webp,bmp,svg|max:2048',
            'product_description'   => 'nullable|string',
            'product_prices'        => 'required|integer|min:1',
            'category_id'           => 'required|exists:categories,id',
            'product_work_duration' => 'nullable|string|max:255',
            'product_garansi'       => 'nullable|string|max:255',
            'service_fee'           => 'nullable|integer|min:0',

            // Materials dan Spareparts (input bebas)
            'product_materials' => 'sometimes|array',
            'product_materials.*.name' => 'required_with:product_materials|string|max:255',
            'product_materials.*.type' => 'required_with:product_materials|in:material,sparepart',
            'product_materials.*.quantity' => 'required_with:product_materials|integer|min:1',
            'product_materials.*.unit' => 'required_with:product_materials|string|max:50',
            'product_spareparts' => 'sometimes|array',
            'product_spareparts.*.name' => 'required_with:product_spareparts|string|max:255',
            'product_spareparts.*.type' => 'required_with:product_spareparts|in:material,sparepart',
            'product_spareparts.*.quantity' => 'required_with:product_spareparts|integer|min:1',
            'product_spareparts.*.unit' => 'required_with:product_spareparts|string|max:50',

            // Specifications (opsional)
            'product_specifications' => 'sometimes|array',
            'product_specifications.*.key' => 'nullable|string|max:255',
            'product_specifications.*.value' => 'nullable|string|max:255',
        ]);

        DB::beginTransaction();

        try {
            // Simpan produk utama
            $product = Product::create([
                'product_title'         => $request->product_title,
                'product_description'   => $request->product_description,
                'product_prices'        => $request->product_prices ?? 0,
                'category_id'           => $request->category_id,
                'product_work_duration' => $request->product_work_duration,
                'product_garansi'       => $request->product_garansi,
                'service_fee'           => $request->service_fee ?? 0,
                'is_active'             => true, // Default aktif
            ]);

            // Simpan gambar utama (jika ada)
            if ($request->hasFile('product_image')) {
                $this->saveProductImage($request->file('product_image'), $product, true);
            }

            // Simpan gambar tambahan (boleh kosong)
            if ($request->hasFile('product_images')) {
                foreach ($request->file('product_images') as $file) {
                    if ($file->isValid()) {
                        $this->saveProductImage($file, $product, false);
                    }
                }
            }

            // Simpan materials dan spareparts
            $this->saveProductRelations($product, $request, 'material');
            $this->saveProductRelations($product, $request, 'sparepart');

            // Simpan spesifikasi produk (opsional)
            if ($request->filled('product_specifications')) {
                foreach ($request->product_specifications as $spec) {
                    if (!empty($spec['key']) && !empty($spec['value'])) {
                        $product->specifications()->create([
                            'key'   => $spec['key'],
                            'value' => $spec['value'],
                        ]);
                    }
                }
            }

            DB::commit();

            return redirect()->route('admin.master.products.index')
                ->with('add_message', 'Produk berhasil ditambahkan!');
        } catch (\Exception $e) {
            DB::rollBack();

            // Log error untuk debugging
            \Log::error('Product creation failed: ' . $e->getMessage(), [
                'request_data' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()->withErrors([
                'msg' => 'Gagal menyimpan produk: ' . $e->getMessage()
            ])->withInput();
        }
    }

    /**
     * Form edit produk
     */
    public function edit(Request $request, $id)
    {
        $product = Product::with(['category', 'specifications', 'images', 'materials', 'spareparts', 'productions.order'])
            ->findOrFail($id);

        // Ambil production yang memiliki labor_cost berbeda dari service_fee
        $mismatchedProductions = \App\Models\Production::where('product_id', $id)
            ->whereNotNull('labor_cost')
            ->where('labor_cost', '!=', $product->service_fee ?? 0)
            ->with(['order', 'teknisi'])
            ->orderBy('created_at', 'desc')
            ->get();

        $categories = Category::where('type', 'service')
            ->whereNotNull('parent_id')
            ->with('parent')
            ->orderBy('name')
            ->get();
        $materials = \App\Models\Material::where('is_active', true)->orderBy('name')->get();
        $spareparts = \App\Models\Sparepart::where('is_active', true)->orderBy('name')->get();

        // Simpan query string untuk redirect kembali ke halaman yang sama (filter nilai kosong)
        $queryParams = array_filter($request->only(['page', 'search', 'category', 'status']), function ($value) {
            return $value !== null && $value !== '';
        });

        return view('admin.master.products.edit', compact('product', 'categories', 'materials', 'spareparts', 'queryParams', 'mismatchedProductions'));
    }

    /**
     * Update produk
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'product_title' => 'required|string|max:255',
            'product_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp,bmp,svg|max:2048',
            'product_images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp,bmp,svg|max:2048',
            'product_description' => 'nullable|string',
            'product_prices' => 'required|integer|min:1',
            'category_id' => 'required|exists:categories,id',
            'product_work_duration' => 'nullable|string|max:255',
            'product_garansi' => 'nullable|string|max:255',
            'service_fee' => 'nullable|integer|min:0',

            // Validasi untuk hapus gambar
            'remove_main_image' => 'nullable|in:0,1',
            'remove_images' => 'nullable|string',

            // Materials dan Spareparts (input bebas)
            'product_materials' => 'sometimes|array',
            'product_materials.*.name' => 'required_with:product_materials|string|max:255',
            'product_materials.*.type' => 'required_with:product_materials|in:material,sparepart',
            'product_materials.*.quantity' => 'required_with:product_materials|integer|min:1',
            'product_materials.*.unit' => 'required_with:product_materials|string|max:50',
            'product_spareparts' => 'sometimes|array',
            'product_spareparts.*.name' => 'required_with:product_spareparts|string|max:255',
            'product_spareparts.*.type' => 'required_with:product_spareparts|in:material,sparepart',
            'product_spareparts.*.quantity' => 'required_with:product_spareparts|integer|min:1',
            'product_spareparts.*.unit' => 'required_with:product_spareparts|string|max:50',

            // Specifications
            'product_specifications' => 'sometimes|array',
            'product_specifications.*.key' => 'nullable|string|max:255',
            'product_specifications.*.value' => 'nullable|string|max:255',
        ]);

        $product = Product::with(['specifications', 'images', 'materials', 'spareparts'])->findOrFail($id);

        DB::beginTransaction();

        try {
            // === HANDLE HAPUS GAMBAR UTAMA ===
            if ($request->remove_main_image == '1') {
                $this->deleteProductImage($product->product_image);
                $product->product_image = null;
            }

            // === UPDATE GAMBAR UTAMA (jika ada upload baru) ===
            if ($request->hasFile('product_image')) {
                $this->deleteProductImage($product->product_image);
                $this->saveProductImage($request->file('product_image'), $product, true);
            }

            // === UPDATE FIELD UTAMA ===
            $product->update([
                'product_title' => $request->product_title,
                'product_description' => $request->product_description,
                'product_prices' => $request->product_prices ?? 0,
                'category_id' => $request->category_id,
                'product_work_duration' => $request->product_work_duration,
                'product_garansi' => $request->product_garansi,
                'service_fee' => $request->service_fee ?? 0,
                'is_active' => $request->has('is_active') ? 1 : 0, // Handle checkbox
            ]);

            // === HANDLE HAPUS GAMBAR TAMBAHAN ===
            if ($request->filled('remove_images')) {
                $imageIdsToRemove = explode(',', $request->remove_images);
                foreach ($imageIdsToRemove as $imageId) {
                    $image = $product->images()->find($imageId);
                    if ($image) {
                        $this->deleteProductImage($image->image_path);
                        $image->delete();
                    }
                }
            }

            // === TAMBAH GAMBAR TAMBAHAN BARU ===
            if ($request->hasFile('product_images')) {
                $currentImagesCount = $product->images()->count();
                $maxNew = 5 - $currentImagesCount;
                $uploadedCount = 0;

                foreach ($request->file('product_images') as $file) {
                    if ($uploadedCount >= $maxNew) break;
                    if ($file->isValid()) {
                        $this->saveProductImage($file, $product, false);
                        $uploadedCount++;
                    }
                }
            }

            // === UPDATE MATERIALS DAN SPAREPARTS ===
            // Hapus data lama dari tabel pivot
            DB::table('product_material')->where('product_id', $product->id)->delete();
            DB::table('product_sparepart')->where('product_id', $product->id)->delete();
            // Simpan data baru
            $this->saveProductRelations($product, $request, 'material');
            $this->saveProductRelations($product, $request, 'sparepart');

            // === UPDATE SPESIFIKASI ===
            $product->specifications()->delete();
            if ($request->filled('product_specifications')) {
                foreach ($request->product_specifications as $spec) {
                    if (!empty($spec['key']) && !empty($spec['value'])) {
                        $product->specifications()->create([
                            'key' => $spec['key'],
                            'value' => $spec['value'],
                        ]);
                    }
                }
            }

            DB::commit();

            // Ambil query string dari request untuk kembali ke halaman yang sama
            $queryParams = array_filter($request->only(['page', 'search', 'category', 'status']), function ($value) {
                return $value !== null && $value !== '';
            });

            return redirect()->route('admin.master.products.index', $queryParams)
                ->with('add_message', 'Produk berhasil diperbarui!');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withErrors([
                'msg' => 'Gagal memperbarui produk: ' . $e->getMessage()
            ])->withInput();
        }
    }

    /**
     * Hapus produk
     */
    public function destroy($id)
    {
        $product = Product::with(['images', 'materials', 'spareparts'])->findOrFail($id);

        DB::beginTransaction();
        try {
            // Hapus gambar utama
            $this->deleteProductImage($product->product_image);

            // Hapus gambar tambahan
            foreach ($product->images as $image) {
                $this->deleteProductImage($image->image_path);
            }

            // Hapus relasi pivot (akan otomatis terhapus karena cascade)
            $product->materials()->detach();
            $product->spareparts()->detach();

            $product->delete();

            DB::commit();

            return redirect()->route('admin.master.products.index')
                ->with('add_message', 'Produk berhasil dihapus!');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withErrors([
                'msg' => 'Gagal menghapus produk: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Hapus banyak produk sekaligus
     */
    public function bulkDestroy(Request $request)
    {
        // Handle JSON string from form
        $ids = $request->ids;
        if (is_string($ids)) {
            $ids = json_decode($ids, true);
        }

        $request->merge(['ids' => $ids]);

        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:products,id'
        ]);

        DB::beginTransaction();
        try {
            $deletedCount = 0;

            foreach ($ids as $id) {
                $product = Product::with(['images', 'materials', 'spareparts'])->findOrFail($id);

                // Hapus gambar utama
                $this->deleteProductImage($product->product_image);

                // Hapus gambar tambahan
                foreach ($product->images as $image) {
                    $this->deleteProductImage($image->image_path);
                }

                // Hapus relasi pivot
                $product->materials()->detach();
                $product->spareparts()->detach();

                $product->delete();
                $deletedCount++;
            }

            DB::commit();

            $message = $deletedCount > 1
                ? "{$deletedCount} produk berhasil dihapus!"
                : "Produk berhasil dihapus!";

            return redirect()->route('admin.master.products.index')
                ->with('add_message', $message);
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withErrors([
                'msg' => 'Gagal menghapus produk: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Menampilkan detail produk
     */
    public function show($id)
    {
        $product = Product::with([
            'category',
            'specifications',
            'images',
            'materials',
            'spareparts'
        ])->findOrFail($id);

        return view('admin.master.products.show', compact('product'));
    }

    /**
     * Preview template dokumen lengkap produk jasa
     */
    public function previewTemplate($id)
    {
        $product = Product::with([
            'category',
            'specifications',
            'materials',
            'spareparts'
        ])->findOrFail($id);

        return view('admin.master.products.print-template', compact('product'));
    }

    /**
     * Export template dokumen lengkap produk jasa (PDF)
     */
    public function exportTemplate($id)
    {
        $product = Product::with([
            'category',
            'specifications',
            'materials',
            'spareparts'
        ])->findOrFail($id);

        // Hitung total biaya material dan sparepart
        // Relasi adalah hasMany, field biaya ada di usage_cost
        $totalMaterialCost = $product->materials->sum(function ($material) {
            return ($material->quantity ?? 0) * ($material->usage_cost ?? 0);
        });

        $totalSparepartCost = $product->spareparts->sum(function ($sparepart) {
            return ($sparepart->quantity ?? 0) * ($sparepart->usage_cost ?? 0);
        });

        $data = [
            'product' => $product,
            'totalMaterialCost' => $totalMaterialCost,
            'totalSparepartCost' => $totalSparepartCost,
        ];

        $pdf = Pdf::loadView('admin.master.products.print-template', $data)->setPaper('A4');

        $filename = 'Spesifikasi_Teknis_' . Str::slug($product->product_title) . '.pdf';
        return $pdf->download($filename);
    }

    /**
     * Menampilkan detail produk untuk user/guest
     */
    public function productDetails($id)
    {
        $product = Product::with(['category', 'specifications', 'images', 'materials', 'spareparts'])
            ->findOrFail($id);

        // Simpan URL produk yang sedang dilihat untuk redirect setelah login
        session(['url.intended' => url()->current()]);

        return view('home.product_details', compact('product'));
    }

    /**
     * Tampilkan daftar produk untuk user
     * Redirect ke halaman all (katalog produk)
     */
    public function indexUser()
    {
        return redirect()->route('all');
    }

    /**
     * Tampilkan detail produk untuk user
     * Redirect ke halaman product_details
     */
    public function showUser(Product $product)
    {
        return redirect()->route('product_details', $product->id);
    }
}
