<?php

namespace App\Http\Controllers;

use App\Models\Production;
use App\Models\Order;
use App\Models\Material;
use App\Models\Sparepart;
use App\Models\User;
use App\Models\ProductionMaterial;
use App\Models\ProductionSparepart;
use App\Models\ProductionItemRequest;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;

class ProductionController extends Controller
{
    // --- LIST PRODUCTIONS ---
    public function index(Request $request)
    {
        $search = $request->input('search');
        $statusFilter = $request->input('status');
        $teknisiFilter = $request->input('teknisi');
        
        $query = Production::with([
            'order.product.category', 
            'teknisi', 
            'productionMaterials.material',
            'productionMaterials.creator',
            'productionMaterials.reviser',
            'productionSpareparts.sparepart',
            'productionSpareparts.creator',
            'productionSpareparts.reviser'
        ]);

        // Search filter - perbaiki dengan grouping yang benar
        if ($search && trim($search) !== '') {
            $searchTerm = trim($search);
            $query->where(function($q) use ($searchTerm) {
                $q->whereHas('order', function($q2) use ($searchTerm) {
                    $q2->where('customer_name', 'like', "%{$searchTerm}%")
                        ->orWhere('id', 'like', "%{$searchTerm}%");
                })
                ->orWhereHas('order.product', function($q2) use ($searchTerm) {
                    $q2->where('product_title', 'like', "%{$searchTerm}%");
                })
                ->orWhereHas('order.product.category', function($q2) use ($searchTerm) {
                    $q2->where('name', 'like', "%{$searchTerm}%");
                })
                ->orWhere('id', 'like', "%{$searchTerm}%");
            });
        }

        // Status filter
        if ($statusFilter && in_array($statusFilter, ['menunggu', 'dalam_proses', 'selesai', 'dibatalkan'])) {
            $query->where('status', $statusFilter);
        }

        // Teknisi assignment filter
        if ($teknisiFilter === 'unassigned') {
            $query->whereNull('teknisi_id');
        } elseif ($teknisiFilter === 'assigned') {
            $query->whereNotNull('teknisi_id');
        }

        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $productions = $query->latest()->paginate($itemsPerPage)->withQueryString();
        $pendingOrders = Order::where('production_status', 'menunggu')->latest()->get();

        return view('admin.master.productions.index', compact('productions', 'pendingOrders'));
    }


    // --- SHOW PRODUCTION ---
    public function show(Production $production)
    {
        $production->load([
            'order.product',
            'product',
            'teknisi',
            'supervisor',
            'servicePackage', // Load service package relationship
            'productionMaterials.material',
            'productionMaterials.creator',
            'productionMaterials.reviser',
            'productionSpareparts.sparepart',
            'productionSpareparts.creator',
            'productionSpareparts.reviser',
            'itemRequests.material',
            'itemRequests.sparepart',
            'itemRequests.requester',
            'itemRequests.processor',
            'itemRequests.purchase'
        ]);

        // Gabungkan materials dan spareparts untuk tabel unified dengan info ketersediaan
        $allItems = $this->getCombinedProductionItems($production);
        
        // Group item requests by proposal number
        $itemRequestsByProposal = $production->itemRequests->groupBy('proposal_number');
        
        // Pending item requests
        $pendingRequests = $production->itemRequests->where('status', 'pending');

        return view('admin.master.productions.show', compact('production', 'allItems', 'itemRequestsByProposal', 'pendingRequests'));
    }

    /**
     * Gabungkan materials dan spareparts menjadi satu collection dengan info ketersediaan
     */
    private function getCombinedProductionItems(Production $production)
    {
        $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
        
        $items = collect();
        
        // Add materials
        foreach ($production->productionMaterials as $pm) {
            $totalNeeded = $pm->quantity * $orderQty;
            $currentStock = $pm->material->stock ?? 0;
            $stockStatus = $currentStock >= $totalNeeded ? 'available' : 'insufficient';
            $shortage = max(0, $totalNeeded - $currentStock);
            
            $items->push([
                'id' => $pm->id,
                'type' => 'material',
                'type_label' => 'Bahan',
                'item_id' => $pm->material_id,
                'name' => $pm->material->name ?? 'N/A',
                'quantity' => $pm->quantity,
                'unit' => $pm->unit,
                'unit_cost' => $pm->unit_cost,
                'total_cost' => $pm->total_cost,
                'current_stock' => $currentStock,
                'total_needed' => $totalNeeded,
                'shortage' => $shortage,
                'stock_status' => $stockStatus,
                'stock_status_label' => $stockStatus === 'available' ? 'Tersedia' : 'Perlu Pembelian',
                'is_received' => $pm->is_received,
                'received_at' => $pm->received_at,
                'is_additional' => $pm->is_additional,
                'model' => $pm,
            ]);
        }
        
        // Add spareparts
        foreach ($production->productionSpareparts as $ps) {
            $totalNeeded = $ps->quantity * $orderQty;
            $currentStock = $ps->sparepart->stock ?? 0;
            $stockStatus = $currentStock >= $totalNeeded ? 'available' : 'insufficient';
            $shortage = max(0, $totalNeeded - $currentStock);
            
            $items->push([
                'id' => $ps->id,
                'type' => 'sparepart',
                'type_label' => 'Sparepart',
                'item_id' => $ps->sparepart_id,
                'name' => $ps->sparepart->name ?? 'N/A',
                'quantity' => $ps->quantity,
                'unit' => $ps->unit,
                'unit_cost' => $ps->unit_cost,
                'total_cost' => $ps->total_cost,
                'current_stock' => $currentStock,
                'total_needed' => $totalNeeded,
                'shortage' => $shortage,
                'stock_status' => $stockStatus,
                'stock_status_label' => $stockStatus === 'available' ? 'Tersedia' : 'Perlu Pembelian',
                'is_received' => $ps->is_received,
                'received_at' => $ps->received_at,
                'is_additional' => $ps->is_additional,
                'model' => $ps,
            ]);
        }
        
        return $items;
    }

    // --- UPDATE PRODUCTION ---
    // ⚠️ PENTING: Admin TIDAK bisa input bahan/sparepart langsung
    // Admin hanya bisa input sebagai "suggested" yang akan dikirim ke teknisi untuk direvisi
    public function update(Request $request, Production $production)
    {
        if (in_array($production->status, ['selesai', 'dibatalkan'])) {
            return back()->withErrors(['status' => 'Produksi yang sudah selesai atau dibatalkan tidak bisa diupdate!']);
        }

        // Admin hanya bisa update notes, tidak bisa ubah status atau input bahan/sparepart langsung
        $request->validate([
            'notes' => 'nullable|string|max:1000',
        ]);

        $production->update($request->only(['notes']));

        return redirect()->route('admin.productions.show', $production->id)
            ->with('success', 'Catatan produksi berhasil diupdate!');
    }

    // --- DELETE PRODUCTION ---
    public function destroy(Production $production)
    {
        if ($production->status === 'selesai') {
            return back()->withErrors(['status' => 'Produksi yang sudah selesai tidak bisa dihapus!']);
        }

        return DB::transaction(function () use ($production) {
            // ✅ RESTORE STOK jika production sudah mulai (status = 'dalam_proses')
            if ($production->status === 'dalam_proses') {
                $production->load(['productionMaterials.material', 'productionSpareparts.sparepart']);
                $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
                
                // Restore stok materials
                foreach ($production->productionMaterials as $pm) {
                    $pm->material->increment('stock', $pm->quantity * $orderQty);
                }
                
                // Restore stok spareparts
                foreach ($production->productionSpareparts as $ps) {
                    $ps->sparepart->increment('stock', $ps->quantity * $orderQty);
                }
            }
            
            $production->delete();
            return redirect()->route('admin.productions.index')->with('success', 'Produksi berhasil dihapus!');
        });
    }

    // --- COMPLETE PRODUCTION ---
    // ⚠️ PENTING: Method ini hanya untuk emergency case
    // Normal workflow: Teknisi submit completion → Supervisor approve
    public function complete(Production $production)
    {
        // Validasi: hanya untuk emergency (misalnya teknisi tidak bisa akses sistem)
        if (!in_array($production->status, ['dalam_proses'])) {
            return back()->withErrors(['status' => 'Hanya produksi dengan status "Dalam Proses" yang bisa diselesaikan secara emergency!']);
        }
        
        // Validasi: pastikan completion belum di-submit
        if ($production->completion_status === 'pending_approval') {
            return back()->withErrors(['status' => 'Produksi sudah di-submit untuk approval. Gunakan workflow normal (supervisor approve).']);
            }

        // Validasi: pastikan planning sudah approved
        if ($production->planning_status !== 'approved') {
            return back()->withErrors(['planning' => 'Planning belum disetujui supervisor. Produksi tidak bisa diselesaikan.']);
            }

        // Warning: ini adalah emergency case
        Log::warning('Production completed by admin (emergency)', [
            'production_id' => $production->id,
            'admin_id' => Auth::id(),
            'admin_name' => Auth::user()->name,
        ]);
        
        return DB::transaction(function () use ($production) {
            $production->update([
                'status' => 'selesai',
                'end_date' => now(),
                'completion_status' => 'approved', // Auto-approve untuk emergency
                'completion_approved_at' => now(),
                'completion_approved_by' => Auth::id(),
            ]);
            
            $production->order->updateQuietly(['production_status' => 'selesai']);

            // CATATAN: Sale TIDAK dibuat otomatis di sini, bahkan untuk emergency completion
            // Sale hanya dibuat ketika customer mengkonfirmasi serah terima
            // melalui CustomerConfirmationController atau HandoverController forceComplete
            // Admin harus menggunakan menu "Serah Terima" untuk mengirim link konfirmasi ke customer
            
            return redirect()->route('admin.productions.index')
                ->with('warning', 'Produksi diselesaikan secara emergency. Pastikan workflow normal digunakan untuk produksi selanjutnya. Sale akan dibuat setelah customer mengkonfirmasi serah terima.');
        });
    }

    // --- CANCEL PRODUCTION ---
    public function cancel(Production $production)
    {
        if (in_array($production->status, ['selesai', 'dibatalkan'])) {
            return back()->withErrors(['status' => 'Produksi yang sudah selesai atau dibatalkan tidak bisa dibatalkan lagi!']);
        }

        return DB::transaction(function () use ($production) {
            $production->update(['status' => 'dibatalkan']);

            // Update status order otomatis
            $production->order->update(['production_status' => 'dibatalkan']);

            return redirect()->route('admin.productions.index')->with('success', 'Produksi berhasil dibatalkan!');
        });
    }

    // --- ASSIGN TO TEKNISI (KEPALA PRODUKSI) ---
    public function assignToTeknisi(Request $request, Production $production)
    {
        try {
            Log::info('Assign teknisi called', [
                'production_id' => $production->id,
                'current_teknisi_id' => $production->teknisi_id,
                'current_status' => $production->status
            ]);

            // Validasi: pastikan produksi belum ditugaskan
            if ($production->teknisi_id) {
                Log::warning('Production already assigned', ['production_id' => $production->id]);
                $error = 'Produksi ini sudah ditugaskan ke teknisi!';
                if ($request->expectsJson() || $request->ajax()) {
                    return response()->json(['success' => false, 'message' => $error], 422);
                }
                return redirect()->route('admin.productions.index')->withErrors(['assign' => $error]);
            }

            // Validasi: pastikan produksi tidak selesai atau dibatalkan
            if (in_array($production->status, ['selesai', 'dibatalkan'])) {
                Log::warning('Production cannot be assigned', [
                    'production_id' => $production->id,
                    'status' => $production->status
                ]);
                $error = 'Produksi yang sudah selesai atau dibatalkan tidak bisa ditugaskan!';
                if ($request->expectsJson() || $request->ajax()) {
                    return response()->json(['success' => false, 'message' => $error], 422);
                }
                return redirect()->route('admin.productions.index')->withErrors(['assign' => $error]);
            }

            // Validasi: Pastikan order status sudah diterima sebelum assign ke teknisi
            if ($production->order && $production->order->order_status !== 'diterima') {
                $error = 'Order harus dalam status "diterima" sebelum bisa ditugaskan ke teknisi!';
                if ($request->expectsJson() || $request->ajax()) {
                    return response()->json(['success' => false, 'message' => $error], 422);
                }
                return redirect()->route('admin.productions.index')->withErrors(['assign' => $error]);
            }

            // Validasi: Ambil teknisi dari request jika ada, jika tidak ambil teknisi pertama
            $teknisiId = $request->input('teknisi_id');
            if ($teknisiId) {
                $teknisi = User::where('user_type', 'teknisi')->find($teknisiId);
                if (!$teknisi) {
                    $error = 'Teknisi yang dipilih tidak ditemukan atau bukan teknisi!';
                    if ($request->expectsJson() || $request->ajax()) {
                        return response()->json(['success' => false, 'message' => $error], 422);
                    }
                    return redirect()->route('admin.productions.index')->withErrors(['assign' => $error]);
                }
            } else {
                // Ambil teknisi pertama (kepala produksi) - fallback untuk backward compatibility
            $teknisi = User::where('user_type', 'teknisi')->first();
            }
            
            Log::info('Teknisi query result', [
                'teknisi_found' => $teknisi ? true : false,
                'teknisi_id' => $teknisi ? $teknisi->id : null,
                'teknisi_name' => $teknisi ? $teknisi->name : null
            ]);
            
            if (!$teknisi) {
                Log::error('No teknisi found in database');
                // Cek semua user untuk debugging
                $allUsers = User::select('id', 'name', 'email', 'user_type')->get();
                Log::info('All users in database', ['users' => $allUsers->toArray()]);
                
                $error = 'Tidak ada teknisi yang tersedia! Pastikan ada user dengan role teknisi.';
                if ($request->expectsJson() || $request->ajax()) {
                    return response()->json(['success' => false, 'message' => $error], 422);
                }
                return redirect()->route('admin.productions.index')->withErrors(['assign' => $error]);
            }

            // Set estimated duration dari product jika belum ada
            $estimatedDuration = $production->estimated_duration_days;
            if (!$estimatedDuration && $production->product && $production->product->product_work_duration) {
                // Convert string duration (e.g., "2 minggu", "7 hari") to integer days
                $estimatedDuration = $this->parseDurationToDays($production->product->product_work_duration);
            }
            
            // Ensure estimatedDuration is integer
            $estimatedDuration = (int) ($estimatedDuration ?? 7); // Default 7 hari jika tidak ada

            // Update produksi dengan teknisi
            // Set planning_status ke 'planning' agar teknisi bisa mulai input bahan/sparepart
            $updateData = [
                'teknisi_id' => $teknisi->id,
                'assigned_at' => now(),
                'status' => 'menunggu',
                'planning_status' => 'planning', // Teknisi bisa mulai planning
                'estimated_duration_days' => $estimatedDuration,
                'materials_status' => 'pending',
                'spareparts_status' => 'pending',
                'material_preparation_days' => $production->material_preparation_days ?? 2,
            ];

            Log::info('Updating production', [
                'production_id' => $production->id,
                'update_data' => $updateData
            ]);

            $updated = $production->update($updateData);

            Log::info('Production update result', [
                'production_id' => $production->id,
                'update_success' => $updated,
                'teknisi_id_after' => $production->fresh()->teknisi_id,
                'assigned_at_after' => $production->fresh()->assigned_at
            ]);

            if (!$updated) {
                Log::error('Failed to update production', ['production_id' => $production->id]);
                $error = 'Gagal mengupdate produksi!';
                if ($request->expectsJson() || $request->ajax()) {
                    return response()->json(['success' => false, 'message' => $error], 500);
                }
                return redirect()->route('admin.productions.index')->withErrors(['assign' => $error]);
            }

            // Kirim notifikasi untuk produksi ditugaskan
            try {
                $notificationService = new \App\Services\NotificationService();
                $notificationService->notifyProductionAssigned($production->fresh(), $teknisi->id);
            } catch (\Exception $e) {
                Log::error('Failed to send production assignment notification: ' . $e->getMessage());
            }

            // Handle AJAX request
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => "Produksi berhasil ditugaskan ke {$teknisi->name} (Kepala Produksi)!",
                    'redirect' => route('admin.productions.index')
                ]);
            }

            return redirect()->route('admin.productions.index')
                ->with('success', "Produksi berhasil ditugaskan ke {$teknisi->name} (Kepala Produksi)!");
                
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            Log::error('Production not found', ['error' => $e->getMessage()]);
            $error = 'Produksi tidak ditemukan!';
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json(['success' => false, 'message' => $error], 404);
            }
            return redirect()->route('admin.productions.index')->withErrors(['assign' => $error]);
        } catch (\Exception $e) {
            Log::error('Error assigning teknisi', [
                'production_id' => $production->id ?? null,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            $error = 'Terjadi kesalahan saat menugaskan produksi: ' . $e->getMessage();
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json(['success' => false, 'message' => $error], 500);
            }
            return redirect()->route('admin.productions.index')->withErrors(['assign' => $error]);
        }
    }

    // --- PREPARE MATERIALS (Admin siapkan bahan) ---
    public function prepareMaterials(Production $production)
    {
        // Validasi: pastikan status belum "received"
        if ($production->materials_status === 'received') {
            return back()->withErrors(['materials' => 'Bahan (materials) sudah diterima teknisi.']);
        }

        // Validasi: pastikan status belum "preparing" (mencegah klik berulang)
        if ($production->materials_status === 'preparing') {
            return back()->with('info', 'Status bahan (materials) sudah "Sedang Disiapkan". Tidak perlu diklik lagi.');
        }

        // Validasi: pastikan ada materials di production
        if ($production->productionMaterials->count() === 0) {
            return back()->withErrors(['materials' => 'Tidak ada bahan (materials) di production ini.']);
        }

        $production->update([
            'materials_status' => 'preparing',
        ]);

        return back()->with('success', 'Status bahan (materials) diubah menjadi "Sedang Disiapkan". Teknisi akan mendapat notifikasi.');
    }

    // --- PREPARE SPAREPARTS (Admin siapkan spareparts) ---
    public function prepareSpareparts(Production $production)
    {
        // Validasi: pastikan status belum "received"
        if ($production->spareparts_status === 'received') {
            return back()->withErrors(['spareparts' => 'Spareparts sudah diterima teknisi.']);
        }

        // Validasi: pastikan status belum "preparing" (mencegah klik berulang)
        if ($production->spareparts_status === 'preparing') {
            return back()->with('info', 'Status spareparts sudah "Sedang Disiapkan". Tidak perlu diklik lagi.');
        }

        // Validasi: pastikan ada spareparts di production
        if ($production->productionSpareparts->count() === 0) {
            return back()->withErrors(['spareparts' => 'Tidak ada spareparts di production ini.']);
        }

        $production->update([
            'spareparts_status' => 'preparing',
        ]);

        return back()->with('success', 'Status spareparts diubah menjadi "Sedang Disiapkan". Teknisi akan mendapat notifikasi.');
    }

    // --- UNASSIGN FROM TEKNISI ---
    public function unassignFromTeknisi(Request $request, Production $production)
    {
        try {
            // Validasi: pastikan produksi sudah ditugaskan
            if (!$production->teknisi_id) {
                return back()->withErrors(['unassign' => 'Produksi ini belum ditugaskan ke teknisi!']);
            }

            // Validasi: pastikan produksi tidak selesai atau dibatalkan
            if (in_array($production->status, ['selesai', 'dibatalkan'])) {
                return back()->withErrors(['unassign' => 'Produksi yang sudah selesai atau dibatalkan tidak bisa di-unassign!']);
            }

            // Unassign teknisi
            $production->update([
                'teknisi_id' => null,
                'assigned_at' => null,
            ]);

            return redirect()->route('admin.productions.index')
                ->with('success', 'Produksi berhasil di-unassign dari teknisi!');
                
        } catch (\Exception $e) {
            \Log::error('Error unassigning teknisi: ' . $e->getMessage(), [
                'production_id' => $production->id,
                'trace' => $e->getTraceAsString()
            ]);
            return back()->withErrors(['unassign' => 'Terjadi kesalahan saat membatalkan tugas: ' . $e->getMessage()]);
        }
    }

    // --- CHECK STOCK AJAX ---
    public function checkStock(Request $request)
    {
        $production = Production::with('materials', 'spareparts')->find($request->id);
        if (!$production) return response()->json(['ok' => false, 'message' => 'Produksi tidak ditemukan']);

        $insufficient = [];
        $orderQty = max(1, (int) ($production->quantity ?? 1));

        foreach ($production->materials as $m) {
            $needed = (int) (($m->pivot->quantity ?? 0) * $orderQty);
            if ($m->stock < $needed) {
                $insufficient[] = "{$m->name} (stok: {$m->stock}, butuh: {$needed})";
            }
        }

        foreach ($production->spareparts as $s) {
            $needed = (int) (($s->pivot->quantity ?? 0) * $orderQty);
            if ($s->stock < $needed) {
                $insufficient[] = "{$s->name} (stok: {$s->stock}, butuh: {$needed})";
            }
        }

        return response()->json(['ok' => count($insufficient) === 0, 'message' => implode('<br>', $insufficient)]);
    }

    // --- HELPER: PARSE DURATION STRING TO DAYS ---
    /**
     * Convert duration string to integer days
     * Examples: "2 minggu" -> 14, "7 hari" -> 7, "1 bulan" -> 30, etc.
     */
    private function parseDurationToDays(?string $duration): int
    {
        if (!$duration) {
            return 7; // Default 7 hari
        }

        // Remove extra spaces and convert to lowercase
        $duration = strtolower(trim($duration));

        // Extract number and unit
        preg_match('/(\d+)\s*(minggu|hari|bulan|week|day|month)?/i', $duration, $matches);

        if (empty($matches) || !isset($matches[1])) {
            // If no match, try to extract just the number
            preg_match('/(\d+)/', $duration, $numMatches);
            if (isset($numMatches[1])) {
                return (int) $numMatches[1]; // Assume it's already in days
            }
            return 7; // Default
        }

        $number = (int) $matches[1];
        $unit = isset($matches[2]) ? strtolower($matches[2]) : 'hari';

        // Convert to days based on unit
        return match ($unit) {
            'minggu', 'week' => $number * 7,
            'bulan', 'month' => $number * 30,
            'hari', 'day', '' => $number,
            default => $number, // Default assume days
        };
    }

    // --- ADMIN INPUT BAHAN/SPAREPART SEBAGAI SUGGESTED ---
    /**
     * Admin bisa input bahan dan sparepart sebagai "suggested" yang akan dikirim ke teknisi
     * Teknisi bisa merevisi atau mengedit bahan/sparepart yang diinput admin
     */
    public function suggestMaterialsAndSpareparts(Request $request, Production $production)
    {
        // Validasi: produksi belum selesai atau dibatalkan
        if (in_array($production->status, ['selesai', 'dibatalkan'])) {
            return back()->withErrors(['status' => 'Produksi yang sudah selesai atau dibatalkan tidak bisa diinput bahan/sparepart!']);
        }

        $request->validate([
            'production_materials.*.material_id' => 'required_with:production_materials|exists:materials,id',
            'production_materials.*.quantity' => 'required_with:production_materials|numeric|min:0',
            'production_materials.*.unit' => 'nullable|string',
            'production_materials.*.unit_cost' => 'nullable|numeric|min:0',
            'production_spareparts.*.sparepart_id' => 'required_with:production_spareparts|exists:spareparts,id',
            'production_spareparts.*.quantity' => 'required_with:production_spareparts|numeric|min:0',
            'production_spareparts.*.unit' => 'nullable|string',
            'production_spareparts.*.unit_cost' => 'nullable|numeric|min:0',
        ]);

        return DB::transaction(function () use ($request, $production) {
            $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
            $orderQty = max(1, (int) $orderQty);

            // Input materials sebagai suggested
            if ($request->production_materials) {
                foreach ($request->production_materials as $item) {
                    if (!empty($item['material_id']) && $item['quantity'] > 0) {
                        $material = Material::findOrFail($item['material_id']);
                        $unitCost = $item['unit_cost'] ?? $material->price;
                        $totalCost = $item['quantity'] * $unitCost * $orderQty;

                        ProductionMaterial::create([
                            'production_id' => $production->id,
                            'material_id' => $item['material_id'],
                            'quantity' => $item['quantity'],
                            'unit' => $item['unit'] ?? $material->unit,
                            'unit_cost' => $unitCost,
                            'total_cost' => $totalCost,
                            'created_by' => Auth::id(),
                            'is_admin_suggested' => true,
                        ]);
                    }
                }
            }

            // Input spareparts sebagai suggested
            if ($request->production_spareparts) {
                foreach ($request->production_spareparts as $item) {
                    if (!empty($item['sparepart_id']) && $item['quantity'] > 0) {
                        $sparepart = Sparepart::findOrFail($item['sparepart_id']);
                        $unitCost = $item['unit_cost'] ?? $sparepart->price;
                        $totalCost = $item['quantity'] * $unitCost * $orderQty;

                        ProductionSparepart::create([
                            'production_id' => $production->id,
                            'sparepart_id' => $item['sparepart_id'],
                            'quantity' => $item['quantity'],
                            'unit' => $item['unit'] ?? $sparepart->unit,
                            'unit_cost' => $unitCost,
                            'total_cost' => $totalCost,
                            'created_by' => Auth::id(),
                            'is_admin_suggested' => true,
                        ]);
                    }
                }
            }

            // Update total cost
            $production->calculateCosts();

            return redirect()->route('admin.productions.show', $production)
                ->with('success', 'Bahan dan sparepart berhasil diinput sebagai suggested. Silakan kirim ke teknisi untuk direvisi.');
        });
    }

    // --- ADMIN KIRIM BAHAN/SPAREPART KE TEKNISI ---
    /**
     * Admin kirim bahan/sparepart yang sudah diinput ke teknisi untuk direvisi
     */
    public function sendToTeknisi(Production $production)
    {
        // Validasi: produksi harus sudah ditugaskan ke teknisi
        if (!$production->teknisi_id) {
            if (request()->expectsJson() || request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Produksi belum ditugaskan ke teknisi!'
                ], 400);
            }
            return back()->withErrors(['teknisi' => 'Produksi belum ditugaskan ke teknisi!']);
        }

        // Validasi: produksi belum selesai atau dibatalkan
        if (in_array($production->status, ['selesai', 'dibatalkan'])) {
            if (request()->expectsJson() || request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Produksi yang sudah selesai atau dibatalkan tidak bisa dikirim ke teknisi!'
                ], 400);
            }
            return back()->withErrors(['status' => 'Produksi yang sudah selesai atau dibatalkan tidak bisa dikirim ke teknisi!']);
        }

        return DB::transaction(function () use ($production) {
            // Update semua bahan/sparepart yang suggested untuk dikirim ke teknisi
            $materialsCount = $production->productionMaterials()
                ->where('is_admin_suggested', true)
                ->whereNull('sent_to_teknisi_at')
                ->update(['sent_to_teknisi_at' => now()]);

            $sparepartsCount = $production->productionSpareparts()
                ->where('is_admin_suggested', true)
                ->whereNull('sent_to_teknisi_at')
                ->update(['sent_to_teknisi_at' => now()]);

            // Update planning status jika belum ada
            if ($production->planning_status === null || $production->planning_status === 'planning') {
                $production->update(['planning_status' => 'planning']);
            }

            // Return JSON jika request AJAX
            if (request()->expectsJson() || request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Bahan dan sparepart berhasil dikirim ke teknisi. Teknisi dapat merevisi atau mengedit bahan/sparepart tersebut.',
                    'materials_count' => $materialsCount,
                    'spareparts_count' => $sparepartsCount
                ]);
            }

            return redirect()->route('admin.productions.show', $production)
                ->with('success', 'Bahan dan sparepart berhasil dikirim ke teknisi. Teknisi dapat merevisi atau mengedit bahan/sparepart tersebut.');
        });
    }

    // --- PROCESS ITEM REQUEST FROM TEKNISI ---
    /**
     * Approve item request dari teknisi
     * Admin bisa langsung approve atau redirect ke pembelian
     */
    public function approveItemRequest(Request $request, Production $production, $itemRequestId)
    {
        // Cari item request berdasarkan ID dan production_id
        $itemRequest = ProductionItemRequest::where('id', $itemRequestId)
            ->where('production_id', $production->id)
            ->first();
            
        if (!$itemRequest) {
            return back()->withErrors(['production' => 'Item request tidak ditemukan untuk produksi ini.']);
        }

        // Validasi: item request harus pending
        if ($itemRequest->status !== 'pending') {
            return back()->withErrors(['status' => 'Item request sudah diproses.']);
        }

        $request->validate([
            'admin_notes' => 'nullable|string|max:1000',
        ]);

        // Check stock availability
        $hasStock = false;
        $currentStock = 0;
        
        if ($itemRequest->item_type === 'material' && $itemRequest->material_id) {
            $material = Material::find($itemRequest->material_id);
            $currentStock = $material ? $material->stock : 0;
            $hasStock = $currentStock >= $itemRequest->quantity;
        } elseif ($itemRequest->item_type === 'sparepart' && $itemRequest->sparepart_id) {
            $sparepart = Sparepart::find($itemRequest->sparepart_id);
            $currentStock = $sparepart ? $sparepart->stock : 0;
            $hasStock = $currentStock >= $itemRequest->quantity;
        }

        // Jika item baru atau stok tidak cukup, langsung approve dan redirect ke pembelian
        // (Tombol "Buat Pembelian" sudah langsung approve dan redirect)
        if ($itemRequest->is_new_item || !$hasStock) {
            // Mark as approved but needs purchase
            $itemRequest->update([
                'status' => 'approved',
                'processed_by' => Auth::id(),
                'processed_at' => now(),
                'admin_notes' => $request->admin_notes,
            ]);

            // Kirim notifikasi ke teknisi
            try {
                $notificationService = new \App\Services\NotificationService();
                $notificationService->notifyItemRequestApproved($production->fresh(), $production->teknisi_id, $itemRequest->item_type);
            } catch (\Exception $e) {
                \Log::error('Failed to send item request approved notification: ' . $e->getMessage());
            }

            // Redirect to purchase page with pre-filled data
            return redirect()->route('admin.transaksi.purchases.create', [
                'from_item_request' => $itemRequest->id,
                'item_type' => $itemRequest->item_type,
                'item_name' => $itemRequest->item_name_display,
                'quantity' => $itemRequest->quantity,
                'unit' => $itemRequest->unit,
                'estimated_price' => $itemRequest->estimated_price,
                'material_id' => $itemRequest->material_id,
                'sparepart_id' => $itemRequest->sparepart_id,
            ])->with('info', 'Item request disetujui. Silakan buat pembelian untuk item ini.');
        }

        // If stock available, add to production materials/spareparts
        return DB::transaction(function () use ($request, $production, $itemRequest) {
            $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
            $productionStatus = $production->status;
            $isProductionStarted = in_array($productionStatus, ['dalam_proses', 'selesai']);
            
            if ($itemRequest->item_type === 'material') {
                $material = Material::lockForUpdate()->find($itemRequest->material_id);
                if (!$material) {
                    throw new \Exception("Material tidak ditemukan.");
                }
                
                $totalCost = $itemRequest->quantity * ($material->price ?? 0) * $orderQty;
                $needed = $itemRequest->quantity * $orderQty;
                
                // Jika produksi sudah dimulai, kurangi stok langsung
                if ($isProductionStarted) {
                    if ($material->stock < $needed) {
                        throw new \Exception("Stok material '{$material->name}' tidak cukup! Stok tersedia: {$material->stock}, butuh: {$needed}");
                    }
                    $material->decrement('stock', $needed);
                }
                
                ProductionMaterial::create([
                    'production_id' => $production->id,
                    'material_id' => $itemRequest->material_id,
                    'quantity' => $itemRequest->quantity,
                    'unit' => $itemRequest->unit,
                    'unit_cost' => $material->price ?? 0,
                    'total_cost' => $totalCost,
                    'created_by' => Auth::id(),
                    'is_additional' => true,
                    'request_reason' => $itemRequest->reason,
                ]);
            } else {
                $sparepart = Sparepart::lockForUpdate()->find($itemRequest->sparepart_id);
                if (!$sparepart) {
                    throw new \Exception("Sparepart tidak ditemukan.");
                }
                
                $totalCost = $itemRequest->quantity * ($sparepart->price ?? 0) * $orderQty;
                $needed = $itemRequest->quantity * $orderQty;
                
                // Jika produksi sudah dimulai, kurangi stok langsung
                if ($isProductionStarted) {
                    if ($sparepart->stock < $needed) {
                        throw new \Exception("Stok sparepart '{$sparepart->name}' tidak cukup! Stok tersedia: {$sparepart->stock}, butuh: {$needed}");
                    }
                    $sparepart->decrement('stock', $needed);
                }
                
                ProductionSparepart::create([
                    'production_id' => $production->id,
                    'sparepart_id' => $itemRequest->sparepart_id,
                    'quantity' => $itemRequest->quantity,
                    'unit' => $itemRequest->unit,
                    'unit_cost' => $sparepart->price ?? 0,
                    'total_cost' => $totalCost,
                    'created_by' => Auth::id(),
                    'is_additional' => true,
                    'request_reason' => $itemRequest->reason,
                ]);
            }

            // Update item request status
            $itemRequest->update([
                'status' => 'approved',
                'processed_by' => Auth::id(),
                'processed_at' => now(),
                'admin_notes' => $request->admin_notes,
            ]);

            // Recalculate production costs
            $production->calculateCosts();

            // Kirim notifikasi ke teknisi
            try {
                $notificationService = new \App\Services\NotificationService();
                $notificationService->notifyItemRequestApproved($production->fresh(), $production->teknisi_id, $itemRequest->item_type);
            } catch (\Exception $e) {
                \Log::error('Failed to send item request approved notification: ' . $e->getMessage());
            }

            return back()->with('success', 'Item request disetujui dan ditambahkan ke produksi.');
        });
    }

    /**
     * Reject item request dari teknisi
     */
    public function rejectItemRequest(Request $request, Production $production, $itemRequestId)
    {
        // Cari item request berdasarkan ID dan production_id
        $itemRequest = ProductionItemRequest::where('id', $itemRequestId)
            ->where('production_id', $production->id)
            ->first();
            
        if (!$itemRequest) {
            return back()->withErrors(['production' => 'Item request tidak ditemukan untuk produksi ini.']);
        }

        // Validasi: item request harus pending
        if ($itemRequest->status !== 'pending') {
            return back()->withErrors(['status' => 'Item request sudah diproses.']);
        }

        $request->validate([
            'admin_notes' => 'required|string|max:1000',
        ]);

        $itemRequest->update([
            'status' => 'rejected',
            'processed_by' => Auth::id(),
            'processed_at' => now(),
            'admin_notes' => $request->admin_notes,
        ]);

        // Kirim notifikasi ke teknisi
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyItemRequestRejected($production->fresh(), $production->teknisi_id, $itemRequest->item_type);
        } catch (\Exception $e) {
            \Log::error('Failed to send item request rejected notification: ' . $e->getMessage());
        }

        return back()->with('success', 'Item request ditolak.');
    }

    /**
     * Mark item request as purchased (setelah pembelian selesai)
     */
    public function markItemRequestPurchased(Request $request, Production $production, ProductionItemRequest $itemRequest)
    {
        // Validasi: item request harus approved
        if ($itemRequest->status !== 'approved') {
            return back()->withErrors(['status' => 'Item request harus disetujui terlebih dahulu.']);
        }

        $request->validate([
            'purchase_id' => 'required|exists:purchases,id',
        ]);

        $itemRequest->update([
            'status' => 'purchased',
            'purchase_id' => $request->purchase_id,
        ]);

        return back()->with('success', 'Item request ditandai sudah dibeli.');
    }

    /**
     * Send item request ke teknisi setelah pembelian selesai
     * Menambahkan item ke production materials/spareparts dan update status
     */
    public function sendItemRequest(Production $production, $itemRequestId)
    {
        // Cari item request
        $itemRequest = ProductionItemRequest::where('id', $itemRequestId)
            ->where('production_id', $production->id)
            ->first();
            
        if (!$itemRequest) {
            return back()->withErrors(['production' => 'Item request tidak ditemukan untuk produksi ini.']);
        }

        // Validasi: item request harus purchased
        if ($itemRequest->status !== 'purchased') {
            return back()->withErrors(['status' => 'Item harus sudah dibeli terlebih dahulu.']);
        }

        return DB::transaction(function () use ($production, $itemRequest) {
            $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
            $productionStatus = $production->status;
            $isProductionStarted = in_array($productionStatus, ['dalam_proses', 'selesai']);
            
            // Cek apakah item sudah ada di production
            $existingItem = null;
            if ($itemRequest->item_type === 'material' && $itemRequest->material_id) {
                $existingItem = $production->productionMaterials()
                    ->where('material_id', $itemRequest->material_id)
                    ->first();
            } elseif ($itemRequest->item_type === 'sparepart' && $itemRequest->sparepart_id) {
                $existingItem = $production->productionSpareparts()
                    ->where('sparepart_id', $itemRequest->sparepart_id)
                    ->first();
            }

            if ($existingItem) {
                // Update quantity jika item sudah ada
                $additionalQuantity = $itemRequest->quantity;
                $existingItem->increment('quantity', $additionalQuantity);
                $existingItem->update([
                    'is_additional' => true,
                    'request_reason' => $itemRequest->reason,
                ]);
                
                // SELALU kurangi stok untuk quantity tambahan saat admin kirim item ke teknisi
                // Karena item sudah dibeli dan dikirim ke teknisi, stok harus dikurangi
                $needed = $additionalQuantity * $orderQty;
                
                if ($itemRequest->item_type === 'material' && $itemRequest->material_id) {
                    $material = Material::lockForUpdate()->find($itemRequest->material_id);
                    if ($material) {
                        if ($material->stock < $needed) {
                            throw new \Exception("Stok material '{$material->name}' tidak cukup! Stok tersedia: {$material->stock}, butuh: {$needed}");
                        }
                        $material->decrement('stock', $needed);
                    }
                } elseif ($itemRequest->item_type === 'sparepart' && $itemRequest->sparepart_id) {
                    $sparepart = Sparepart::lockForUpdate()->find($itemRequest->sparepart_id);
                    if ($sparepart) {
                        if ($sparepart->stock < $needed) {
                            throw new \Exception("Stok sparepart '{$sparepart->name}' tidak cukup! Stok tersedia: {$sparepart->stock}, butuh: {$needed}");
                        }
                        $sparepart->decrement('stock', $needed);
                    }
                }
            } else {
                // Tambahkan item baru ke production
                if ($itemRequest->item_type === 'material' && $itemRequest->material_id) {
                    $material = Material::lockForUpdate()->find($itemRequest->material_id);
                    if (!$material) {
                        throw new \Exception("Material tidak ditemukan.");
                    }
                    
                    $totalCost = $itemRequest->quantity * ($material->price ?? 0) * $orderQty;
                    $needed = $itemRequest->quantity * $orderQty;
                    
                    // SELALU kurangi stok saat admin kirim item ke teknisi
                    // Karena item sudah dibeli dan dikirim ke teknisi, stok harus dikurangi
                    if ($material->stock < $needed) {
                        throw new \Exception("Stok material '{$material->name}' tidak cukup! Stok tersedia: {$material->stock}, butuh: {$needed}");
                    }
                    $material->decrement('stock', $needed);
                    
                    ProductionMaterial::create([
                        'production_id' => $production->id,
                        'material_id' => $itemRequest->material_id,
                        'quantity' => $itemRequest->quantity,
                        'unit' => $itemRequest->unit,
                        'unit_cost' => $material->price ?? 0,
                        'total_cost' => $totalCost,
                        'created_by' => Auth::id(),
                        'is_additional' => true,
                        'request_reason' => $itemRequest->reason,
                    ]);
                } elseif ($itemRequest->item_type === 'sparepart' && $itemRequest->sparepart_id) {
                    $sparepart = Sparepart::lockForUpdate()->find($itemRequest->sparepart_id);
                    if (!$sparepart) {
                        throw new \Exception("Sparepart tidak ditemukan.");
                    }
                    
                    $totalCost = $itemRequest->quantity * ($sparepart->price ?? 0) * $orderQty;
                    $needed = $itemRequest->quantity * $orderQty;
                    
                    // SELALU kurangi stok saat admin kirim item ke teknisi
                    // Karena item sudah dibeli dan dikirim ke teknisi, stok harus dikurangi
                    if ($sparepart->stock < $needed) {
                        throw new \Exception("Stok sparepart '{$sparepart->name}' tidak cukup! Stok tersedia: {$sparepart->stock}, butuh: {$needed}");
                    }
                    $sparepart->decrement('stock', $needed);
                    
                    ProductionSparepart::create([
                        'production_id' => $production->id,
                        'sparepart_id' => $itemRequest->sparepart_id,
                        'quantity' => $itemRequest->quantity,
                        'unit' => $itemRequest->unit,
                        'unit_cost' => $sparepart->price ?? 0,
                        'total_cost' => $totalCost,
                        'created_by' => Auth::id(),
                        'is_additional' => true,
                        'request_reason' => $itemRequest->reason,
                    ]);
                }
            }

            // Update item request status menjadi 'sent' (belum diterima teknisi)
            $itemRequest->update([
                'status' => 'sent',
                'processed_by' => Auth::id(),
                'processed_at' => now(),
            ]);

            // Recalculate production costs
            $production->calculateCosts();

            return back()->with('success', 'Item berhasil dikirim ke teknisi dan ditambahkan ke produksi.');
        });
    }

    /**
     * Check stock availability for all items in production
     * Returns items that need purchase
     */
    public function checkAllStockAvailability(Production $production)
    {
        $production->load(['productionMaterials.material', 'productionSpareparts.sparepart']);
        $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
        
        $insufficientItems = [];
        
        foreach ($production->productionMaterials as $pm) {
            $totalNeeded = $pm->quantity * $orderQty;
            $currentStock = $pm->material->stock ?? 0;
            
            if ($currentStock < $totalNeeded) {
                $insufficientItems[] = [
                    'type' => 'material',
                    'id' => $pm->material_id,
                    'name' => $pm->material->name ?? 'N/A',
                    'needed' => $totalNeeded,
                    'available' => $currentStock,
                    'shortage' => $totalNeeded - $currentStock,
                    'unit' => $pm->unit,
                    'unit_price' => $pm->material->price ?? 0,
                ];
            }
        }
        
        foreach ($production->productionSpareparts as $ps) {
            $totalNeeded = $ps->quantity * $orderQty;
            $currentStock = $ps->sparepart->stock ?? 0;
            
            if ($currentStock < $totalNeeded) {
                $insufficientItems[] = [
                    'type' => 'sparepart',
                    'id' => $ps->sparepart_id,
                    'name' => $ps->sparepart->name ?? 'N/A',
                    'needed' => $totalNeeded,
                    'available' => $currentStock,
                    'shortage' => $totalNeeded - $currentStock,
                    'unit' => $ps->unit,
                    'unit_price' => $ps->sparepart->price ?? 0,
                ];
            }
        }

        if (request()->expectsJson()) {
            return response()->json([
                'all_available' => empty($insufficientItems),
                'insufficient_items' => $insufficientItems,
            ]);
        }

        if (!empty($insufficientItems)) {
            // Store in session for purchase page
            session(['purchase_items_from_production' => [
                'production_id' => $production->id,
                'items' => $insufficientItems,
            ]]);
            
            return redirect()->route('admin.transaksi.purchases.create')
                ->with('info', 'Beberapa item stoknya tidak mencukupi. Silakan buat pembelian.');
        }

        return back()->with('success', 'Semua item tersedia di stok.');
    }

    // --- HELPER ATTACH MATERIAL & SPAREPART ---
    private function attachMaterialsAndSpareparts(Production $production, array $materials, array $spareparts, int $orderQty)
    {
        $totalMaterialCost = $totalSparepartCost = 0;
        $materialsData = $sparepartsData = [];

        foreach ($materials as $item) {
            if (!empty($item['material_id']) && $item['quantity'] > 0) {
                $m = Material::findOrFail($item['material_id']);
                $total = $item['quantity'] * $m->price * $orderQty;
                $materialsData[$m->id] = [
                    'quantity' => $item['quantity'],
                    'unit' => $m->unit,
                    'unit_cost' => $m->price,
                    'total_cost' => $total,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
                $totalMaterialCost += $total;
            }
        }
        if ($materialsData) $production->materials()->attach($materialsData);

        foreach ($spareparts as $item) {
            if (!empty($item['sparepart_id']) && $item['quantity'] > 0) {
                $s = Sparepart::findOrFail($item['sparepart_id']);
                $total = $item['quantity'] * $s->price * $orderQty;
                $sparepartsData[$s->id] = [
                    'quantity' => $item['quantity'],
                    'unit' => $s->unit,
                    'unit_cost' => $s->price,
                    'total_cost' => $total,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
                $totalSparepartCost += $total;
            }
        }
        if ($sparepartsData) $production->spareparts()->attach($sparepartsData);

        $production->update([
            'total_material_cost' => $totalMaterialCost,
            'total_sparepart_cost' => $totalSparepartCost,
            'total_production_cost' => $totalMaterialCost + $totalSparepartCost,
        ]);
    }

    /**
     * Fix production stock via web interface (untuk cPanel tanpa SSH)
     * Memperbaiki stok untuk produksi yang sudah selesai tapi stok belum dikurangi
     */
    public function fixStock(Request $request)
    {
        $productionId = $request->input('production_id');
        $dryRun = $request->has('dry_run');

        // Ambil produksi yang sudah selesai atau dalam proses
        $query = Production::with(['productionMaterials.material', 'productionSpareparts.sparepart'])
            ->whereIn('status', ['dalam_proses', 'selesai']);

        if ($productionId) {
            $query->where('id', $productionId);
        }

        $productions = $query->get();

        if ($productions->isEmpty()) {
            return back()->with('error', 'Tidak ada produksi yang ditemukan.');
        }

        $results = [];
        $totalFixed = 0;
        $totalSkipped = 0;
        $totalErrors = 0;

        foreach ($productions as $production) {
            $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
            $startDate = $production->start_date ?? $production->actual_start_date;

            if (!$startDate) {
                $results[] = [
                    'production_id' => $production->id,
                    'status' => 'skipped',
                    'message' => "Produksi #{$production->id} tidak memiliki start_date"
                ];
                $totalSkipped++;
                continue;
            }

            $fixed = 0;
            $errors = 0;
            $details = [];

            try {
                DB::beginTransaction();

                // Process materials
                foreach ($production->productionMaterials as $pm) {
                    if (!$pm->material_id || !$pm->material) {
                        $details[] = "⚠️ Production Material ID {$pm->id} tidak memiliki material_id atau material tidak ditemukan";
                        continue;
                    }

                    $needed = $pm->quantity * $orderQty;
                    $material = $pm->material;

                    // Cek apakah stok sudah dikurangi
                    // PERBEDAAN CARA PENGURANGAN STOK:
                    // 1. Daftar Bahan & Sparepart (is_additional = false): selalu dikurangi
                    // 2. Request Item (is_additional = true):
                    //    - created_at < start_date: dikurangi (ditambahkan sebelum produksi dimulai)
                    //    - created_at >= start_date: skip (sudah dikurangi saat ditambahkan)
                    $shouldReduce = true;
                    $skipReason = null;
                    
                    // Debug info
                    $debugInfo = "is_additional: " . ($pm->is_additional ? 'true' : 'false') . 
                                ", created_at: " . ($pm->created_at ? $pm->created_at->format('Y-m-d H:i:s') : 'null') . 
                                ", start_date: " . ($startDate ? $startDate->format('Y-m-d H:i:s') : 'null');
                    
                    if ($pm->is_additional && $pm->created_at && $startDate) {
                        // Convert to Carbon untuk perbandingan yang akurat
                        $createdAt = \Carbon\Carbon::parse($pm->created_at);
                        $startDateCarbon = \Carbon\Carbon::parse($startDate);
                        
                        if ($createdAt >= $startDateCarbon) {
                            $shouldReduce = false;
                            $skipReason = "Item request ditambahkan setelah produksi dimulai (created_at: {$createdAt->format('Y-m-d H:i:s')} >= start_date: {$startDateCarbon->format('Y-m-d H:i:s')})";
                        }
                    }

                    if ($shouldReduce) {
                        if (!$dryRun) {
                            $materialLocked = Material::lockForUpdate()->find($material->id);
                            if (!$materialLocked) {
                                $details[] = "❌ Material '{$material->name}' (ID: {$material->id}): Material tidak ditemukan di database";
                                $errors++;
                                continue;
                            }
                            
                            $oldStock = $materialLocked->stock;
                            
                            if ($oldStock >= $needed) {
                                $materialLocked->decrement('stock', $needed);
                                // Refresh untuk mendapatkan stok terbaru
                                $materialLocked->refresh();
                                $newStock = $materialLocked->stock;
                                $details[] = "✅ Material '{$material->name}' (ID: {$material->id}): {$oldStock} → {$newStock} (-{$needed}) [{$debugInfo}]";
                                $fixed++;
                                
                                // Log ke file untuk debugging
                                \Log::info("Fix Stock: Material '{$material->name}' (ID: {$material->id}) dikurangi dari {$oldStock} menjadi {$newStock} (-{$needed})");
                            } else {
                                $details[] = "⚠️ Material '{$material->name}' (ID: {$material->id}): Stok tidak cukup (stok: {$oldStock}, butuh: {$needed}) - mungkin sudah dikurangi sebelumnya";
                                // Tetap hitung sebagai fixed karena mungkin sudah benar
                                $fixed++;
                            }
                        } else {
                            $details[] = "🔄 [DRY RUN] Material '{$material->name}' (ID: {$material->id}): Akan dikurangi {$needed} (stok saat ini: {$material->stock}, {$debugInfo})";
                            $fixed++;
                        }
                    } else {
                        $details[] = "⏭️ Material '{$material->name}' (ID: {$material->id}): Skip - {$skipReason} [{$debugInfo}]";
                    }
                }

                // Process spareparts
                foreach ($production->productionSpareparts as $ps) {
                    if (!$ps->sparepart_id || !$ps->sparepart) {
                        $details[] = "⚠️ Production Sparepart ID {$ps->id} tidak memiliki sparepart_id atau sparepart tidak ditemukan";
                        continue;
                    }

                    $needed = $ps->quantity * $orderQty;
                    $sparepart = $ps->sparepart;

                    // Cek apakah stok sudah dikurangi
                    // PERBEDAAN CARA PENGURANGAN STOK:
                    // 1. Daftar Bahan & Sparepart (is_additional = false): selalu dikurangi
                    // 2. Request Item (is_additional = true):
                    //    - created_at < start_date: dikurangi (ditambahkan sebelum produksi dimulai)
                    //    - created_at >= start_date: skip (sudah dikurangi saat ditambahkan)
                    $shouldReduce = true;
                    $skipReason = null;
                    
                    // Debug info
                    $debugInfo = "is_additional: " . ($ps->is_additional ? 'true' : 'false') . 
                                ", created_at: " . ($ps->created_at ? $ps->created_at->format('Y-m-d H:i:s') : 'null') . 
                                ", start_date: " . ($startDate ? $startDate->format('Y-m-d H:i:s') : 'null');
                    
                    if ($ps->is_additional && $ps->created_at && $startDate) {
                        // Convert to Carbon untuk perbandingan yang akurat
                        $createdAt = \Carbon\Carbon::parse($ps->created_at);
                        $startDateCarbon = \Carbon\Carbon::parse($startDate);
                        
                        if ($createdAt >= $startDateCarbon) {
                            $shouldReduce = false;
                            $skipReason = "Item request ditambahkan setelah produksi dimulai (created_at: {$createdAt->format('Y-m-d H:i:s')} >= start_date: {$startDateCarbon->format('Y-m-d H:i:s')})";
                        }
                    }

                    if ($shouldReduce) {
                        if (!$dryRun) {
                            $sparepartLocked = Sparepart::lockForUpdate()->find($sparepart->id);
                            if (!$sparepartLocked) {
                                $details[] = "❌ Sparepart '{$sparepart->name}' (ID: {$sparepart->id}): Sparepart tidak ditemukan di database";
                                $errors++;
                                continue;
                            }
                            
                            $oldStock = $sparepartLocked->stock;
                            
                            if ($oldStock >= $needed) {
                                $sparepartLocked->decrement('stock', $needed);
                                // Refresh untuk mendapatkan stok terbaru
                                $sparepartLocked->refresh();
                                $newStock = $sparepartLocked->stock;
                                $details[] = "✅ Sparepart '{$sparepart->name}' (ID: {$sparepart->id}): {$oldStock} → {$newStock} (-{$needed}) [{$debugInfo}]";
                                $fixed++;
                                
                                // Log ke file untuk debugging
                                \Log::info("Fix Stock: Sparepart '{$sparepart->name}' (ID: {$sparepart->id}) dikurangi dari {$oldStock} menjadi {$newStock} (-{$needed})");
                            } else {
                                $details[] = "⚠️ Sparepart '{$sparepart->name}' (ID: {$sparepart->id}): Stok tidak cukup (stok: {$oldStock}, butuh: {$needed}) - mungkin sudah dikurangi sebelumnya";
                                // Tetap hitung sebagai fixed karena mungkin sudah benar
                                $fixed++;
                            }
                        } else {
                            $details[] = "🔄 [DRY RUN] Sparepart '{$sparepart->name}' (ID: {$sparepart->id}): Akan dikurangi {$needed} (stok saat ini: {$sparepart->stock}, {$debugInfo})";
                            $fixed++;
                        }
                    } else {
                        $details[] = "⏭️ Sparepart '{$sparepart->name}' (ID: {$sparepart->id}): Skip - {$skipReason} [{$debugInfo}]";
                    }
                }

                if (!$dryRun) {
                    DB::commit();
                }

                $totalFixed += $fixed;
                $results[] = [
                    'production_id' => $production->id,
                    'status' => 'success',
                    'fixed' => $fixed,
                    'details' => $details
                ];

            } catch (\Exception $e) {
                if (!$dryRun) {
                    DB::rollBack();
                }
                $results[] = [
                    'production_id' => $production->id,
                    'status' => 'error',
                    'message' => $e->getMessage()
                ];
                $totalErrors++;
            }
        }

        $message = $dryRun 
            ? "Simulasi: {$totalFixed} item akan diperbaiki, {$totalSkipped} produksi di-skip, {$totalErrors} error"
            : "Berhasil: {$totalFixed} item diperbaiki, {$totalSkipped} produksi di-skip, {$totalErrors} error";

        return back()->with('success', $message)->with('fix_stock_results', $results);
    }

    /**
     * Fix stock untuk satu produksi spesifik
     */
    public function fixStockSingle(Request $request, Production $production)
    {
        // Buat request baru dengan production_id
        $newRequest = new Request(['production_id' => $production->id]);
        if ($request->has('dry_run')) {
            $newRequest->merge(['dry_run' => true]);
        }
        
        return $this->fixStock($newRequest);
    }
}
