<?php

namespace App\Http\Controllers;

use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\Material;
use App\Models\Sparepart;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;

class PurchaseController extends Controller
{
    public function index(Request $request)
    {
        $search = trim($request->input('search'));
        $type = $request->input('type');

        $purchases = Purchase::with('items')
            ->when($search && $search !== '', function ($query) use ($search) {
                $query->where(function ($q) use ($search) {
                    $q->where('purchase_code', 'like', "%{$search}%")
                        ->orWhere('supplier_name', 'like', "%{$search}%")
                        ->orWhere('purchase_type', 'like', "%{$search}%")
                        ->orWhere('purchase_date', 'like', "%{$search}%")
                        ->orWhere('notes', 'like', "%{$search}%");
                    // Tambahkan pencarian ID jika search adalah numerik
                    if (is_numeric($search)) {
                        $q->orWhere('id', $search);
                    }
                });
            })
            ->when($type && $type !== '', function ($query) use ($type) {
                $query->where('purchase_type', $type);
            })
            ->latest('purchase_date')
            ->latest('id');
        
        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $purchases = $purchases->paginate($itemsPerPage)->withQueryString();

        return view('admin.transaksi.purchases.index', compact('purchases'));
    }

    public function create(Request $request)
    {
        // Handle dari item request
        $fromItemRequest = null;
        $prefillData = null;
        $bulkFromProduction = session('purchase_items_from_production');
        
        if ($request->has('from_item_request')) {
            $fromItemRequest = \App\Models\ProductionItemRequest::find($request->from_item_request);
            
            if ($fromItemRequest) {
                // Auto-approve item request jika belum approved
                if ($fromItemRequest->status === 'pending') {
                    $fromItemRequest->update([
                        'status' => 'approved',
                        'processed_by' => Auth::id(),
                        'processed_at' => now(),
                    ]);
                }
                
                // Siapkan data untuk prefill
                $prefillData = [
                    'item_type' => $fromItemRequest->item_type,
                    'item_name' => $fromItemRequest->item_name_display,
                    'quantity' => $fromItemRequest->quantity,
                    'unit' => $fromItemRequest->unit,
                    'estimated_price' => $fromItemRequest->estimated_price,
                    'material_id' => $fromItemRequest->material_id,
                    'sparepart_id' => $fromItemRequest->sparepart_id,
                    'item_request_id' => $fromItemRequest->id,
                ];
                
                $type = $fromItemRequest->item_type; // material atau sparepart
            } else {
                $type = $request->get('type', 'material');
            }
        } elseif ($request->has('from_production')) {
            // Handle dari produksi (Daftar Bahan & Sparepart)
            $prefillData = [
                'item_type' => $request->get('item_type', 'material'),
                'item_name' => $request->get('item_name', ''),
                'quantity' => $request->get('quantity', 1),
                'unit' => $request->get('unit', 'pcs'),
                'estimated_price' => $request->get('estimated_price', 0),
                'material_id' => $request->get('material_id'),
                'sparepart_id' => $request->get('sparepart_id'),
                'from_production' => $request->get('from_production'),
            ];
            
            $type = $request->get('item_type', 'material');
        } else {
            if ($bulkFromProduction && is_array($bulkFromProduction) && !empty($bulkFromProduction['items'])) {
                $items = collect($bulkFromProduction['items']);
                $types = $items->pluck('type')->unique()->values();
                if ($types->count() === 1 && in_array($types[0], ['material', 'sparepart'])) {
                    $type = $types[0];
                } else {
                    $type = $request->get('type', 'material');
                }
            } else {
                $type = $request->get('type', 'material'); // Default material
            }
        }
        
        $materials = Material::all();
        $spareparts = Sparepart::all();

        // PERBAIKAN: Tambahkan description dan perbaiki path gambar
        $materialsForJs = $materials->map(function ($i) {
            return [
                'id' => $i->id,
                'name' => $i->name,
                'unit' => $i->unit,
                'material_code' => $i->material_code,
                'category' => $i->category,
                'price' => $i->price,
                'description' => $i->description, // TAMBAHKAN INI
                'image' => $i->image ? asset('uploads/materials/' . $i->image) : '',
            ];
        })->toArray();

        // PERBAIKAN: Tambahkan description dan perbaiki path gambar
        $sparepartsForJs = $spareparts->map(function ($i) {
            return [
                'id' => $i->id,
                'name' => $i->name,
                'unit' => $i->unit,
                'part_code' => $i->part_code,
                'brand' => $i->brand,
                'type' => $i->type,
                'part_number' => $i->part_number,
                'price' => $i->price,
                'description' => $i->description, // TAMBAHKAN INI
                'image' => $i->image ? asset('uploads/spareparts/' . $i->image) : '',
            ];
        })->toArray();

        if ($bulkFromProduction) {
            session()->forget('purchase_items_from_production');
        }

        return view('admin.transaksi.purchases.create', compact('type', 'materialsForJs', 'sparepartsForJs', 'prefillData', 'fromItemRequest', 'bulkFromProduction'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'purchase_date' => 'required|date',
            'purchase_type' => 'required|in:material,sparepart',
            'supplier_name' => 'nullable|string|max:255',
            'notes' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.item_id' => 'nullable|integer',
            'items.*.item_name' => 'required|string|max:255',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.unit' => 'nullable|string|max:50',
            'items.*.image' => 'nullable|image|max:2048',
        ], [
            'purchase_date.required' => 'Tanggal pembelian wajib diisi.',
            'purchase_date.date' => 'Format tanggal tidak valid.',
            'purchase_type.required' => 'Tipe pembelian wajib dipilih.',
            'purchase_type.in' => 'Tipe pembelian harus Material atau Sparepart.',
            'items.required' => 'Minimal harus ada 1 item.',
            'items.min' => 'Minimal harus ada 1 item.',
            'items.*.item_name.required' => 'Nama item wajib diisi.',
            'items.*.quantity.required' => 'Quantity wajib diisi.',
            'items.*.quantity.min' => 'Quantity minimal 1.',
            'items.*.unit_price.required' => 'Harga satuan wajib diisi.',
            'items.*.unit_price.min' => 'Harga satuan tidak boleh negatif.',
        ]);

        DB::beginTransaction();
        try {
            $purchaseCode = 'PB-' . strtoupper(Str::random(6));
            $totalAmount = collect($request->items)->sum(fn($i) => ($i['quantity'] ?? 0) * ($i['unit_price'] ?? 0));

            $purchase = Purchase::create([
                'purchase_code' => $purchaseCode,
                'purchase_date' => $request->purchase_date,
                'supplier_name' => $request->supplier_name,
                'purchase_type' => $request->purchase_type,
                'total_amount' => $totalAmount,
                'notes' => $request->notes ?? null,
                'status' => 'approved', // Langsung approved, admin bisa beli tanpa persetujuan supervisor
                'approved_at' => now(),
                'approved_by' => auth()->id(), // Admin yang membuat purchase
            ]);

            foreach ($request->items as $item) {
                if (empty($item['item_name'])) continue;

                $itemId = !empty($item['item_id']) ? (int)$item['item_id'] : null;
                $itemName = $item['item_name'];
                $quantity = (int)$item['quantity']; // Pastikan integer
                $unitPrice = $item['unit_price'];
                $totalPrice = $quantity * $unitPrice;

                // PERBAIKAN: Handle upload gambar dengan lebih baik
                $imagePath = null;
                if (isset($item['image']) && $item['image'] instanceof \Illuminate\Http\UploadedFile) {
                    $folder = $request->purchase_type === 'material' ? 'materials' : 'spareparts';
                    $filename = Str::uuid() . '.' . $item['image']->getClientOriginalExtension();
                    $item['image']->storeAs($folder, $filename, 'uploads');
                    $imagePath = $filename;
                }

                // Jika item baru (item_id kosong atau 0)
                if (!$itemId) {
                    if ($request->purchase_type === 'material') {
                        $material = Material::create([
                            'material_code' => $item['material_code'] ?? 'MAT-' . strtoupper(Str::random(6)),
                            'name' => $itemName,
                            'unit' => $item['unit'] ?? 'pcs',
                            'price' => $unitPrice,
                            'stock' => $quantity, // Stock langsung ditambahkan saat pembelian
                            'category' => $item['category'] ?? null,
                            'supplier' => $request->supplier_name ?? null,
                            'image' => $imagePath,
                            'description' => $item['description'] ?? null,
                            'is_active' => isset($item['is_active']) ? true : false,
                        ]);
                        $itemId = $material->id;
                    } else {
                        $sparepart = Sparepart::create([
                            'part_code' => $item['part_code'] ?? 'PART-' . strtoupper(Str::random(6)),
                            'name' => $itemName,
                            'brand' => $item['brand'] ?? null,
                            'type' => $item['type'] ?? null,
                            'part_number' => $item['part_number'] ?? null,
                            'unit' => $item['unit'] ?? 'pcs',
                            'price' => $unitPrice,
                            'stock' => $quantity, // Stock langsung ditambahkan saat pembelian
                            'image' => $imagePath,
                            'description' => $item['description'] ?? null,
                            'is_active' => isset($item['is_active']) ? true : false,
                        ]);
                        $itemId = $sparepart->id;
                    }
                } else {
                    // Update item yang sudah ada dan tambahkan stock
                    if ($request->purchase_type === 'material') {
                        $material = Material::find($itemId);
                        if ($material) {
                            // PERBAIKAN: Hanya update gambar jika ada gambar baru
                            $updateData = [
                                'material_code' => $item['material_code'] ?? $material->material_code,
                                'name' => $itemName,
                                'unit' => $item['unit'] ?? $material->unit,
                                'price' => $unitPrice,
                                'category' => $item['category'] ?? $material->category,
                                'supplier' => $request->supplier_name ?? $material->supplier,
                                'description' => $item['description'] ?? $material->description,
                                'is_active' => isset($item['is_active']) ? true : $material->is_active,
                            ];

                            // PERBAIKAN: Update gambar hanya jika ada upload baru
                            if ($imagePath) {
                                // Hapus gambar lama jika ada
                                if ($material->image && Storage::disk('uploads')->exists('materials/' . $material->image)) {
                                    Storage::disk('uploads')->delete('materials/' . $material->image);
                                }
                                $updateData['image'] = $imagePath;
                            }

                            $material->update($updateData);
                            // Tambahkan stock langsung saat pembelian
                            $material->increment('stock', $quantity);
                        }
                    } else {
                        $sparepart = Sparepart::find($itemId);
                        if ($sparepart) {
                            // PERBAIKAN: Hanya update gambar jika ada gambar baru
                            $updateData = [
                                'part_code' => $item['part_code'] ?? $sparepart->part_code,
                                'name' => $itemName,
                                'brand' => $item['brand'] ?? $sparepart->brand,
                                'type' => $item['type'] ?? $sparepart->type,
                                'part_number' => $item['part_number'] ?? $sparepart->part_number,
                                'unit' => $item['unit'] ?? $sparepart->unit,
                                'price' => $unitPrice,
                                'description' => $item['description'] ?? $sparepart->description,
                                'is_active' => isset($item['is_active']) ? true : $sparepart->is_active,
                            ];

                            // PERBAIKAN: Update gambar hanya jika ada upload baru
                            if ($imagePath) {
                                // Hapus gambar lama jika ada
                                if ($sparepart->image && Storage::disk('uploads')->exists('spareparts/' . $sparepart->image)) {
                                    Storage::disk('uploads')->delete('spareparts/' . $sparepart->image);
                                }
                                $updateData['image'] = $imagePath;
                            }

                            $sparepart->update($updateData);
                            // Tambahkan stock langsung saat pembelian
                            $sparepart->increment('stock', $quantity);
                        }
                    }
                }

                // Simpan purchase item
                PurchaseItem::create([
                    'purchase_id' => $purchase->id,
                    'item_id' => $itemId,
                    'item_type' => $request->purchase_type,
                    'item_name' => $itemName,
                    'quantity' => $quantity,
                    'unit_price' => $unitPrice,
                    'total_price' => $totalPrice,
                    'unit' => $item['unit'] ?? null,
                ]);
            }

            // Update item request jika ada (cek dari_item_request atau item_request_id untuk backward compatibility)
            $itemRequestId = $request->input('from_item_request') ?? $request->input('item_request_id');
            if ($itemRequestId) {
                $itemRequest = \App\Models\ProductionItemRequest::find($itemRequestId);
                if ($itemRequest) {
                    // Jika item baru dibuat, update material_id atau sparepart_id di item request
                    $updateData = [
                        'status' => 'purchased',
                        'purchase_id' => $purchase->id,
                    ];
                    
                    // Cari item yang baru dibuat berdasarkan nama (exact match atau similar)
                    if ($itemRequest->is_new_item && $itemRequest->item_name) {
                        if ($itemRequest->item_type === 'material') {
                            // Coba exact match dulu
                            $foundMaterial = \App\Models\Material::where('name', $itemRequest->item_name)->first();
                            // Jika tidak ada, coba partial match
                            if (!$foundMaterial) {
                                $foundMaterial = \App\Models\Material::where('name', 'LIKE', '%' . $itemRequest->item_name . '%')
                                    ->orWhere('name', 'LIKE', '%' . trim($itemRequest->item_name) . '%')
                                    ->first();
                            }
                            if ($foundMaterial) {
                                $updateData['material_id'] = $foundMaterial->id;
                            }
                        } elseif ($itemRequest->item_type === 'sparepart') {
                            // Coba exact match dulu
                            $foundSparepart = \App\Models\Sparepart::where('name', $itemRequest->item_name)->first();
                            // Jika tidak ada, coba partial match
                            if (!$foundSparepart) {
                                $foundSparepart = \App\Models\Sparepart::where('name', 'LIKE', '%' . $itemRequest->item_name . '%')
                                    ->orWhere('name', 'LIKE', '%' . trim($itemRequest->item_name) . '%')
                                    ->first();
                            }
                            if ($foundSparepart) {
                                $updateData['sparepart_id'] = $foundSparepart->id;
                            }
                        }
                    }
                    
                    $itemRequest->update($updateData);
                }
            }

            DB::commit();
            
            $purchase->load('items');
            $fromProductionId = $request->input('from_production');
            $itemRequestId = $request->input('from_item_request') ?? $request->input('item_request_id');
            $kdProject = null;
            $namaProyek = null;

            if ($fromProductionId) {
                $production = \App\Models\Production::with('order')->find($fromProductionId);
                if ($production) {
                    $kdProject = $production->kd_project;
                    $project = null;

                    if ($kdProject) {
                        $project = \App\Models\Project::where('kode_bantu', $kdProject)->first();
                    }

                    if ((!$project || !$kdProject) && $production->order && method_exists($production->order, 'ensureProjectExists')) {
                        $project = $production->order->ensureProjectExists();
                        if ($project && !$kdProject) {
                            $kdProject = $project->kode_bantu;
                        }
                    }

                    if ($project && !$kdProject) {
                        $kdProject = $project->kode_bantu;
                    }

                    if ($project) {
                        $namaProyek = $project->nama_proyek;
                    }
                }
            } elseif ($itemRequestId) {
                $ir = \App\Models\ProductionItemRequest::find($itemRequestId);
                if ($ir && $ir->production) {
                    $kdProject = $ir->production->kd_project;
                    $project = \App\Models\Project::where('kode_bantu', $kdProject)->first();
                    $namaProyek = $project ? $project->nama_proyek : null;
                }
            } else {
                $linkedIr = \App\Models\ProductionItemRequest::where('purchase_id', $purchase->id)->first();
                if ($linkedIr && $linkedIr->production) {
                    $kdProject = $linkedIr->production->kd_project;
                    $project = \App\Models\Project::where('kode_bantu', $kdProject)->first();
                    $namaProyek = $project ? $project->nama_proyek : null;
                }
            }

            $purchaseCode = 'PUR-' . $purchase->id;
            $existsCredit = \App\Models\JournalEntry::where('kode', $purchaseCode)->where('kd_perk', '101-01')->exists();
            if (!$existsCredit) {
                foreach ($purchase->items as $pi) {
                    $itemCode = $purchaseCode . '-ITEM-' . $pi->id;
                    $existsItem = \App\Models\JournalEntry::where('kode', $itemCode)->exists();
                    if ($existsItem) { continue; }
                    \App\Models\JournalEntry::create([
                        'tanggal_transaksi' => $purchase->purchase_date ?? now(),
                        'bukti_transaksi' => $purchase->purchase_code,
                        'keterangan' => 'Pembelian ' . ($pi->item_name ?? '-') . ' ' . ($pi->quantity ?? 0) . ' ' . ($pi->unit ?? '') . ' untuk ' . ($namaProyek ?? ''),
                        'kd_perk' => '501-01',
                        'customer' => $purchase->supplier_name,
                        'kode' => $itemCode,
                        'nama_proyek' => $namaProyek,
                        'kd_project' => $kdProject,
                        'debit' => $pi->total_price,
                        'kredit' => null,
                        'created_by' => auth()->id(),
                    ]);
                }
                \App\Models\JournalEntry::create([
                    'tanggal_transaksi' => $purchase->purchase_date ?? now(),
                    'bukti_transaksi' => $purchase->purchase_code,
                    'keterangan' => 'Pembelian bahan/sparepart',
                    'kd_perk' => '101-01',
                    'customer' => $purchase->supplier_name,
                    'kode' => $purchaseCode,
                    'nama_proyek' => $namaProyek,
                    'kd_project' => $kdProject,
                    'debit' => null,
                    'kredit' => $purchase->total_amount,
                    'created_by' => auth()->id(),
                ]);
            }

            // Redirect berdasarkan apakah ada item request
            if ($itemRequestId) {
                $itemRequest = \App\Models\ProductionItemRequest::find($itemRequestId);
                if ($itemRequest) {
                    return redirect()->route('admin.productions.show', $itemRequest->production_id)
                        ->with('success', 'Pembelian berhasil disimpan dan stock telah diperbarui! Item request sudah ditandai sebagai purchased.');
                }
            }
            
            return redirect()->route('admin.transaksi.purchases.index')->with('success', 'Pembelian berhasil disimpan dan stock telah diperbarui!');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase Store Error: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage())->withInput();
        }
    }

    public function show($id)
    {
        $purchase = Purchase::with('items')->findOrFail($id);
        
        // Load item data (materials/spareparts) for display
        foreach ($purchase->items as $item) {
            if ($item->item_type === 'material' && $item->item_id) {
                $item->item_data = Material::find($item->item_id);
            } elseif ($item->item_type === 'sparepart' && $item->item_id) {
                $item->item_data = Sparepart::find($item->item_id);
            }
        }
        
        return view('admin.transaksi.purchases.show', compact('purchase'));
    }

    public function previewPdf($id)
    {
        $purchase = Purchase::with('items')->findOrFail($id);
        
        // Load item data (materials/spareparts) for display
        foreach ($purchase->items as $item) {
            if ($item->item_type === 'material' && $item->item_id) {
                $item->item_data = Material::find($item->item_id);
            } elseif ($item->item_type === 'sparepart' && $item->item_id) {
                $item->item_data = Sparepart::find($item->item_id);
            }
        }
        
        return view('admin.transaksi.purchases.pdf', compact('purchase'));
    }

    public function downloadPdf($id)
    {
        $purchase = Purchase::with('items')->findOrFail($id);
        
        // Load item data (materials/spareparts) for display
        foreach ($purchase->items as $item) {
            if ($item->item_type === 'material' && $item->item_id) {
                $item->item_data = Material::find($item->item_id);
            } elseif ($item->item_type === 'sparepart' && $item->item_id) {
                $item->item_data = Sparepart::find($item->item_id);
            }
        }
        
        $pdf = Pdf::loadView('admin.transaksi.purchases.pdf', compact('purchase'))
            ->setPaper('A4', 'portrait')
            ->setOptions([
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
                'defaultFont' => 'Arial',
                'chroot' => public_path(),
            ]);
        
        $filename = 'Bukti_Pembelian_' . $purchase->purchase_code . '_' . now()->format('Y-m-d') . '.pdf';
        
        return $pdf->download($filename);
    }

    public function edit($id)
    {
        $purchase = Purchase::with('items')->findOrFail($id);
        $materials = Material::all();
        $spareparts = Sparepart::all();

        // Prepare data for JavaScript
        $materialsForJs = $materials->map(function ($i) {
            return [
                'id' => $i->id,
                'name' => $i->name,
                'unit' => $i->unit,
                'material_code' => $i->material_code,
                'category' => $i->category,
                'price' => $i->price,
                'description' => $i->description,
                'image' => $i->image ? asset('uploads/materials/' . $i->image) : '',
            ];
        })->toArray();

        $sparepartsForJs = $spareparts->map(function ($i) {
            return [
                'id' => $i->id,
                'name' => $i->name,
                'unit' => $i->unit,
                'part_code' => $i->part_code,
                'brand' => $i->brand,
                'type' => $i->type,
                'part_number' => $i->part_number,
                'price' => $i->price,
                'description' => $i->description,
                'image' => $i->image ? asset('uploads/spareparts/' . $i->image) : '',
            ];
        })->toArray();

        return view('admin.transaksi.purchases.edit', compact('purchase', 'materialsForJs', 'sparepartsForJs'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'purchase_date' => 'required|date',
            'purchase_type' => 'required|in:material,sparepart',
            'supplier_name' => 'required|string|max:255',
            'notes' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.item_id' => 'nullable|integer',
            'items.*.item_name' => 'required|string|max:255',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.unit' => 'nullable|string|max:50',
            'items.*.image' => 'nullable|image|max:2048',
        ]);

        DB::beginTransaction();
        try {
            $purchase = Purchase::findOrFail($id);
            
            // Update purchase data
            $totalAmount = collect($request->items)->sum(fn($i) => ($i['quantity'] ?? 0) * ($i['unit_price'] ?? 0));
            
            $purchase->update([
                'purchase_date' => $request->purchase_date,
                'supplier_name' => $request->supplier_name,
                'purchase_type' => $request->purchase_type,
                'total_amount' => $totalAmount,
                'notes' => $request->notes ?? null,
            ]);

            // Delete existing items
            $purchase->items()->delete();

            // Create new items (same logic as store)
            foreach ($request->items as $item) {
                if (empty($item['item_name'])) continue;

                $itemId = !empty($item['item_id']) ? (int)$item['item_id'] : null;
                $itemName = $item['item_name'];
                $quantity = (int)$item['quantity']; // Pastikan integer
                $unitPrice = $item['unit_price'];
                $totalPrice = $quantity * $unitPrice;

                // Handle upload gambar
                $imagePath = null;
                if (isset($item['image']) && $item['image'] instanceof \Illuminate\Http\UploadedFile) {
                    $folder = $request->purchase_type === 'material' ? 'materials' : 'spareparts';
                    $filename = Str::uuid() . '.' . $item['image']->getClientOriginalExtension();
                    $item['image']->storeAs($folder, $filename, 'uploads');
                    $imagePath = $filename;
                }

                // Update or create item
                if (!$itemId) {
                    if ($request->purchase_type === 'material') {
                        $material = Material::create([
                            'material_code' => $item['material_code'] ?? 'MAT-' . strtoupper(Str::random(6)),
                            'name' => $itemName,
                            'unit' => $item['unit'] ?? 'pcs',
                            'price' => $unitPrice,
                            'stock' => 0, // Stock akan ditambahkan setelah approval supervisor
                            'category' => $item['category'] ?? null,
                            'supplier' => $request->supplier_name ?? null,
                            'image' => $imagePath,
                            'description' => $item['description'] ?? null,
                            'is_active' => isset($item['is_active']) ? true : false,
                        ]);
                        $itemId = $material->id;
                    } else {
                        $sparepart = Sparepart::create([
                            'part_code' => $item['part_code'] ?? 'PART-' . strtoupper(Str::random(6)),
                            'name' => $itemName,
                            'brand' => $item['brand'] ?? null,
                            'type' => $item['type'] ?? null,
                            'part_number' => $item['part_number'] ?? null,
                            'unit' => $item['unit'] ?? 'pcs',
                            'price' => $unitPrice,
                            'stock' => 0, // Stock akan ditambahkan setelah approval supervisor
                            'image' => $imagePath,
                            'description' => $item['description'] ?? null,
                            'is_active' => isset($item['is_active']) ? true : false,
                        ]);
                        $itemId = $sparepart->id;
                    }
                } else {
                    // Update existing item (TIDAK tambahkan stock di sini)
                    if ($request->purchase_type === 'material') {
                        $material = Material::find($itemId);
                        if ($material) {
                            $updateData = [
                                'material_code' => $item['material_code'] ?? $material->material_code,
                                'name' => $itemName,
                                'unit' => $item['unit'] ?? $material->unit,
                                'price' => $unitPrice,
                                'category' => $item['category'] ?? $material->category,
                                'supplier' => $request->supplier_name ?? $material->supplier,
                                'description' => $item['description'] ?? $material->description,
                                'is_active' => isset($item['is_active']) ? true : $material->is_active,
                                // Stock TIDAK diupdate di sini, akan ditambah setelah approval supervisor
                            ];

                            if ($imagePath) {
                                if ($material->image && Storage::disk('uploads')->exists('materials/' . $material->image)) {
                                    Storage::disk('uploads')->delete('materials/' . $material->image);
                                }
                                $updateData['image'] = $imagePath;
                            }

                            $material->update($updateData);
                            // Stock TIDAK ditambahkan di sini, akan ditambahkan setelah approval supervisor
                        }
                    } else {
                        $sparepart = Sparepart::find($itemId);
                        if ($sparepart) {
                            $updateData = [
                                'part_code' => $item['part_code'] ?? $sparepart->part_code,
                                'name' => $itemName,
                                'brand' => $item['brand'] ?? $sparepart->brand,
                                'type' => $item['type'] ?? $sparepart->type,
                                'part_number' => $item['part_number'] ?? $sparepart->part_number,
                                'unit' => $item['unit'] ?? $sparepart->unit,
                                'price' => $unitPrice,
                                'description' => $item['description'] ?? $sparepart->description,
                                'is_active' => isset($item['is_active']) ? true : $sparepart->is_active,
                            ];

                            if ($imagePath) {
                                if ($sparepart->image && Storage::disk('uploads')->exists('spareparts/' . $sparepart->image)) {
                                    Storage::disk('uploads')->delete('spareparts/' . $sparepart->image);
                                }
                                $updateData['image'] = $imagePath;
                            }

                            $sparepart->update($updateData);
                            // Stock TIDAK ditambahkan di sini, akan ditambahkan setelah approval supervisor
                        }
                    }
                }

                // Create purchase item
                PurchaseItem::create([
                    'purchase_id' => $purchase->id,
                    'item_id' => $itemId,
                    'item_type' => $request->purchase_type,
                    'item_name' => $itemName,
                    'quantity' => $quantity,
                    'unit_price' => $unitPrice,
                    'total_price' => $totalPrice,
                    'unit' => $item['unit'] ?? null,
                ]);
            }

            DB::commit();
            return redirect()->route('admin.transaksi.purchases.index')->with('success', 'Pembelian berhasil diperbarui!');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase Update Error: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage())->withInput();
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $purchase = Purchase::with('items')->findOrFail($id);
            
            // Delete related purchase items
            $purchase->items()->delete();
            
            // Delete purchase
        $purchase->delete();
            
            DB::commit();
            return redirect()->route('admin.transaksi.purchases.index')->with('success', 'Data pembelian berhasil dihapus.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase Delete Error: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }
}
