<?php

namespace App\Http\Controllers\Supervisor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Purchase;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PurchaseController extends Controller
{
    public function index()
    {
        $purchases = Purchase::with(['items.material', 'items.sparepart', 'supervisor'])
            ->where('status', 'pending')
            ->latest('created_at')
            ->paginate(20);
        
        return view('supervisor.purchases.index', compact('purchases'));
    }

    public function show(Purchase $purchase)
    {
        $purchase->load([
            'purchaseItems.material', 
            'purchaseItems.sparepart',
            'supervisor'
        ]);
        
        return view('supervisor.purchases.show', compact('purchase'));
    }

    public function approve(Request $request, Purchase $purchase)
    {
        $request->validate([
            'feedback' => 'nullable|string|max:1000',
        ]);

        DB::beginTransaction();
        try {
            // Update status purchase
            $purchase->update([
                'status' => 'approved',
                'approved_at' => now(),
                'approved_by' => Auth::id(),
                'supervisor_feedback' => $request->feedback,
            ]);

            // Update stock untuk setiap item di purchase
            foreach ($purchase->items as $purchaseItem) {
                $quantity = $purchaseItem->quantity;
                $itemId = $purchaseItem->item_id;
                
                // Gunakan item_type dari purchaseItem untuk menentukan material atau sparepart
                if ($purchaseItem->item_type === 'material') {
                    $material = \App\Models\Material::find($itemId);
                    if ($material) {
                        $material->increment('stock', $quantity);
                    }
                } elseif ($purchaseItem->item_type === 'sparepart') {
                    $sparepart = \App\Models\Sparepart::find($itemId);
                    if ($sparepart) {
                        $sparepart->increment('stock', $quantity);
                    }
                }
            }

            DB::commit();
            return back()->with('success', 'Purchase berhasil disetujui dan stock telah diperbarui.');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Purchase Approval Error: ' . $e->getMessage());
            return back()->with('error', 'Gagal menyetujui purchase: ' . $e->getMessage());
        }
    }

    public function reject(Request $request, Purchase $purchase)
    {
        $request->validate([
            'feedback' => 'required|string|max:1000',
        ]);

        $purchase->update([
            'status' => 'rejected',
            'supervisor_feedback' => $request->feedback,
        ]);

        return back()->with('success', 'Purchase ditolak.');
    }
}