<?php

namespace App\Http\Requests\Auth;

use Illuminate\Auth\Events\Lockout;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class LoginRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'email' => ['required', 'string', 'email:rfc,dns', 'max:255'], // Validasi email dengan DNS check
            'password' => ['required', 'string', 'max:255'], // Limit panjang password untuk mencegah DoS
        ];
    }

    /**
     * Get custom error messages for validation.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'email.required' => 'Alamat email wajib diisi.',
            'email.email' => 'Format email tidak valid. Pastikan email yang Anda masukkan benar.',
            'email.max' => 'Alamat email terlalu panjang. Maksimal 255 karakter.',
            'password.required' => 'Kata sandi wajib diisi.',
            'password.string' => 'Kata sandi harus berupa teks.',
            'password.max' => 'Kata sandi terlalu panjang. Maksimal 255 karakter.',
        ];
    }

    /**
     * Attempt to authenticate the request's credentials.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function authenticate(): void
    {
        $this->ensureIsNotRateLimited();

        // Sanitasi email input untuk mencegah injection
        $email = filter_var($this->string('email'), FILTER_SANITIZE_EMAIL);
        $password = $this->string('password');

        // Validasi email format tambahan
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            RateLimiter::hit($this->throttleKey());
            \Log::warning('Invalid email format attempted', [
                'email' => $email,
                'ip' => $this->ip(),
            ]);
            throw ValidationException::withMessages([
                'email' => 'Format email tidak valid. Pastikan email yang Anda masukkan benar.',
            ]);
        }

        // Attempt authentication (tidak cek user dulu untuk mencegah user enumeration)
        if (! Auth::attempt(['email' => $email, 'password' => $password], $this->boolean('remember'))) {
            RateLimiter::hit($this->throttleKey());

            // Log failed login attempt untuk audit dan security monitoring
            \Log::warning('Failed login attempt', [
                'email' => $email,
                'ip_address' => $this->ip(),
                'user_agent' => $this->userAgent(),
                'attempts' => RateLimiter::attempts($this->throttleKey()),
            ]);

            throw ValidationException::withMessages([
                'email' => 'Email atau kata sandi yang Anda masukkan salah. Silakan periksa kembali dan coba lagi.',
            ]);
        }

        // Clear rate limiter setelah login berhasil
        RateLimiter::clear($this->throttleKey());
    }

    /**
     * Ensure the login request is not rate limited.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function ensureIsNotRateLimited(): void
    {
        if (! RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            return;
        }

        event(new Lockout($this));

        $seconds = RateLimiter::availableIn($this->throttleKey());

        $minutes = ceil($seconds / 60);
        throw ValidationException::withMessages([
            'email' => "Terlalu banyak percobaan login yang gagal. Silakan coba lagi dalam {$minutes} menit.",
        ]);
    }

    /**
     * Get the rate limiting throttle key for the request.
     */
    public function throttleKey(): string
    {
        return Str::transliterate(Str::lower($this->string('email')).'|'.$this->ip());
    }
}
