<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Notification extends Model
{
    use HasFactory;

    protected $fillable = [
        'type',
        'subtype',
        'title',
        'message',
        'data',
        'user_id',
        'role',
        'is_read',
        'read_at',
        'email_sent',
        'email_sent_at',
        'related_order_id',
        'related_inquiry_id',
        'related_contact_id',
        'related_production_id',
    ];

    protected $casts = [
        'data' => 'array',
        'is_read' => 'boolean',
        'email_sent' => 'boolean',
        'read_at' => 'datetime',
        'email_sent_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function order()
    {
        return $this->belongsTo(Order::class, 'related_order_id');
    }

    public function inquiry()
    {
        return $this->belongsTo(ProductInquiry::class, 'related_inquiry_id');
    }

    public function contact()
    {
        return $this->belongsTo(Contact::class, 'related_contact_id');
    }

    public function production()
    {
        return $this->belongsTo(Production::class, 'related_production_id');
    }

    // Scopes
    public function scopeUnread($query)
    {
        return $query->where('is_read', false);
    }

    public function scopeRead($query)
    {
        return $query->where('is_read', true);
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function scopeForRole($query, $role)
    {
        return $query->where('role', $role)->whereNull('user_id');
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    // Methods
    public function markAsRead()
    {
        $this->update([
            'is_read' => true,
            'read_at' => now(),
        ]);
    }

    public function markEmailSent()
    {
        $this->update([
            'email_sent' => true,
            'email_sent_at' => now(),
        ]);
    }
}

