<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Models\User;
use App\Models\Product;
use App\Models\Production;
use App\Models\Project;

class Order extends Model
{
    use HasFactory;

    /**
     * Field yang bisa diisi mass-assignment
     */
    protected $fillable = [
        // Data pembeli
        'customer_name',
        'customer_email',
        'phone_number',
        'address',

        // Relasi
        'user_id',
        'product_id',

        // Data pembelian
        'quantity',          // jumlah unit yang dipesan
        'price',             // harga satuan produk
        'total_price',       // hasil perhitungan otomatis (quantity * price)

        // Status & pembayaran
        'payment_method',       // lunas, dp, transfer, tunai
        'dp_percentage',        // persentase DP (default 30%)
        'status',               // status pembayaran
        'order_status',         // status pesanan (menunggu, diterima, ditolak)
        'rejection_reason',      // alasan penolakan pesanan
        'rejected_by',           // ID admin yang menolak pesanan
        'rejected_at',           // waktu penolakan pesanan
        'order_type',           // jenis pesanan (beli_langsung, penawaran)
        'production_status',    // status produksi
        'is_read',              // sudah dibaca admin atau belum

        // Upload bukti pembayaran
        'proof_file',

        // Midtrans
        'midtrans_transaction_id',
        'midtrans_payment_type',
        'midtrans_va_number',
        'midtrans_status',

        // Handover / Serah Terima
        'handover_status', // pending, ready, delivered, confirmed
        'handover_at',
        'handover_by',
        'handover_notes',
        'handover_photos',
        'customer_confirmed_at',
        'customer_feedback',
        'customer_rating',
    ];

    /**
     * Casting tipe data
     */
    protected $casts = [
        'price' => 'decimal:2',
        'total_price' => 'decimal:2',
        'dp_percentage' => 'decimal:2',
        'is_read' => 'boolean',
        'handover_at' => 'datetime',
        'handover_photos' => 'array',
        'customer_confirmed_at' => 'datetime',
        'customer_rating' => 'integer',
        'rejected_at' => 'datetime',
    ];

    /**
     * Relasi ke user (pembeli)
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Relasi ke produk yang dibeli
     */
    public function product()
    {
        return $this->belongsTo(Product::class, 'product_id');
    }

    /**
     * Relasi ke penjualan produk
     */
    public function sale()
    {
        return $this->hasOne(Sale::class, 'order_id');
    }


    /**
     * Relasi ke produksi (jika sudah masuk tahap produksi)
     */
    public function production()
    {
        return $this->hasOne(Production::class);
    }

    /**
     * Relasi ke piutang (jika pembayaran DP)
     */
    public function receivable()
    {
        return $this->hasOne(Receivable::class);
    }

    /**
     * Relasi ke Product Inquiry (jika order dibuat dari inquiry)
     */
    public function inquiry()
    {
        return $this->hasOne(ProductInquiry::class, 'order_id');
    }

    /**
     * Akses format total harga
     */
    public function getFormattedTotalPriceAttribute()
    {
        return 'Rp ' . number_format($this->total_price, 0, ',', '.');
    }

    /**
     * Label status pesanan
     */
    public function getOrderStatusLabelAttribute()
    {
        return match ($this->order_status) {
            'menunggu'  => 'Menunggu Konfirmasi',
            'diterima'  => 'Diterima',
            'ditolak'   => 'Ditolak',
            default     => ucfirst($this->order_status),
        };
    }

    /**
     * Label status pembayaran
     */
    public function getStatusLabelAttribute()
    {
        return match ($this->status) {
            // 'menunggu'   => 'Menunggu Pembayaran',
            'menunggu_verifikasi' => 'Menunggu Verifikasi Admin',
            'diproses'   => 'Pembayaran Diproses',
            'selesai'    => 'Pembayaran Selesai',
            'gagal'      => 'Pembayaran Gagal',
            'dibatalkan' => 'Dibatalkan',
            default      => ucfirst($this->status),
        };
    }

    /**
     * Label status produksi
     */
    public function getProductionStatusLabelAttribute()
    {
        return match ($this->production_status) {
            'menunggu'     => 'Menunggu Produksi',
            'dalam_proses' => 'Sedang Diproduksi',
            'selesai'      => 'Produksi Selesai',
            'dibatalkan'   => 'Produksi Dibatalkan',
            default        => ucfirst($this->production_status),
        };
    }

    /**
     * Label status handover/serah terima
     */
    public function getHandoverStatusLabelAttribute()
    {
        return match ($this->handover_status) {
            'pending'   => 'Menunggu Serah Terima',
            'ready'     => 'Siap Diserahkan',
            'delivered' => 'Sudah Diserahkan',
            'confirmed' => 'Dikonfirmasi Customer',
            default     => ucfirst($this->handover_status ?? 'Belum Siap'),
        };
    }

    /**
     * Relasi ke user yang melakukan handover
     */
    public function handoverBy()
    {
        return $this->belongsTo(User::class, 'handover_by');
    }

    /**
     * Relasi ke admin yang menolak pesanan
     */
    public function rejectedBy()
    {
        return $this->belongsTo(User::class, 'rejected_by');
    }

    /**
     * Check if order is ready for handover
     */
    public function isReadyForHandover()
    {
        // Order siap untuk handover jika:
        // 1. Ada production dan status produksi = selesai, ATAU
        // 2. Ada production dengan completion_status = approved
        if (!$this->production) {
            return false;
        }

        return $this->production->status === 'selesai' ||
            $this->production->completion_status === 'approved';
    }

    /**
     * Check if customer can confirm
     */
    public function canCustomerConfirm()
    {
        return $this->handover_status === 'delivered' &&
            is_null($this->customer_confirmed_at);
    }

    /**
     * Label jenis pesanan
     */
    public function getOrderTypeLabelAttribute()
    {
        return match ($this->order_type) {
            'penawaran' => 'Penawaran',
            default => 'Beli Langsung',
        };
    }

    /**
     * Scope filter berdasarkan status pesanan
     */
    public function scopeByOrderStatus($query, $status)
    {
        return $query->where('order_status', $status);
    }

    /**
     * Scope filter berdasarkan status pembayaran
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope filter berdasarkan status produksi
     */
    public function scopeByProductionStatus($query, $status)
    {
        return $query->where('production_status', $status);
    }

    /**
     * Custom route model binding untuk user orders
     * Memastikan hanya order milik user yang bisa diakses
     */
    public function resolveRouteBinding($value, $field = null)
    {
        // Jika field tidak ditentukan, gunakan 'id'
        $field = $field ?: $this->getRouteKeyName();

        // Cari order berdasarkan field
        $order = $this->where($field, $value)->first();

        // Jika order tidak ditemukan, return null (akan trigger 404)
        if (!$order) {
            return null;
        }

        // Untuk route user orders, pastikan order milik user yang login
        // Cek apakah request ke route user orders
        try {
            $route = request()->route();
            $routeName = $route ? $route->getName() : null;
            $routeUri = $route ? $route->uri() : null;

            // Cek berdasarkan route name atau URI pattern
            $isUserOrderRoute = false;
            if ($routeName && in_array($routeName, ['my.orders.show', 'orders.confirm.form', 'orders.confirm', 'orders.success', 'orders.proof'])) {
                $isUserOrderRoute = true;
            } elseif ($routeUri && preg_match('#^orders/\{order\}/(confirm|success|proof)#', $routeUri)) {
                // Fallback: cek berdasarkan URI pattern jika route name tidak tersedia
                $isUserOrderRoute = true;
            }

            if ($isUserOrderRoute) {
                // Untuk route orders.proof, kita skip authorization check di route model binding
                // karena authorization akan di-handle di controller method showProof()
                // Ini untuk menghindari 404 jika ada masalah dengan route name resolution
                if ($routeName === 'orders.proof') {
                    // Skip authorization check di route model binding untuk orders.proof
                    // Authorization akan di-handle di controller
                    return $order;
                }

                if (\Illuminate\Support\Facades\Auth::check()) {
                    $user = \Illuminate\Support\Facades\Auth::user();

                    // Jika order punya user_id, harus cocok
                    if ($order->user_id !== null) {
                        if ((int)$order->user_id !== (int)$user->id) {
                            return null; // Return null akan trigger 404
                        }
                    } else {
                        // Jika user_id null, cek berdasarkan email (case-insensitive untuk production)
                        if (strtolower(trim($order->customer_email)) !== strtolower(trim($user->email))) {
                            return null; // Return null akan trigger 404
                        }
                    }
                } else {
                    // Jika route memerlukan auth tapi user tidak login, return null
                    return null;
                }
            }
        } catch (\Exception $e) {
            // Jika ada error saat cek route, skip authorization check
            // Ini untuk mencegah error di production jika ada masalah dengan route
            // Log error untuk debugging
            \Illuminate\Support\Facades\Log::warning('Error in Order route model binding: ' . $e->getMessage());
        }

        return $order;
    }

    /**
     * Check if payment is verified based on payment method
     * 
     * Production hanya bisa dibuat jika payment sudah terverifikasi:
     * - DP: minimal payment_status = 'diproses' (DP sudah dibayar)
     * - Transfer: payment_status = 'selesai' (Pembayaran lunas)
     * - Midtrans: payment_status = 'selesai' (Settlement)
     * 
     * @return bool
     */
    public function isPaymentVerified(): bool
    {
        // Untuk DP: minimal DP sudah dibayar (status = 'diproses')
        if ($this->payment_method === 'dp') {
            return in_array($this->status, ['diproses', 'selesai']);
        }

        // Untuk Lunas (transfer, midtrans, lunas): harus sudah selesai
        if (in_array($this->payment_method, ['transfer', 'midtrans', 'lunas'])) {
            return $this->status === 'selesai';
        }

        // Default: tidak terverifikasi
        return false;
    }

    /**
     * Ensure production exists if order is accepted AND payment is verified
     * 
     * Production hanya dibuat jika:
     * 1. order_status === 'diterima'
     * 2. payment_status sudah TERVERIFIKASI (sesuai payment method)
     * 
     * @return \App\Models\Production|null
     */
    public function ensureProductionExists()
    {
        // 1. Cek apakah order sudah diterima
        if ($this->order_status !== 'diterima') {
            return null;
        }

        // 2. Cek apakah payment sudah terverifikasi
        if (!$this->isPaymentVerified()) {
            return null;
        }

        $project = $this->ensureProjectExists();
        $kdProject = $project ? $project->kode_bantu : null;

        // 3. Cek apakah production sudah ada (gunakan firstOrCreate untuk prevent duplicate)
        $production = Production::firstOrCreate(
            ['order_id' => $this->id],
            [
                'product_id' => $this->product_id,
                'kd_project' => $kdProject,
                'quantity' => $this->quantity,
                'start_date' => now(),
                'status' => 'menunggu',
                'total_material_cost' => 0,
                'total_sparepart_cost' => 0,
                'total_production_cost' => 0,
            ]
        );

        if (!$production->kd_project && $kdProject) {
            $production->updateQuietly(['kd_project' => $kdProject]);
        }

        // Update order production_status (hanya jika production baru dibuat)
        if ($production->wasRecentlyCreated) {
            $this->updateQuietly([
                'production_status' => 'menunggu',
            ]);
        }

        return $production;
    }

    public function ensureProjectExists()
    {
        if ($this->order_status !== 'diterima') {
            return null;
        }
        if (!$this->product) {
            return null;
        }

        $productTitle = trim($this->product->product_title);
        if ($productTitle === '') {
            return null;
        }

        $project = Project::whereRaw('LOWER(nama_proyek) = ?', [strtolower($productTitle)])->first();
        if (!$project) {
            $kode = Project::generateNextCode();
            $project = Project::create([
                'kode_bantu' => $kode,
                'nama_proyek' => $productTitle,
                'tabel_bantuan' => $kode,
                'nilai_kontrak' => 0,
                'status' => 'aktif',
                'is_active' => true,
                'sort_order' => 0,
            ]);
        }

        $totalKontrak = static::where('product_id', $this->product_id)
            ->where('order_status', 'diterima')
            ->sum('total_price');

        $project->update([
            'nilai_kontrak' => $totalKontrak,
        ]);

        return $project;
    }

    /**
     * Event otomatis:
     * - Jika pembayaran DP → buat catatan piutang
     * - Saat pembayaran selesai → aktifkan produksi
     */
    protected static function booted()
    {
        // Saat order dibuat → buat piutang jika pembayaran DP
        static::created(function ($order) {
            // Jika pembayaran DP, buat catatan piutang
            if ($order->payment_method === 'dp') {
                $receivable = \App\Models\Receivable::create([
                    'order_id' => $order->id,
                    'user_id' => $order->user_id,
                    'receivable_number' => \App\Models\Receivable::generateReceivableNumber(),
                    'total_amount' => $order->total_price,
                    'paid_amount' => 0,
                    'remaining_amount' => $order->total_price,
                    'payment_status' => 'pending',
                    'due_date' => now()->addDays(30), // Jatuh tempo 30 hari
                    'notes' => 'Pembayaran DP - Sisa pembayaran saat mesin selesai',
                    'status' => 'active'
                ]);

                // Kirim notifikasi
                try {
                    $notificationService = new \App\Services\NotificationService();
                    $notificationService->notifyReceivableCreated($receivable);
                } catch (\Exception $e) {
                    \Log::error('Failed to send receivable created notification: ' . $e->getMessage());
                }
            }
        });

        // CATATAN: Production status TIDAK bergantung pada payment status
        // Payment status 'selesai' tidak langsung mengubah production status ke 'dalam_proses'
        // Production status hanya berubah melalui workflow produksi:
        // - Teknisi mulai produksi (startProduction) → status = 'dalam_proses'
        // - Supervisor approve completion → status = 'selesai'

        // Saat order_status berubah ke 'diterima' → cek apakah production perlu dibuat
        static::updated(function ($order) {
            // Prevent duplicate production dengan cek apakah production sudah ada
            // Hanya panggil ensureProductionExists() jika production belum ada
            if ($order->order_status === 'diterima' && !$order->production) {
                // Jika order_status berubah ke 'diterima', cek apakah production perlu dibuat
                if ($order->wasChanged('order_status')) {
                    $order->ensureProductionExists();
                    $order->ensureProjectExists();
                }
                // Jika payment_status berubah dan order sudah diterima, cek apakah production perlu dibuat
                elseif ($order->wasChanged('status')) {
                    $order->ensureProductionExists();
                    $order->ensureProjectExists();
                }
            }

            if ($order->wasChanged('product_id') && $order->order_status === 'diterima') {
                $order->ensureProjectExists();
            }

            // Untuk DP: update piutang - LUNAS jika pembayaran selesai
            if ($order->wasChanged('status') && $order->status === 'selesai' && $order->payment_method === 'dp' && $order->receivable) {
                $receivable = $order->receivable;

                // Jika pembayaran selesai, set sebagai lunas
                if ($receivable->payment_status === 'pending' || $receivable->payment_status === 'partial') {
                    // Update paid_amount terlebih dahulu
                    $receivable->update([
                        'paid_amount' => $receivable->total_amount,
                    ]);

                    // Trigger update status (menggunakan logic di method)
                    $receivable->updatePaymentStatus();
                }
            }

            if ($order->wasChanged('status') && in_array($order->status, ['diproses', 'selesai']) && $order->payment_method === 'dp') {
                $project = $order->ensureProjectExists();
                $kdProject = $project ? $project->kode_bantu : null;
                $namaProyek = $project ? $project->nama_proyek : null;
                $dpAmount = (($order->total_price ?? 0) * (float)($order->dp_percentage ?? 0)) / 100;
                if ($dpAmount > 0) {
                    $kode = 'ORD-' . $order->id . '-DP';
                    $exists = \App\Models\JournalEntry::where('kode', $kode)->exists();
                    if (!$exists) {
                        \App\Models\JournalEntry::create([
                            'tanggal_transaksi' => now(),
                            'bukti_transaksi' => null,
                            'keterangan' => 'DP ' . ($namaProyek ?? ($order->product->product_title ?? '')),
                            'kd_perk' => '101-01',
                            'customer' => $order->customer_name,
                            'kode' => $kode,
                            'nama_proyek' => $namaProyek,
                            'kd_project' => $kdProject,
                            'debit' => $dpAmount,
                            'kredit' => null,
                            'created_by' => auth()->id(),
                        ]);
                        \App\Models\JournalEntry::create([
                            'tanggal_transaksi' => now(),
                            'bukti_transaksi' => null,
                            'keterangan' => 'DP ' . ($namaProyek ?? ($order->product->product_title ?? '')),
                            'kd_perk' => '201-02',
                            'customer' => $order->customer_name,
                            'kode' => $kode,
                            'nama_proyek' => $namaProyek,
                            'kd_project' => $kdProject,
                            'debit' => null,
                            'kredit' => $dpAmount,
                            'created_by' => auth()->id(),
                        ]);
                    }
                }
            }
        });
    }
}
