<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Sale extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'product_id',
        'user_id',
        'quantity',
        'unit_price',
        'total_price',
        'sale_date',
        'payment_method',
        'notes',
        'status',
    ];

    protected $casts = [
        'sale_date' => 'date',
        'unit_price' => 'decimal:2',
        'total_price' => 'decimal:2',
    ];

    /*
    |--------------------------------------------------------------------------
    | RELATIONS
    |--------------------------------------------------------------------------
    */
    public function order()
    {
        return $this->belongsTo(Order::class, 'order_id');
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /*
    |--------------------------------------------------------------------------
    | EVENT HOOKS
    |--------------------------------------------------------------------------
    */
    protected static function booted()
    {
        static::creating(function ($sale) {
            // Hitung total harga otomatis
            if (empty($sale->total_price)) {
                $sale->total_price = $sale->quantity * $sale->unit_price;
            }
        });
    }

    /*
    |--------------------------------------------------------------------------
    | ACCESSORS
    |--------------------------------------------------------------------------
    */
    public function getFormattedTotalPriceAttribute()
    {
        return 'Rp ' . number_format($this->total_price, 0, ',', '.');
    }

    public function getStatusLabelAttribute()
    {
        return match ($this->status) {
            'selesai' => 'Penjualan Selesai',
            'dibatalkan' => 'Penjualan Dibatalkan',
            default => ucfirst($this->status),
        };
    }
}
