<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'photo',
        'user_type',
        'is_active',
        'last_login_at',
        'must_change_password',
        'phone',
        'address',
        'postal_code',
    ];

    /**
     * The attributes that should be guarded from mass assignment.
     * user_type dan is_active hanya bisa diubah oleh admin melalui controller khusus
     *
     * @var array<string>
     */
    protected $guarded = [];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'last_login_at' => 'datetime',
            'is_active' => 'boolean',
            'must_change_password' => 'boolean',
            'password' => 'hashed',
        ];
    }

    public function orders()
    {
        return $this->hasMany(Order::class, 'user_id');
    }

    // relasi ke penjualan
    public function sales()
    {
        return $this->hasMany(Sale::class);
    }

    // relasi ke produksi (untuk teknisi)
    public function productions()
    {
        return $this->hasMany(Production::class, 'teknisi_id');
    }

    /**
     * Helper methods untuk role checking
     */
    public function isAdmin()
    {
        return $this->user_type === 'admin';
    }

    public function isSupervisor()
    {
        return $this->user_type === 'supervisor';
    }

    public function isTeknisi()
    {
        return $this->user_type === 'teknisi';
    }

    public function isUser()
    {
        return $this->user_type === 'user';
    }

    public function hasRole($role)
    {
        return $this->user_type === $role;
    }

    /**
     * Get the URL of the user's photo
     */
    public function getPhotoUrlAttribute()
    {
        if ($this->photo && \Storage::disk('uploads')->exists('profiles/' . $this->photo)) {
            return asset('uploads/profiles/' . $this->photo);
        }
        return null;
    }

    /**
     * Scope untuk user aktif
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope untuk user nonaktif
     */
    public function scopeInactive($query)
    {
        return $query->where('is_active', false);
    }

    /**
     * Scope untuk user yang belum pernah login
     */
    public function scopeNeverLoggedIn($query)
    {
        return $query->whereNull('last_login_at');
    }

    /**
     * Cek apakah user aktif
     */
    public function isActive(): bool
    {
        return $this->is_active === true;
    }

    /**
     * Update last login timestamp
     */
    public function updateLastLogin()
    {
        $this->update(['last_login_at' => now()]);
    }
}
