/**
 * User Notification System
 * Real-time notification system for user dashboard
 */
class UserNotificationSystem {
    constructor() {
        this.pollingInterval = 30000; // 30 seconds
        this.isPolling = false;
        this.lastUpdate = null;
        this.notificationPanel = null;
        this.init();
    }

    init() {
        console.log('[UserNotificationSystem] Initializing...');
        
        // Skip jika di halaman profile
        if (window.location.pathname === '/profile' || window.location.pathname === '/profile/') {
            console.log('[UserNotificationSystem] Skipping initialization on profile page');
            return;
        }
        
        // Clear old format viewed notifications untuk migrasi ke format baru
        this.clearOldViewedNotifications();
        
        this.createNotificationPanel();
        this.startPolling();
        this.bindEvents();
        
        console.log('[UserNotificationSystem] Initialization complete');
    }

    createNotificationPanel() {
        // Create notification panel HTML
        const panelHTML = `
            <div id="user-notification-panel" class="fixed right-4 z-50 hidden" style="top: 119px;">
                <div class="bg-white rounded-xl shadow-2xl border border-gray-200 max-w-sm w-full max-h-[80vh] flex flex-col">
                    <div class="p-4 border-b border-gray-200 flex-shrink-0">
                        <div class="flex items-center justify-between">
                            <h3 class="text-lg font-semibold text-gray-900">Notifikasi</h3>
                            <div class="flex items-center space-x-2">
                                <button onclick="if(window.userNotificationSystem){window.userNotificationSystem.markAllAsRead();}" 
                                    class="text-xs bg-blue-600 hover:bg-blue-700 text-white px-3 py-1.5 rounded transition-colors">
                                    Tandai Semua Dibaca
                                </button>
                                <button id="close-notification-panel" 
                                        class="text-gray-400 hover:text-gray-600 transition-colors p-1 rounded hover:bg-gray-100"
                                    title="Tutup">
                                    <i class="fas fa-times text-xl"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    <div id="notification-content" class="flex-1 overflow-y-auto p-4" style="max-height: calc(80vh - 120px);">
                        <div class="text-center py-8">
                            <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
                            <p class="text-gray-500 mt-2">Memuat notifikasi...</p>
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Add to body
        document.body.insertAdjacentHTML('beforeend', panelHTML);
        this.notificationPanel = document.getElementById('user-notification-panel');
    }

    bindEvents() {
        // Close panel event - check for the button or icon inside
        document.addEventListener('click', (e) => {
            // Check if clicking the close button or its icon
            if (e.target.id === 'close-notification-panel' || 
                e.target.closest('#close-notification-panel') ||
                e.target.classList.contains('fa-times') && e.target.closest('#user-notification-panel')) {
                this.hideNotificationPanel();
            }
        });

        // Click outside to close
        document.addEventListener('click', (e) => {
            if (this.notificationPanel && !this.notificationPanel.contains(e.target) && 
                !e.target.closest('.notification-bell')) {
                this.hideNotificationPanel();
            }
        });
    }
    
    /**
     * Bind click events untuk notification items
     */
    bindNotificationClicks() {
        const notificationItems = document.querySelectorAll('.notification-item');
        notificationItems.forEach(item => {
            // Remove existing listeners dengan clone
            const newItem = item.cloneNode(true);
            item.parentNode.replaceChild(newItem, item);
            
            // Add new click listener
            newItem.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                
                const type = newItem.dataset.type;
                const id = newItem.dataset.id;
                const notificationId = newItem.dataset.notificationId || newItem.getAttribute('data-notification-id') || null;
                let url = newItem.dataset.url;
                
                // Jika tidak ada data-url, coba buat dari type dan id
                if (!url) {
                    if (type === 'inquiry') {
                        url = `/user/inquiries/${id}`;
                    } else if (type === 'contact') {
                        url = `/user/contacts/${id}`;
                    } else if (type === 'order') {
                        url = `/my-orders/${id}`;
                    }
                }
                
                // Get inquiry data dari data attributes (hanya untuk inquiry)
                let inquiryData = null;
                let orderData = null;
                if (type === 'inquiry') {
                    // Note: dataset mengkonversi kebab-case ke camelCase
                    inquiryData = {
                        status: newItem.dataset.status || 'pending',
                        admin_replied_at: newItem.dataset.adminRepliedAt || newItem.getAttribute('data-admin-replied-at') || '',
                        converted_to_order_at: newItem.dataset.convertedToOrderAt || newItem.getAttribute('data-converted-to-order-at') || ''
                    };
                } else if (type === 'order') {
                    orderData = {
                        subtype: newItem.dataset.subtype || 'default'
                    };
                }
                
                if (type && id && url && window.userNotificationSystem) {
                    console.log('[UserNotificationSystem] Clicking notification:', { type, id, notificationId, url, inquiryData, orderData });
                    window.userNotificationSystem.handleNotificationClick(type, id, url, inquiryData, orderData, notificationId);
                } else {
                    console.error('[UserNotificationSystem] Missing data for click:', { type, id, url });
                }
            });
        });
    }

    startPolling() {
        // Skip jika di halaman profile
        if (window.location.pathname === '/profile' || window.location.pathname === '/profile/') {
            console.log('[UserNotificationSystem] Skipping polling on profile page');
            return;
        }
        
        if (this.isPolling) {
            console.log('[UserNotificationSystem] Polling already started');
            return;
        }
        
        console.log('[UserNotificationSystem] Starting polling...');
        this.isPolling = true;
        this.fetchNotifications();
        
        setInterval(() => {
            // Skip jika di halaman profile
            if (window.location.pathname === '/profile' || window.location.pathname === '/profile/') {
                return;
            }
            this.fetchNotifications();
        }, this.pollingInterval);
    }

    async fetchNotifications() {
        try {
            const csrfToken = document.querySelector('meta[name="csrf-token"]');
            if (!csrfToken) {
                console.error('[UserNotificationSystem] CSRF token not found');
                return;
            }
            
            // Gunakan route yang benar untuk user notifications
            console.log('[UserNotificationSystem] Fetching notifications from /user/notifications');
            const response = await fetch('/user/notifications', {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': csrfToken.getAttribute('content')
                },
                credentials: 'same-origin' // Include cookies for authentication
            });

            console.log('[UserNotificationSystem] Response status:', response.status, response.statusText);

            if (!response.ok) {
                const errorText = await response.text();
                console.error('[UserNotificationSystem] Response error:', errorText);
                throw new Error(`Network response was not ok: ${response.status} ${response.statusText}`);
            }

            const data = await response.json();
            
            // Debug logging
            console.log('[UserNotificationSystem] ===== RESPONSE RECEIVED =====');
            console.log('[UserNotificationSystem] Success:', data.success);
            console.log('[UserNotificationSystem] Unread counts:', data.data?.unread_counts);
            console.log('[UserNotificationSystem] Recent notifications count:', data.data?.recent_items?.notifications?.length || 0);
            console.log('[UserNotificationSystem] Recent notifications:', data.data?.recent_items?.notifications || []);
            
            if (data.success && data.data) {
                // Validate data structure
                if (!data.data.recent_items) {
                    console.error('[UserNotificationSystem] Missing recent_items in response');
                    data.data.recent_items = { notifications: [] };
                }
                if (!Array.isArray(data.data.recent_items.notifications)) {
                    console.error('[UserNotificationSystem] recent_items.notifications is not an array');
                    data.data.recent_items.notifications = [];
                }
                
                // Convert notifications array to inquiries, contacts, and orders format for compatibility
                const notifications = data.data.recent_items.notifications || [];
                
                // Process order notifications
                const orderNotifications = notifications.filter(n => n.type === 'order');
                
                // Process inquiry notifications
                const inquiries = notifications.filter(n => n.type === 'inquiry').map(n => {
                    // Convert notification format to inquiry format
                    // Handle different inquiry subtypes
                    let status = n.data?.status || 'pending';
                    if (n.subtype === 'inquiry_created') {
                        status = 'pending'; // New inquiry
                    } else if (n.subtype === 'inquiry_status_changed') {
                        status = n.data?.new_status || n.data?.status || 'pending';
                    } else if (n.subtype === 'final_offer_sent') {
                        status = 'approved'; // Final offer - harus ditampilkan dengan jelas
                    } else if (n.subtype === 'inquiry_converted_to_order') {
                        status = 'converted_to_order'; // Converted to order
                    }
                    
                    // Untuk final offer, pastikan admin_replied_at ada
                    let adminRepliedAt = n.data?.admin_replied_at || null;
                    if (n.subtype === 'final_offer_sent' && !adminRepliedAt) {
                        // Jika tidak ada admin_replied_at, gunakan created_at dari notifikasi
                        adminRepliedAt = n.created_at;
                    }
                    
                    return {
                        id: n.related_inquiry_id || n.data?.inquiry_id,
                        notification_id: n.id, // Add notification ID for mark as read
                        name: n.inquiry?.name || n.data?.name || 'User',
                        status: status,
                        product: n.inquiry?.product_title ? { product_title: n.inquiry.product_title } : (n.data?.product_title ? { product_title: n.data.product_title } : null),
                        admin_reply: n.data?.admin_reply || null,
                        admin_replied_at: adminRepliedAt,
                        final_offer_price: n.data?.final_offer_price || n.data?.final_price || null,
                        min_price: n.data?.min_price || null,
                        max_price: n.data?.max_price || null,
                        converted_to_order_at: n.data?.converted_to_order_at || null,
                        order_id: n.data?.order_id || n.related_order_id || null,
                        tracking_code: n.data?.tracking_code || null,
                        validity_until: n.data?.validity_until || null,
                        lead_time_days: n.data?.lead_time_days || null,
                        created_at: n.created_at,
                        updated_at: n.created_at,
                        is_read: n.is_read,
                        subtype: n.subtype // Keep subtype for reference
                    };
                }).filter(i => i.id); // Filter out invalid inquiries
                
                const contacts = notifications.filter(n => n.type === 'contact').map(n => {
                    // Convert notification format to contact format
                    return {
                        id: n.related_contact_id || n.data?.contact_id,
                        notification_id: n.id, // Add notification ID for mark as read
                        nama: n.contact?.nama || n.data?.nama || 'User',
                        pesan: n.message || n.data?.pesan || '',
                        admin_reply: n.data?.admin_reply || null,
                        admin_replied_at: n.data?.admin_replied_at || null,
                        created_at: n.created_at
                    };
                }).filter(c => c.id); // Filter out invalid contacts
                
                // Convert to old format for compatibility
                const formattedData = {
                    ...data.data,
                    recent_items: {
                        inquiries: inquiries,
                        contacts: contacts,
                        notifications: notifications // Keep original for reference
                    }
                };
                
                this.updateNotifications(formattedData);
                // Pass recent items to updateBadges for accurate counting
                this.updateBadges(data.data.unread_counts, formattedData.recent_items);
            } else {
                console.error('[UserNotificationSystem] API returned error:', data);
            }
        } catch (error) {
            console.error('[UserNotificationSystem] Error fetching notifications:', error);
        }
    }

    updateNotifications(data) {
        const content = document.getElementById('notification-content');
        
        if (!content) return;

        const { unread_counts, recent_items } = data;
        
        // Process order notifications from notifications array
        const orderNotifications = recent_items.notifications?.filter(n => n.type === 'order') || [];
        const filteredOrderNotifications = orderNotifications.filter(notif => {
            const orderId = notif.related_order_id || notif.data?.order_id;
            if (!orderId) return false;
            
            // Check if this order notification has been viewed
            const viewedNotifications = this.getViewedNotifications();
            const notificationKey = `order_${orderId}_${notif.subtype || 'default'}`;
            return !viewedNotifications.orders?.has(notificationKey);
        });
        
        // Filter out viewed notifications
        // IMPORTANT: Gunakan kombinasi ID + status untuk tracking
        // Karena inquiry yang sama bisa punya notifikasi berbeda (pending -> approved -> converted_to_order)
        const viewedNotifications = this.getViewedNotifications();
        const filteredInquiries = recent_items.inquiries.filter(inquiry => {
            const inquiryId = parseInt(inquiry.id);
            const status = inquiry.status || 'pending';
            
            // Buat key unik: ID + status + timestamp (untuk final offer dan converted to order)
            let notificationKey = `${inquiryId}_${status}`;
            
            // Untuk final offer (approved), tambahkan admin_replied_at
            if (status === 'approved' && inquiry.admin_replied_at) {
                const repliedAt = new Date(inquiry.admin_replied_at).getTime();
                notificationKey = `${inquiryId}_approved_${repliedAt}`;
            }
            
            // Untuk converted to order, tambahkan converted_to_order_at
            if (status === 'converted_to_order' && inquiry.converted_to_order_at) {
                const convertedAt = new Date(inquiry.converted_to_order_at).getTime();
                notificationKey = `${inquiryId}_converted_${convertedAt}`;
            }
            
            const isViewed = viewedNotifications.inquiries.has(notificationKey);
            if (isViewed) {
                console.log(`Inquiry #${inquiryId} (${status}) already viewed with key: ${notificationKey}`);
            }
            return !isViewed;
        });
        const filteredContacts = recent_items.contacts.filter(contact => 
            !viewedNotifications.contacts.has(parseInt(contact.id))
        );
        
        const totalUnread = filteredInquiries.length + filteredContacts.length + filteredOrderNotifications.length;
        
        if (totalUnread === 0) {
            content.innerHTML = `
                <div class="text-center py-8">
                    <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <i class="fas fa-bell-slash text-2xl text-gray-400"></i>
                    </div>
                    <p class="text-gray-500">Tidak ada notifikasi baru</p>
                </div>
            `;
            return;
        }

        let html = '<div class="space-y-3">';
        
        // Recent order notifications (filtered)
        if (filteredOrderNotifications.length > 0) {
            html += '<div class="text-sm font-medium text-gray-700 mb-2">Notifikasi Pesanan</div>';
            filteredOrderNotifications.forEach(notif => {
                const orderId = notif.related_order_id || notif.data?.order_id;
                const orderUrl = `/my-orders/${orderId}`;
                const productTitle = notif.order?.product_title || notif.data?.product_title || 'Produk';
                const customerName = notif.order?.customer_name || notif.data?.customer_name || 'Customer';
                
                // Determine badge and icon based on subtype
                let badgeHtml = '';
                let iconHtml = '';
                let cardBgColor = 'bg-blue-50';
                let cardBorderColor = 'border-blue-500';
                let cardHoverColor = 'hover:bg-blue-100';
                let textColor = 'text-blue-900';
                
                if (notif.subtype === 'order_status_changed') {
                    const newStatus = notif.data?.new_status || 'unknown';
                    if (newStatus === 'diterima') {
                        badgeHtml = '<span class="px-2 py-0.5 bg-green-100 text-green-700 text-xs rounded-full font-semibold">Diterima</span>';
                        iconHtml = '<i class="fas fa-check-circle mr-1"></i>';
                        cardBgColor = 'bg-green-50';
                        cardBorderColor = 'border-green-500';
                        cardHoverColor = 'hover:bg-green-100';
                        textColor = 'text-green-900';
                    } else if (newStatus === 'ditolak') {
                        badgeHtml = '<span class="px-2 py-0.5 bg-red-100 text-red-700 text-xs rounded-full">Ditolak</span>';
                        iconHtml = '<i class="fas fa-times-circle mr-1"></i>';
                        cardBgColor = 'bg-red-50';
                        cardBorderColor = 'border-red-500';
                        cardHoverColor = 'hover:bg-red-100';
                        textColor = 'text-red-900';
                    }
                } else if (notif.subtype === 'payment_received') {
                    badgeHtml = '<span class="px-2 py-0.5 bg-emerald-100 text-emerald-700 text-xs rounded-full">Pembayaran Diterima</span>';
                    iconHtml = '<i class="fas fa-money-bill-wave mr-1"></i>';
                    cardBgColor = 'bg-emerald-50';
                    cardBorderColor = 'border-emerald-500';
                    cardHoverColor = 'hover:bg-emerald-100';
                    textColor = 'text-emerald-900';
                } else if (notif.subtype === 'order_created') {
                    badgeHtml = '<span class="px-2 py-0.5 bg-blue-100 text-blue-700 text-xs rounded-full">Pesanan Dibuat</span>';
                    iconHtml = '<i class="fas fa-shopping-cart mr-1"></i>';
                }
                
                html += `
                    <div class="p-3 ${cardBgColor} ${cardBorderColor} ${cardHoverColor} rounded-lg border-l-4 transition-colors cursor-pointer notification-item" 
                         data-type="order" 
                         data-id="${orderId}"
                         data-notification-id="${notif.id || ''}"
                         data-subtype="${notif.subtype || 'default'}"
                         data-url="${orderUrl}">
                        <div class="flex items-start justify-between">
                            <div class="flex-1">
                                <div class="flex items-center gap-2 mb-1">
                                    <p class="font-medium ${textColor}">Pesanan #${orderId}</p>
                                    ${badgeHtml}
                                </div>
                                <p class="text-sm ${textColor} mt-1">
                                    ${iconHtml}
                                    ${notif.message || notif.title}
                                </p>
                                <p class="text-xs ${textColor.replace('900', '600')} mt-1">
                                    ${this.formatTime(notif.created_at)}
                                </p>
                            </div>
                            <div class="${textColor.replace('900', '600')} hover:${textColor.replace('900', '800')} ml-2">
                                <i class="fas fa-external-link-alt"></i>
                            </div>
                        </div>
                    </div>
                `;
            });
        }
        
        // Recent contacts (filtered)
        if (filteredContacts.length > 0) {
            html += '<div class="text-sm font-medium text-gray-700 mb-2">Balasan Admin</div>';
            filteredContacts.forEach(contact => {
                const hasReply = contact.admin_reply && contact.admin_replied_at;
                const contactUrl = `/user/contacts/${contact.id}`;
                html += `
                    <div class="p-3 bg-blue-50 rounded-lg border-l-4 border-blue-500 hover:bg-blue-100 transition-colors cursor-pointer notification-item" 
                         data-type="contact" 
                         data-id="${contact.id}"
                         data-notification-id="${contact.notification_id || ''}"
                         data-url="${contactUrl}">
                        <div class="flex items-start justify-between">
                            <div class="flex-1">
                                <div class="flex items-center gap-2 mb-1">
                                    <p class="font-medium text-blue-900">${contact.nama}</p>
                                    ${hasReply ? '<span class="px-2 py-0.5 bg-green-100 text-green-700 text-xs rounded-full">Sudah Dibalas</span>' : ''}
                                </div>
                                ${hasReply ? 
                                    `<p class="text-sm text-blue-700 mt-1"><i class="fas fa-reply mr-1"></i>${this.truncateText(contact.admin_reply, 60)}</p>` :
                                    `<p class="text-sm text-blue-700 mt-1">${this.truncateText(contact.pesan, 50)}</p>`
                                }
                                <p class="text-xs text-blue-600 mt-1">
                                    ${hasReply ? `Dibalas: ${this.formatTime(contact.admin_replied_at)}` : `Dikirim: ${this.formatTime(contact.created_at)}`}
                                </p>
                            </div>
                            <div class="text-blue-600 hover:text-blue-800 ml-2">
                                <i class="fas fa-external-link-alt"></i>
                            </div>
                        </div>
                    </div>
                `;
            });
        }

        // Recent inquiries (filtered)
        if (filteredInquiries.length > 0) {
            html += '<div class="text-sm font-medium text-gray-700 mb-2 mt-4">Notifikasi Penawaran</div>';
            // Backend sudah sort, tapi kita sort lagi untuk memastikan prioritas
            const sortedInquiries = [...filteredInquiries].sort((a, b) => {
                // Priority: Converted > Final Offer (approved) > Accepted > Others
                const aIsConverted = a.status === 'converted_to_order' && a.order_id;
                const bIsConverted = b.status === 'converted_to_order' && b.order_id;
                const aIsFinalOffer = a.status === 'approved' && a.admin_replied_at;
                const bIsFinalOffer = b.status === 'approved' && b.admin_replied_at;
                const aIsAccepted = a.status === 'customer_accepted';
                const bIsAccepted = b.status === 'customer_accepted';
                
                if (aIsConverted && !bIsConverted) return -1;
                if (!aIsConverted && bIsConverted) return 1;
                if (aIsFinalOffer && !bIsFinalOffer) return -1;
                if (!aIsFinalOffer && bIsFinalOffer) return 1;
                if (aIsAccepted && !bIsAccepted) return -1;
                if (!aIsAccepted && bIsAccepted) return 1;
                
                // Then sort by time (newest first)
                const aTime = new Date(a.converted_to_order_at || a.admin_replied_at || a.updated_at || a.created_at);
                const bTime = new Date(b.converted_to_order_at || b.admin_replied_at || b.updated_at || b.created_at);
                return bTime - aTime;
            });
            
            sortedInquiries.forEach(inquiry => {
                const hasReply = inquiry.admin_reply && inquiry.admin_replied_at;
                // Final offer: status approved dengan admin_replied_at, bisa punya final_offer_price ATAU min_price/max_price
                // ATAU jika subtype adalah 'final_offer_sent' (untuk notifikasi langsung dari final offer)
                const hasFinalOffer = (inquiry.status === 'approved' && inquiry.admin_replied_at && 
                                     (inquiry.final_offer_price || (inquiry.min_price && inquiry.max_price))) ||
                                     inquiry.subtype === 'final_offer_sent';
                const isConvertedToOrder = inquiry.status === 'converted_to_order' && inquiry.order_id;
                const isCustomerAccepted = inquiry.status === 'customer_accepted';
                const isNewInquiry = inquiry.status === 'pending' && !inquiry.is_read;
                
                let displayText = '';
                let timeText = '';
                let badgeHtml = '';
                let iconHtml = '';
                let url = `/user/inquiries/${inquiry.id}`;
                let sectionTitle = 'Balasan Admin';
                
                if (isConvertedToOrder) {
                    // Notifikasi khusus: Inquiry sudah di-convert ke order
                    displayText = `Penawaran Anda telah diubah menjadi pesanan! Order #${inquiry.order_id}`;
                    timeText = `Diubah: ${this.formatTime(inquiry.converted_to_order_at || inquiry.updated_at)}`;
                    badgeHtml = '<span class="px-2 py-0.5 bg-emerald-100 text-emerald-700 text-xs rounded-full">Pesanan Dibuat</span>';
                    iconHtml = '<i class="fas fa-shopping-cart mr-1"></i>';
                    url = `/my-orders/${inquiry.order_id}`; // Redirect ke detail order
                } else if (isCustomerAccepted) {
                    // User sudah setujui penawaran
                    displayText = `Anda telah menyetujui penawaran harga final. Admin akan segera membuat pesanan.`;
                    timeText = `Disetujui: ${this.formatTime(inquiry.updated_at)}`;
                    badgeHtml = '<span class="px-2 py-0.5 bg-green-100 text-green-700 text-xs rounded-full">Disetujui</span>';
                    iconHtml = '<i class="fas fa-check-circle mr-1"></i>';
                } else if (hasFinalOffer) {
                    // Final offer harus ditampilkan dengan jelas dan prioritas tinggi
                    const finalPrice = inquiry.final_offer_price || inquiry.min_price || inquiry.max_price;
                    if (finalPrice) {
                        displayText = `Penawaran Harga Final: Rp ${this.formatCurrency(finalPrice)}`;
                        if (inquiry.validity_until) {
                            const validityDate = new Date(inquiry.validity_until);
                            displayText += ` (Berlaku hingga ${validityDate.toLocaleDateString('id-ID', { day: 'numeric', month: 'short', year: 'numeric' })})`;
                        }
                        if (inquiry.lead_time_days) {
                            displayText += ` - Lead Time: ${inquiry.lead_time_days} hari`;
                        }
                    } else {
                        displayText = `Penawaran Harga Final telah dikirim. Silakan cek detail penawaran Anda.`;
                    }
                    timeText = `Dikirim: ${this.formatTime(inquiry.admin_replied_at || inquiry.updated_at || inquiry.created_at)}`;
                    badgeHtml = '<span class="px-2 py-0.5 bg-emerald-100 text-emerald-700 text-xs rounded-full font-semibold">Harga Final</span>';
                    iconHtml = '<i class="fas fa-tag mr-1"></i>';
                } else if (hasReply) {
                    displayText = this.truncateText(inquiry.admin_reply, 60);
                    timeText = `Dibalas: ${this.formatTime(inquiry.admin_replied_at)}`;
                    badgeHtml = '<span class="px-2 py-0.5 bg-green-100 text-green-700 text-xs rounded-full">Sudah Dibalas</span>';
                    iconHtml = '<i class="fas fa-reply mr-1"></i>';
                } else if (isNewInquiry) {
                    // Inquiry baru yang dibuat user
                    displayText = `Penawaran Anda telah diterima. Admin akan segera meninjau.`;
                    if (inquiry.tracking_code) {
                        displayText += ` Kode Tracking: ${inquiry.tracking_code}`;
                    }
                    timeText = `Dikirim: ${this.formatTime(inquiry.created_at)}`;
                    badgeHtml = '<span class="px-2 py-0.5 bg-blue-100 text-blue-700 text-xs rounded-full">Diterima</span>';
                    iconHtml = '<i class="fas fa-check mr-1"></i>';
                } else {
                    displayText = inquiry.product?.product_title || 'Produk tidak ditemukan';
                    timeText = `Dikirim: ${this.formatTime(inquiry.created_at)}`;
                }
                
                // Tentukan warna card berdasarkan jenis notifikasi
                let cardBgColor = 'bg-purple-50';
                let cardBorderColor = 'border-purple-500';
                let cardHoverColor = 'hover:bg-purple-100';
                let textColor = 'text-purple-900';
                
                if (isConvertedToOrder) {
                    cardBgColor = 'bg-emerald-50';
                    cardBorderColor = 'border-emerald-500';
                    cardHoverColor = 'hover:bg-emerald-100';
                    textColor = 'text-emerald-900';
                } else if (hasFinalOffer) {
                    // Final offer harus lebih menonjol
                    cardBgColor = 'bg-emerald-50';
                    cardBorderColor = 'border-emerald-500';
                    cardHoverColor = 'hover:bg-emerald-100';
                    textColor = 'text-emerald-900';
                }
                
                // Escape URL untuk data attribute
                const escapedUrl = url.replace(/'/g, "\\'").replace(/"/g, '&quot;');
                
                html += `
                    <div class="p-3 ${cardBgColor} ${cardBorderColor} ${cardHoverColor} rounded-lg border-l-4 transition-colors cursor-pointer notification-item" 
                         data-type="inquiry" 
                         data-id="${inquiry.id}"
                         data-notification-id="${inquiry.notification_id || ''}"
                         data-status="${inquiry.status || 'pending'}"
                         data-admin-replied-at="${(inquiry.admin_replied_at || '').replace(/"/g, '&quot;')}"
                         data-converted-to-order-at="${(inquiry.converted_to_order_at || '').replace(/"/g, '&quot;')}"
                         data-url="${escapedUrl}">
                        <div class="flex items-start justify-between">
                            <div class="flex-1">
                                <div class="flex items-center gap-2 mb-1">
                                    <p class="font-medium ${textColor}">${inquiry.name}</p>
                                    ${badgeHtml}
                                </div>
                                <p class="text-sm ${isConvertedToOrder ? 'text-emerald-700' : 'text-purple-700'} mt-1">
                                    ${iconHtml}
                                    ${displayText}
                                </p>
                                <p class="text-xs ${isConvertedToOrder ? 'text-emerald-600' : 'text-purple-600'} mt-1">${timeText}</p>
                            </div>
                            <div class="text-purple-600 hover:text-purple-800 ml-2">
                                <i class="fas fa-external-link-alt"></i>
                            </div>
                        </div>
                    </div>
                `;
            });
        }

        html += '</div>';

        content.innerHTML = html;
        
        // Add footer di luar content (setelah content, di dalam panel)
        const panel = document.getElementById('user-notification-panel');
        if (panel) {
            // Hapus footer lama jika ada
            const oldFooter = panel.querySelector('.notification-footer');
            if (oldFooter) {
                oldFooter.remove();
            }
            
            // Tambahkan footer baru
            const footer = document.createElement('div');
            footer.className = 'notification-footer p-4 border-t border-gray-200 flex-shrink-0 bg-white';
            footer.innerHTML = `
                <div class="flex justify-between items-center text-sm text-gray-500">
                    <span>Total: ${totalUnread} notifikasi baru</span>
                    <span>Terakhir update: ${this.formatTime(data.last_updated)}</span>
                </div>
            `;
            
            // Insert footer setelah notification-content
            const notificationContent = document.getElementById('notification-content');
            if (notificationContent && notificationContent.parentNode) {
                notificationContent.parentNode.insertBefore(footer, notificationContent.nextSibling);
            }
        }
        
        // Bind click events setelah HTML di-render
        this.bindNotificationClicks();
    }

    updateBadges(unreadCounts, recentItems = null) {
        // Filter out notifications that user has already viewed (stored in localStorage)
        // IMPORTANT: Gunakan kombinasi ID + status untuk tracking
        const viewedNotifications = this.getViewedNotifications();
        let actualUnreadCount = 0;
        let actualUnreadInquiries = 0;
        let actualUnreadContacts = 0;

        // If we have recent items, count only those that haven't been viewed
        if (recentItems) {
            // Count order notifications that haven't been viewed
            const orderNotifications = recentItems.notifications?.filter(n => n.type === 'order') || [];
            const actualUnreadOrders = orderNotifications.filter(notif => {
                const orderId = notif.related_order_id || notif.data?.order_id;
                if (!orderId) return false;
                const subtype = notif.subtype || 'default';
                const notificationKey = `order_${orderId}_${subtype}`;
                const isViewed = viewedNotifications.orders?.has(notificationKey);
                return !isViewed;
            }).length;
            
            // Count inquiries that haven't been viewed
            actualUnreadInquiries = recentItems.inquiries.filter(inquiry => {
                const inquiryId = parseInt(inquiry.id);
                const status = inquiry.status || 'pending';
                
                // Buat key unik: ID + status + timestamp (untuk final offer dan converted to order)
                let notificationKey = `${inquiryId}_${status}`;
                
                // Untuk final offer (approved), tambahkan admin_replied_at
                if (status === 'approved' && inquiry.admin_replied_at) {
                    const repliedAt = new Date(inquiry.admin_replied_at).getTime();
                    notificationKey = `${inquiryId}_approved_${repliedAt}`;
                }
                
                // Untuk converted to order, tambahkan converted_to_order_at
                if (status === 'converted_to_order' && inquiry.converted_to_order_at) {
                    const convertedAt = new Date(inquiry.converted_to_order_at).getTime();
                    notificationKey = `${inquiryId}_converted_${convertedAt}`;
                }
                
                const isViewed = viewedNotifications.inquiries.has(notificationKey);
                if (isViewed) {
                    console.log(`Inquiry #${inquiryId} (${status}) already viewed with key: ${notificationKey}, skipping`);
                }
                return !isViewed;
            }).length;

            // Count contacts that haven't been viewed
            actualUnreadContacts = recentItems.contacts.filter(contact => {
                const contactId = parseInt(contact.id);
                const isViewed = viewedNotifications.contacts.has(contactId);
                if (isViewed) {
                    console.log(`Contact #${contactId} already viewed, skipping`);
                }
                return !isViewed;
            }).length;

            actualUnreadCount = actualUnreadInquiries + actualUnreadContacts + actualUnreadOrders;
            
            console.log(`Badge update: ${actualUnreadInquiries} inquiries + ${actualUnreadContacts} contacts + ${actualUnreadOrders} orders = ${actualUnreadCount} total`);
        } else {
            // Fallback: use simple subtraction (less accurate but works)
            actualUnreadInquiries = Math.max(0, unreadCounts.inquiries - viewedNotifications.inquiries.size);
            actualUnreadContacts = Math.max(0, unreadCounts.contacts - viewedNotifications.contacts.size);
            actualUnreadCount = actualUnreadInquiries + actualUnreadContacts;
        }

        // Update page title
        if (actualUnreadCount > 0) {
            document.title = `(${actualUnreadCount}) Dashboard User - POLJAM TECH`;
        } else {
            document.title = 'Dashboard User - POLJAM TECH';
        }

        // Update ALL notification bell badges (desktop and mobile)
        // Cari badge dengan berbagai selector untuk kompatibilitas
        const bellBadges = document.querySelectorAll('.notification-bell .badge, button.notification-bell .badge, .badge');
        bellBadges.forEach(bellBadge => {
            // Hanya update badge yang ada di dalam notification bell atau button dengan onclick toggleUserNotifications
            const parent = bellBadge.closest('.notification-bell, button');
            if (parent && (
                parent.classList.contains('notification-bell') || 
                (parent.onclick && parent.onclick.toString().includes('toggleUserNotifications'))
            )) {
                if (actualUnreadCount > 0) {
                    bellBadge.textContent = actualUnreadCount > 99 ? '99+' : actualUnreadCount.toString();
                    bellBadge.style.display = 'flex';
                    bellBadge.classList.remove('hidden');
                } else {
                    bellBadge.style.display = 'none';
                    bellBadge.classList.add('hidden');
                }
            }
        });
        
        // Debug log
        if (bellBadges.length === 0) {
            console.warn('[UserNotificationSystem] Notification badge not found in navbar');
        } else {
            console.log(`[UserNotificationSystem] Updated ${bellBadges.length} badge(s) with count: ${actualUnreadCount}`);
        }

        // Update dashboard stats if on dashboard page
        this.updateDashboardStats({
            ...unreadCounts,
            total: actualUnreadCount,
            inquiries: actualUnreadInquiries,
            contacts: actualUnreadContacts
        });
    }

    /**
     * Get viewed notifications from localStorage
     * IMPORTANT: Sekarang menggunakan key berbasis ID + status + timestamp
     */
    getViewedNotifications() {
        try {
            const viewed = localStorage.getItem('viewed_notifications');
            if (viewed) {
                const parsed = JSON.parse(viewed);
                // Support both old format (array of IDs) and new format (array of keys)
                const inquiryKeys = parsed.inquiries || [];
                
                // Convert old format (numbers) to new format (strings) jika perlu
                const convertedKeys = inquiryKeys.map(key => {
                    // Jika key adalah number (format lama), skip karena akan menggunakan format baru
                    if (typeof key === 'number') {
                        return null; // Skip old format
                    }
                    return key;
                }).filter(key => key !== null);
                
                return {
                    inquiries: new Set(convertedKeys), // Hanya string keys (format baru)
                    contacts: new Set(parsed.contacts || []),
                    orders: new Set(parsed.orders || [])
                };
            }
        } catch (e) {
            console.error('Error reading viewed notifications:', e);
        }
        return {
            inquiries: new Set(),
            contacts: new Set(),
            orders: new Set()
        };
    }
    
    /**
     * Clear old format viewed notifications (migrate to new format)
     * Call this once to migrate from old format (ID only) to new format (ID + status + timestamp)
     */
    clearOldViewedNotifications() {
        try {
            const viewed = localStorage.getItem('viewed_notifications');
            if (viewed) {
                const parsed = JSON.parse(viewed);
                const inquiryKeys = parsed.inquiries || [];
                
                // Jika semua keys adalah numbers (format lama), clear semua
                const allNumbers = inquiryKeys.every(key => typeof key === 'number');
                if (allNumbers && inquiryKeys.length > 0) {
                    console.log('[UserNotificationSystem] Clearing old format viewed notifications for migration');
                    localStorage.setItem('viewed_notifications', JSON.stringify({
                        inquiries: [],
                        contacts: parsed.contacts || []
                    }));
                }
            }
        } catch (e) {
            console.error('Error clearing old viewed notifications:', e);
        }
    }

    /**
     * Mark notification as viewed in localStorage
     * IMPORTANT: Sekarang menggunakan key berbasis ID + status + timestamp
     */
    markNotificationAsViewed(type, id, inquiryData = null, orderData = null) {
        try {
            const viewed = this.getViewedNotifications();
            
            if (type === 'order') {
                const orderId = parseInt(id);
                const subtype = orderData?.subtype || 'default';
                const notificationKey = `order_${orderId}_${subtype}`;
                viewed.orders.add(notificationKey);
            } else if (type === 'inquiry') {
                const inquiryId = parseInt(id);
                let notificationKey = inquiryId.toString();
                
                // Jika ada inquiryData, gunakan key yang lebih spesifik
                if (inquiryData) {
                    const status = inquiryData.status || 'pending';
                    notificationKey = `${inquiryId}_${status}`;
                    
                    // Untuk final offer (approved), tambahkan admin_replied_at
                    if (status === 'approved' && inquiryData.admin_replied_at) {
                        const repliedAt = new Date(inquiryData.admin_replied_at).getTime();
                        notificationKey = `${inquiryId}_approved_${repliedAt}`;
                    }
                    
                    // Untuk converted to order, tambahkan converted_to_order_at
                    if (status === 'converted_to_order' && inquiryData.converted_to_order_at) {
                        const convertedAt = new Date(inquiryData.converted_to_order_at).getTime();
                        notificationKey = `${inquiryId}_converted_${convertedAt}`;
                    }
                }
                
                viewed.inquiries.add(notificationKey);
            } else if (type === 'contact') {
                const contactId = parseInt(id);
                viewed.contacts.add(contactId);
            }

            const viewedData = {
                inquiries: Array.from(viewed.inquiries),
                contacts: Array.from(viewed.contacts),
                orders: Array.from(viewed.orders)
            };
            
            localStorage.setItem('viewed_notifications', JSON.stringify(viewedData));
            
            console.log(`Marked ${type} #${id} as viewed with key: ${type === 'inquiry' ? notificationKey : id}. Viewed:`, viewedData);

            // Update badge optimistically
            this.updateBadgeOptimistically(type, id);

            // Refresh notifications to update badge (async, doesn't block)
            this.fetchNotifications();
        } catch (e) {
            console.error('Error saving viewed notification:', e);
        }
    }

    updateDashboardStats(unreadCounts) {
        // Update contact badge
        const contactBadge = document.querySelector('#contact-unread-badge');
        if (contactBadge) {
            contactBadge.textContent = unreadCounts.contacts;
            contactBadge.style.display = unreadCounts.contacts > 0 ? 'inline' : 'none';
        }

        // Update inquiry badge
        const inquiryBadge = document.querySelector('#inquiry-unread-badge');
        if (inquiryBadge) {
            inquiryBadge.textContent = unreadCounts.inquiries;
            inquiryBadge.style.display = unreadCounts.inquiries > 0 ? 'inline' : 'none';
        }
    }

    showNotificationPanel() {
        if (this.notificationPanel) {
            this.notificationPanel.classList.remove('hidden');
            this.fetchNotifications(); // Refresh data when showing
        }
    }

    hideNotificationPanel() {
        if (this.notificationPanel) {
            this.notificationPanel.classList.add('hidden');
        }
    }

    toggleNotificationPanel() {
        if (this.notificationPanel.classList.contains('hidden')) {
            this.showNotificationPanel();
        } else {
            this.hideNotificationPanel();
        }
    }

    /**
     * Mark all notifications as read
     */
    async markAllAsRead() {
        try {
            const csrfToken = document.querySelector('meta[name="csrf-token"]');
            if (!csrfToken) {
                console.error('[UserNotificationSystem] CSRF token not found');
                return;
            }

            const response = await fetch('/user/notifications/mark-all-read', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken.getAttribute('content')
                }
            });

            if (!response.ok) {
                throw new Error(`Network response was not ok: ${response.status} ${response.statusText}`);
            }

            const data = await response.json();
            
            if (data.success) {
                // Clear all viewed notifications from localStorage
                const keys = Object.keys(localStorage);
                keys.forEach(key => {
                    if (key.startsWith('notification_viewed_')) {
                        localStorage.removeItem(key);
                    }
                });

                // Update badge menggunakan unread_counts dari response (jika ada)
                if (data.data && data.data.unread_counts) {
                    console.log('[UserNotificationSystem] Updating badges from response:', data.data.unread_counts);
                    this.updateBadges(data.data.unread_counts);
                } else {
                    // Fallback: update badge to 0
                    const bellBadges = document.querySelectorAll('.notification-bell .badge, .badge');
                    bellBadges.forEach(bellBadge => {
                        // Hanya update badge yang ada di dalam notification bell
                        const parent = bellBadge.closest('.notification-bell, button');
                        if (parent && (parent.classList.contains('notification-bell') || parent.onclick && parent.onclick.toString().includes('toggleUserNotifications'))) {
                            bellBadge.textContent = '0';
                            bellBadge.style.display = 'none';
                        }
                    });
                }

                // Reload notifications to refresh the display
                setTimeout(() => {
                    this.fetchNotifications();
                }, 100);
            } else {
                console.error('[UserNotificationSystem] Failed to mark all as read:', data.message);
            }
        } catch (error) {
            console.error('[UserNotificationSystem] Error marking all notifications as read:', error);
        }
    }

    /**
     * Handle notification click - mark as viewed (localStorage) and navigate
     */
    async handleNotificationClick(type, id, url, inquiryData = null, orderData = null, notificationId = null) {
        try {
            // Mark as viewed in localStorage FIRST (synchronously)
            // Gunakan markNotificationAsViewed yang sudah support key berbasis status
            this.markNotificationAsViewed(type, id, inquiryData, orderData);

            // Update badge immediately (optimistic update)
            this.updateBadgeOptimistically(type, id);

            // Call API untuk mark as read jika ada notification ID
            if (notificationId) {
                try {
                    const csrfToken = document.querySelector('meta[name="csrf-token"]');
                    if (csrfToken) {
                        const response = await fetch(`/user/notifications/${notificationId}/mark-read`, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': csrfToken.getAttribute('content'),
                                'Accept': 'application/json'
                            }
                        });

                        if (response.ok) {
                            const data = await response.json();
                            if (data.success && data.data?.unread_counts) {
                                // Update badge dengan unread_counts dari response
                                this.updateBadges(data.data.unread_counts);
                            }
                        }
                    }
                } catch (apiError) {
                    console.error('[UserNotificationSystem] Error marking notification as read via API:', apiError);
                    // Continue navigation even if API call fails
                }
            }

            // Small delay to ensure badge update is visible, then navigate
            setTimeout(() => {
                window.location.href = url;
            }, 100);
        } catch (error) {
            console.error('Error handling notification click:', error);
            // Still navigate even if there's an error
            window.location.href = url;
        }
    }

    /**
     * Optimistically update badge when notification is marked as viewed
     */
    updateBadgeOptimistically(type, id) {
        const bellBadges = document.querySelectorAll('.notification-bell .badge');
        bellBadges.forEach(bellBadge => {
            const currentCount = parseInt(bellBadge.textContent) || 0;
            if (currentCount > 0) {
                const newCount = currentCount - 1;
                if (newCount > 0) {
                    bellBadge.textContent = newCount;
                    bellBadge.style.display = 'flex';
                } else {
                    bellBadge.style.display = 'none';
                }
            }
        });
        
        console.log(`Optimistically updated badge for ${type} #${id}`);
    }

    // Utility functions
    truncateText(text, maxLength) {
        if (text.length <= maxLength) return text;
        return text.substring(0, maxLength) + '...';
    }

    formatTime(timestamp) {
        const date = new Date(timestamp);
        const now = new Date();
        const diffInSeconds = Math.floor((now - date) / 1000);

        if (diffInSeconds < 60) {
            return 'Baru saja';
        } else if (diffInSeconds < 3600) {
            const minutes = Math.floor(diffInSeconds / 60);
            return `${minutes} menit yang lalu`;
        } else if (diffInSeconds < 86400) {
            const hours = Math.floor(diffInSeconds / 3600);
            return `${hours} jam yang lalu`;
        } else {
            return date.toLocaleDateString('id-ID', {
                day: 'numeric',
                month: 'short',
                hour: '2-digit',
                minute: '2-digit'
            });
        }
    }

    formatCurrency(amount) {
        if (!amount) return '0';
        return new Intl.NumberFormat('id-ID').format(amount);
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Only initialize if user is authenticated
    if (document.querySelector('meta[name="user-authenticated"]')) {
        window.userNotificationSystem = new UserNotificationSystem();
    }
});

// Global functions for HTML onclick events
function toggleUserNotifications() {
    if (window.userNotificationSystem) {
        window.userNotificationSystem.toggleNotificationPanel();
    }
}

function showUserNotifications() {
    if (window.userNotificationSystem) {
        window.userNotificationSystem.showNotificationPanel();
    }
}

function hideUserNotifications() {
    if (window.userNotificationSystem) {
        window.userNotificationSystem.hideNotificationPanel();
    }
}
