/**
 * Modern SweetAlert Helper Functions
 * Untuk notifikasi dan konfirmasi yang konsisten di seluruh aplikasi
 */

// SweetAlert Configuration
const SwalConfig = {
    confirmButton: {
        class: 'px-6 py-2.5 rounded-lg font-semibold transition-all duration-200 shadow-lg hover:shadow-xl',
    },
    cancelButton: {
        class: 'px-6 py-2.5 rounded-lg font-semibold transition-all duration-200',
    },
    popup: {
        class: 'rounded-2xl shadow-2xl border border-gray-200 dark:border-gray-700',
    }
};

/**
 * Modern Confirm Dialog
 */
function showConfirm(options) {
    const {
        title = 'Konfirmasi',
        text = 'Apakah Anda yakin?',
        confirmText = 'Ya, Lanjutkan',
        cancelText = 'Batal',
        icon = 'question',
        confirmColor = '#3b82f6',
        onConfirm = null,
        onCancel = null,
        showLoader = false
    } = options;

    return Swal.fire({
        title: title,
        text: text,
        icon: icon,
        showCancelButton: true,
        confirmButtonColor: confirmColor,
        cancelButtonColor: '#6b7280',
        confirmButtonText: `<i class="fas fa-check mr-2"></i>${confirmText}`,
        cancelButtonText: `<i class="fas fa-times mr-2"></i>${cancelText}`,
        reverseButtons: true,
        allowOutsideClick: !showLoader,
        allowEscapeKey: !showLoader,
        customClass: {
            popup: SwalConfig.popup.class,
            confirmButton: SwalConfig.confirmButton.class,
            cancelButton: SwalConfig.cancelButton.class,
            title: 'text-xl font-bold text-gray-900 dark:text-white',
            htmlContainer: 'text-gray-700 dark:text-gray-300',
        },
        buttonsStyling: false,
        didOpen: () => {
            if (showLoader) {
                Swal.showLoading();
            }
        }
    }).then((result) => {
        if (result.isConfirmed && onConfirm) {
            onConfirm(result);
        } else if (result.isDismissed && onCancel) {
            onCancel(result);
        }
        return result;
    });
}

/**
 * Modern Success Notification
 */
function showSuccess(message, title = 'Berhasil!', duration = 3000) {
    return Swal.fire({
        icon: 'success',
        title: title,
        text: message,
        timer: duration,
        timerProgressBar: true,
        showConfirmButton: false,
        customClass: {
            popup: SwalConfig.popup.class,
            title: 'text-xl font-bold text-gray-900 dark:text-white',
            htmlContainer: 'text-gray-700 dark:text-gray-300',
        },
        buttonsStyling: false,
    });
}

/**
 * Modern Error Notification
 */
function showError(message, title = 'Terjadi Kesalahan!', duration = 4000) {
    return Swal.fire({
        icon: 'error',
        title: title,
        text: message,
        timer: duration,
        timerProgressBar: true,
        showConfirmButton: true,
        confirmButtonText: '<i class="fas fa-times mr-2"></i>Tutup',
        confirmButtonColor: '#ef4444',
        customClass: {
            popup: SwalConfig.popup.class,
            confirmButton: SwalConfig.confirmButton.class,
            title: 'text-xl font-bold text-gray-900 dark:text-white',
            htmlContainer: 'text-gray-700 dark:text-gray-300',
        },
        buttonsStyling: false,
    });
}

/**
 * Modern Warning Notification
 */
function showWarning(message, title = 'Peringatan!', duration = 4000) {
    return Swal.fire({
        icon: 'warning',
        title: title,
        text: message,
        timer: duration,
        timerProgressBar: true,
        showConfirmButton: true,
        confirmButtonText: '<i class="fas fa-check mr-2"></i>Mengerti',
        confirmButtonColor: '#f59e0b',
        customClass: {
            popup: SwalConfig.popup.class,
            confirmButton: SwalConfig.confirmButton.class,
            title: 'text-xl font-bold text-gray-900 dark:text-white',
            htmlContainer: 'text-gray-700 dark:text-gray-300',
        },
        buttonsStyling: false,
    });
}

/**
 * Modern Info Notification
 */
function showInfo(message, title = 'Informasi', duration = 3000) {
    return Swal.fire({
        icon: 'info',
        title: title,
        text: message,
        timer: duration,
        timerProgressBar: true,
        showConfirmButton: false,
        customClass: {
            popup: SwalConfig.popup.class,
            title: 'text-xl font-bold text-gray-900 dark:text-white',
            htmlContainer: 'text-gray-700 dark:text-gray-300',
        },
        buttonsStyling: false,
    });
}

/**
 * Modern Loading Indicator
 */
function showLoading(title = 'Memproses...', text = 'Harap tunggu sebentar') {
    return Swal.fire({
        title: title,
        text: text,
        allowOutsideClick: false,
        allowEscapeKey: false,
        showConfirmButton: false,
        didOpen: () => {
            Swal.showLoading();
        },
        customClass: {
            popup: SwalConfig.popup.class,
            title: 'text-xl font-bold text-gray-900 dark:text-white',
            htmlContainer: 'text-gray-700 dark:text-gray-300',
        },
    });
}

/**
 * Replace native confirm with modern SweetAlert
 */
function modernConfirm(message, onConfirm, onCancel) {
    return showConfirm({
        title: 'Konfirmasi',
        text: message,
        onConfirm: onConfirm,
        onCancel: onCancel
    });
}

// Export functions globally
window.showConfirm = showConfirm;
window.showSuccess = showSuccess;
window.showError = showError;
window.showWarning = showWarning;
window.showInfo = showInfo;
window.showLoading = showLoading;
window.modernConfirm = modernConfirm;

// Ensure functions are available immediately
if (typeof Swal === 'undefined') {
    console.error('SweetAlert2 is not loaded. Please ensure SweetAlert2 is loaded before sweetalert-helper.js');
} else {
    console.log('SweetAlert Helper loaded successfully');
}

