@extends('layouts.admin')

@section('title', 'Dashboard Admin - Daftar Pesanan')

@section('content')
    <!-- Breadcrumb -->
    @php
        $breadcrumbs = [
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Pesanan', 'url' => route('admin.master.orders.index')],
        ];
    @endphp
    <x-breadcrumb :items="$breadcrumbs" />

    <!-- Notifikasi Modern -->
    <x-admin-notification />

    <!-- Header -->
    <div class="mb-6">
        <h1 class="text-2xl font-bold text-gray-800 dark:text-white mb-4">Daftar Pesanan</h1>

        <!-- Filter Section (Single Card) -->
        <div class="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-4">
            <form method="GET" action="{{ route('admin.master.orders.index') }}" id="filterForm">
                <!-- Row 1: Search Bar & Add Button -->
                <div class="flex flex-col sm:flex-row items-end gap-3 mb-3">
                    <!-- Search Bar -->
                    <div class="relative flex-1 w-full">
                        <input type="text" name="search" id="searchInput"
                            value="{{ request('search') ?? ($search ?? '') }}" placeholder="Cari nama, email, produk..."
                            class="form-input-search" />
                        <i
                            class="fas fa-search absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none"></i>
                        @if (!empty(request('search')) || !empty($search))
                            <button type="button" onclick="clearSearch()"
                                class="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 hover:text-gray-600 dark:hover:text-gray-300">
                                <i class="fas fa-times"></i>
                            </button>
                        @endif
                    </div>

                    <!-- Add Button -->
                    <button type="button" onclick="toggleTab('create')" id="addOrderBtn"
                        class="action-btn-primary w-full sm:w-auto">
                        <i class="fas fa-plus mr-2"></i>Tambah Pesanan
                    </button>
                </div>

                <!-- Row 2: All Filters (5 equal-width columns) -->
                <div class="grid grid-cols-1 sm:grid-cols-5 gap-3">
                    <!-- Order Status Filter -->
                    <div class="relative w-full">
                        <select name="order_status" id="orderStatusFilter" onchange="submitFilter()"
                            class="form-select w-full">
                            <option value="">Semua Status Pesanan</option>
                            <option value="menunggu" {{ request('order_status') === 'menunggu' ? 'selected' : '' }}>Menunggu
                            </option>
                            <option value="diterima" {{ request('order_status') === 'diterima' ? 'selected' : '' }}>Diterima
                            </option>
                            <option value="ditolak" {{ request('order_status') === 'ditolak' ? 'selected' : '' }}>Ditolak
                            </option>
                        </select>
                        <i
                            class="fas fa-chevron-down absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none text-xs"></i>
                    </div>

                    <!-- Payment Status Filter -->
                    <div class="relative w-full">
                        <select name="payment_status" id="paymentStatusFilter" onchange="submitFilter()"
                            class="form-select w-full">
                            <option value="">Semua Status Bayar</option>
                            <option value="menunggu" {{ request('payment_status') === 'menunggu' ? 'selected' : '' }}>
                                Menunggu
                            </option>
                            <option value="menunggu_verifikasi"
                                {{ request('payment_status') === 'menunggu_verifikasi' ? 'selected' : '' }}>Menunggu
                                Verifikasi
                            </option>
                            <option value="diproses" {{ request('payment_status') === 'diproses' ? 'selected' : '' }}>
                                Diproses
                            </option>
                            <option value="selesai" {{ request('payment_status') === 'selesai' ? 'selected' : '' }}>Selesai
                            </option>
                            <option value="gagal" {{ request('payment_status') === 'gagal' ? 'selected' : '' }}>Gagal
                            </option>
                            <option value="dibatalkan" {{ request('payment_status') === 'dibatalkan' ? 'selected' : '' }}>
                                Dibatalkan</option>
                        </select>
                        <i
                            class="fas fa-chevron-down absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none text-xs"></i>
                    </div>

                    <!-- Production Status Filter -->
                    <div class="relative w-full">
                        <select name="production_status" id="productionStatusFilter" onchange="submitFilter()"
                            class="form-select w-full">
                            <option value="">Semua Status Produksi</option>
                            <option value="menunggu" {{ request('production_status') === 'menunggu' ? 'selected' : '' }}>
                                Menunggu</option>
                            <option value="dalam_proses"
                                {{ request('production_status') === 'dalam_proses' ? 'selected' : '' }}>Dalam Proses
                            </option>
                            <option value="selesai" {{ request('production_status') === 'selesai' ? 'selected' : '' }}>
                                Selesai
                            </option>
                            <option value="dibatalkan"
                                {{ request('production_status') === 'dibatalkan' ? 'selected' : '' }}>
                                Dibatalkan</option>
                        </select>
                        <i
                            class="fas fa-chevron-down absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none text-xs"></i>
                    </div>

                    <!-- Payment Method Filter -->
                    <div class="relative w-full">
                        <select name="payment_method" id="paymentMethodFilter" onchange="submitFilter()"
                            class="form-select w-full">
                            <option value="">Semua Metode</option>
                            <option value="transfer" {{ request('payment_method') === 'transfer' ? 'selected' : '' }}>
                                Transfer
                            </option>
                            <option value="midtrans" {{ request('payment_method') === 'midtrans' ? 'selected' : '' }}>Midtrans</option>
                            <option value="dp" {{ request('payment_method') === 'dp' ? 'selected' : '' }}>DP</option>
                        </select>
                        <i
                            class="fas fa-chevron-down absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none text-xs"></i>
                    </div>

                    <!-- Order Type Filter -->
                    <div class="relative w-full">
                        <select name="order_type" id="orderTypeFilter" onchange="submitFilter()" class="form-select w-full">
                            <option value="">Semua Jenis</option>
                            <option value="penawaran" {{ request('order_type') === 'penawaran' ? 'selected' : '' }}>
                                Penawaran
                            </option>
                            <option value="beli_langsung"
                                {{ request('order_type') === 'beli_langsung' ? 'selected' : '' }}>
                                Beli Langsung</option>
                        </select>
                        <i
                            class="fas fa-chevron-down absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none text-xs"></i>
                    </div>
                </div>
            </form>
        </div>

        <!-- Active Filters Info -->
        @if (request('search') ||
                request('order_status') ||
                request('payment_status') ||
                request('production_status') ||
                request('payment_method') ||
                request('order_type'))
            <div
                class="mt-4 mb-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-700 rounded-lg p-4">
                <div class="flex flex-wrap items-center gap-2">
                    <span class="text-sm font-medium text-blue-800 dark:text-blue-200">Filter Aktif:</span>

                    @if (request('search') || !empty($search))
                        <span
                            class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800 dark:bg-blue-800 dark:text-blue-200">
                            <i class="fas fa-search mr-1"></i>
                            Pencarian: "{{ request('search') ?? $search }}"
                            <button onclick="clearFilter('search')" class="ml-2 hover:text-blue-600">
                                <i class="fas fa-times"></i>
                            </button>
                        </span>
                    @endif

                    @if (request('order_status'))
                        <span
                            class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200">
                            <i class="fas fa-check-circle mr-1"></i>
                            Status Pesanan: {{ ucfirst(request('order_status')) }}
                            <button onclick="clearFilter('order_status')" class="ml-2 hover:text-green-600">
                                <i class="fas fa-times"></i>
                            </button>
                        </span>
                    @endif

                    @if (request('payment_status'))
                        <span
                            class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-purple-100 text-purple-800 dark:bg-purple-800 dark:text-purple-200">
                            <i class="fas fa-money-bill-wave mr-1"></i>
                            Status Bayar: {{ ucfirst(str_replace('_', ' ', request('payment_status'))) }}
                            <button onclick="clearFilter('payment_status')" class="ml-2 hover:text-purple-600">
                                <i class="fas fa-times"></i>
                            </button>
                        </span>
                    @endif

                    @if (request('production_status'))
                        <span
                            class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-orange-100 text-orange-800 dark:bg-orange-800 dark:text-orange-200">
                            <i class="fas fa-industry mr-1"></i>
                            Status Produksi: {{ ucfirst(str_replace('_', ' ', request('production_status'))) }}
                            <button onclick="clearFilter('production_status')" class="ml-2 hover:text-orange-600">
                                <i class="fas fa-times"></i>
                            </button>
                        </span>
                    @endif

                    @if (request('payment_method'))
                        <span
                            class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-200">
                            <i class="fas fa-credit-card mr-1"></i>
                            Metode: {{ strtoupper(request('payment_method')) }}
                            <button onclick="clearFilter('payment_method')" class="ml-2 hover:text-yellow-600">
                                <i class="fas fa-times"></i>
                            </button>
                        </span>
                    @endif

                    @if (request('order_type'))
                        <span
                            class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-indigo-100 text-indigo-800 dark:bg-indigo-800 dark:text-indigo-200">
                            <i class="fas fa-tag mr-1"></i>
                            Jenis: {{ request('order_type') === 'penawaran' ? 'Penawaran' : 'Beli Langsung' }}
                            <button onclick="clearFilter('order_type')" class="ml-2 hover:text-indigo-600">
                                <i class="fas fa-times"></i>
                            </button>
                        </span>
                    @endif

                    <button onclick="clearAllFilters()"
                        class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-200 hover:bg-red-200 dark:hover:bg-red-700 transition-colors">
                        <i class="fas fa-times-circle mr-1"></i>
                        Hapus Semua Filter
                    </button>
                </div>
            </div>
        @endif
    </div>

    <!-- Tab Navigation -->
    <div class="mb-6">
        <div class="border-b border-gray-200 dark:border-gray-700">
            <nav class="-mb-px flex space-x-8">
                <!-- Tab untuk melihat data -->
                <a href="{{ route('admin.master.orders.index', ['tab' => 'all'] + request()->query()) }}"
                    class="py-2 px-1 border-b-2 font-medium text-sm transition-colors duration-200 {{ $tab === 'all' ? 'border-blue-500 text-blue-600 dark:text-blue-400' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 dark:text-gray-400 dark:hover:text-gray-300' }}">
                    <i class="fas fa-list mr-2"></i>Semua ({{ $stats['total_orders'] }})
                </a>
                <a href="{{ route('admin.master.orders.index', ['tab' => 'orders'] + request()->query()) }}"
                    class="py-2 px-1 border-b-2 font-medium text-sm transition-colors duration-200 {{ $tab === 'orders' ? 'border-blue-500 text-blue-600 dark:text-blue-400' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 dark:text-gray-400 dark:hover:text-gray-300' }}">
                    <i class="fas fa-shopping-cart mr-2"></i>Pesanan ({{ $stats['orders_without_sales'] }})
                </a>
            </nav>
        </div>
    </div>

    <!-- Tab Content -->
    <!-- List Tab -->
    <div id="listContent" class="tab-content">
        <!-- Table Container -->
        <div
            class="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden transition-colors duration-300">
            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead class="bg-gray-50 dark:bg-gray-700/50 border-b border-gray-200 dark:border-gray-600">
                        <tr>
                            <th
                                class="w-16 px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                No</th>
                            <th
                                class="px-4 py-3.5 text-left text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                ID Pesanan</th>
                            <th
                                class="px-4 py-3.5 text-left text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Nama Customer</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Jenis Pesanan</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Produk</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Quantity</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Total Harga</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Bukti Pembayaran</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Status Pesanan</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Status Pembayaran</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Status Produksi</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Pembayaran</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Piutang</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Status Sale</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Status</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Tanggal</th>
                            <th
                                class="px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                                Aksi</th>
                        </tr>
                    </thead>

                    <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                        @forelse ($orders as $order)
                            <tr
                                class="hover:bg-blue-50/50 dark:hover:bg-gray-700/80 transition-colors duration-200 cursor-pointer">
                                <td class="px-4 py-3.5 text-sm text-gray-900 dark:text-gray-100 text-center align-middle">
                                    {{ $loop->iteration + $orders->firstItem() - 1 }}
                                </td>
                                <td class="px-4 py-3.5 text-sm text-gray-800 dark:text-gray-200 font-medium align-middle">
                                    #{{ $order->id }}</td>
                                <td class="px-4 py-3.5 text-sm text-gray-800 dark:text-gray-200 font-medium align-middle">
                                    <div class="flex items-center space-x-2">
                                        {{ $order->customer_name }}
                                        @if (!$order->is_read)
                                            <span
                                                class="inline-flex items-center justify-center w-2 h-2 bg-red-500 rounded-full animate-pulse"
                                                title="Pesanan Baru"></span>
                                        @endif
                                    </div>
                                </td>
                                <td class="px-4 py-3.5 text-sm text-center align-middle">
                                    @php $otype = $order->order_type ?? 'beli_langsung'; @endphp
                                    <span
                                        class="inline-flex items-center px-2.5 py-1 text-xs font-medium rounded-md
                                        @if ($otype === 'penawaran') bg-purple-100 dark:bg-purple-900/40 text-purple-700 dark:text-purple-300
                                        @else bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300 @endif">
                                        {{ $otype === 'penawaran' ? 'Penawaran' : 'Beli Langsung' }}
                                    </span>
                                </td>
                                <td class="px-4 py-3.5 text-sm text-gray-700 dark:text-gray-300 align-middle">
                                    <div class="flex items-center space-x-2 justify-center">
                                        @if ($order->product && $order->product->product_image)
                                            <img src="{{ asset('uploads/products/' . $order->product->product_image) }}"
                                                alt="{{ $order->product->product_title }}"
                                                class="h-16 w-16 object-contain rounded" style="background: transparent;">
                                        @else
                                            <div
                                                class="h-16 w-16 bg-gray-100 dark:bg-gray-600 rounded flex items-center justify-center">
                                                <i class="fas fa-image text-gray-400 dark:text-gray-500 text-lg"></i>
                                            </div>
                                        @endif
                                        <span
                                            class="font-medium text-gray-900 dark:text-gray-100 text-sm">{{ Str::limit($order->product->product_title ?? 'N/A', 20) }}</span>
                                    </div>
                                </td>
                                <td class="px-4 py-3.5 text-sm text-gray-900 dark:text-gray-100 text-center align-middle">
                                    {{ $order->quantity }}</td>
                                <td
                                    class="px-4 py-3.5 text-sm font-semibold text-green-600 dark:text-green-400 text-center align-middle">
                                    {{ $order->formatted_total_price ?? 'Rp ' . number_format($order->total_price, 0, ',', '.') }}
                                </td>

                                <!-- Bukti Transfer -->
                                <td class="px-4 py-3.5 text-center align-middle">
                                    @php
                                        // Gunakan Storage facade untuk pengecekan file yang lebih akurat
                                        // File bukti pembayaran disimpan di disk 'uploads' dalam folder 'proofs'
                                        $proofExists = false;
                                        $fileUrl = null;

                                        if ($order->proof_file) {
                                            // Cek apakah file ada di storage
                                            $proofExists = \Storage::disk('uploads')->exists($order->proof_file);

                                            if ($proofExists) {
                                                // Untuk admin, gunakan route admin khusus
                                                try {
                                                    $fileUrl = route('admin.master.orders.proof', $order->id);
                                                } catch (\Exception $e) {
                                                    // Fallback 1: coba route user biasa
                                                    try {
                                                        $fileUrl = route('orders.proof', $order->id);
                                                    } catch (\Exception $e2) {
                                                        // Fallback 2: gunakan asset() langsung
                                                        $fileUrl = asset('uploads/' . $order->proof_file);
                                                    }
                                                }
                                            }
                                        }

                                        $extension = $order->proof_file
                                            ? pathinfo($order->proof_file, PATHINFO_EXTENSION)
                                            : '';
                                        $isImage = in_array(strtolower($extension), [
                                            'jpg',
                                            'jpeg',
                                            'png',
                                            'gif',
                                            'webp',
                                        ]);
                                    @endphp
                                    @if ($proofExists && $fileUrl)
                                        <div class="flex items-center justify-center space-x-2">
                                            @if ($isImage)
                                                <a href="{{ $fileUrl }}" target="_blank"
                                                    class="inline-flex items-center px-2.5 py-1 bg-blue-100 dark:bg-blue-900/40 text-blue-700 dark:text-blue-300 rounded-md text-xs font-medium hover:bg-blue-200 dark:hover:bg-blue-900/60 transition-colors duration-200"
                                                    title="Lihat Bukti Pembayaran">
                                                    <i class="fas fa-image mr-1 text-xs"></i>
                                                    Lihat
                                                </a>
                                            @else
                                                <a href="{{ $fileUrl }}" target="_blank"
                                                    class="inline-flex items-center px-2.5 py-1 bg-blue-100 dark:bg-blue-900/40 text-blue-700 dark:text-blue-300 rounded-md text-xs font-medium hover:bg-blue-200 dark:hover:bg-blue-900/60 transition-colors duration-200"
                                                    title="Lihat Bukti Pembayaran">
                                                    <i class="fas fa-file-alt mr-1 text-xs"></i>
                                                    Lihat
                                                </a>
                                            @endif
                                            <a href="{{ $fileUrl }}" download
                                                class="inline-flex items-center px-2.5 py-1 bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300 rounded-md text-xs font-medium hover:bg-green-200 dark:hover:bg-green-900/60 transition-colors duration-200"
                                                title="Download Bukti Pembayaran">
                                                <i class="fas fa-download mr-1 text-xs"></i>
                                                Download
                                            </a>
                                        </div>
                                    @elseif($order->proof_file)
                                        <span class="text-yellow-600 dark:text-yellow-400 italic text-sm"
                                            title="File tidak ditemukan">File hilang</span>
                                    @else
                                        <span class="text-gray-400 dark:text-gray-500 italic text-sm">Belum diupload</span>
                                    @endif
                                </td>

                                <!-- Status Pesanan -->
                                <td class="px-4 py-3.5 text-center align-middle">
                                    @php
                                        $orderStatusClasses = [
                                            'menunggu' =>
                                                'bg-yellow-100 dark:bg-yellow-900/40 text-yellow-700 dark:text-yellow-300',
                                            'diterima' =>
                                                'bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300',
                                            'ditolak' => 'bg-red-100 dark:bg-red-900/40 text-red-700 dark:text-red-300',
                                        ];
                                    @endphp
                                    <form action="{{ route('admin.master.orders.updateStatus', $order->id) }}"
                                        method="POST">
                                        @csrf
                                        @method('PATCH')
                                        <select name="status" onchange="handleStatusChange(this, {{ $order->id }}, '{{ $order->order_status }}')"
                                            class="px-2.5 py-1 text-xs font-medium rounded-md border border-gray-300 dark:border-gray-600 focus:ring-2 focus:ring-blue-500 transition-colors duration-200 bg-white dark:bg-gray-700 text-gray-900 dark:text-white {{ $orderStatusClasses[$order->order_status] ?? 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200' }}">
                                            <option value="menunggu"
                                                {{ $order->order_status == 'menunggu' ? 'selected' : '' }}>Menunggu
                                            </option>
                                            <option value="diterima"
                                                {{ $order->order_status == 'diterima' ? 'selected' : '' }}>Diterima
                                            </option>
                                            <option value="ditolak"
                                                {{ $order->order_status == 'ditolak' ? 'selected' : '' }}>Ditolak</option>
                                        </select>
                                    </form>
                                </td>

                                <!-- Status Pembayaran -->
                                <td class="px-4 py-3.5 text-center align-middle">
                                    @php
                                        $paymentStatusClasses = [
                                            'menunggu' =>
                                                'bg-yellow-100 dark:bg-yellow-900/40 text-yellow-700 dark:text-yellow-300',
                                            'diproses' =>
                                                'bg-blue-100 dark:bg-blue-900/40 text-blue-700 dark:text-blue-300',
                                            'selesai' =>
                                                'bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300',
                                            'gagal' => 'bg-red-100 dark:bg-red-900/40 text-red-700 dark:text-red-300',
                                            'dibatalkan' =>
                                                'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200',
                                        ];
                                    @endphp
                                    <form action="{{ route('admin.master.orders.updatePaymentStatus', $order->id) }}"
                                        method="POST">
                                        @csrf
                                        @method('PATCH')
                                        <select name="payment_status" onchange="this.form.submit()"
                                            class="px-2.5 py-1 text-xs font-medium rounded-md border border-gray-300 dark:border-gray-600 focus:ring-2 focus:ring-blue-500 transition-colors duration-200 bg-white dark:bg-gray-700 text-gray-900 dark:text-white {{ $paymentStatusClasses[$order->status] ?? 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200' }}">
                                            <option value="menunggu" {{ $order->status == 'menunggu' ? 'selected' : '' }}>
                                                Menunggu</option>
                                            <option value="diproses" {{ $order->status == 'diproses' ? 'selected' : '' }}>
                                                Diproses</option>
                                            <option value="selesai" {{ $order->status == 'selesai' ? 'selected' : '' }}>
                                                Selesai</option>
                                            <option value="gagal" {{ $order->status == 'gagal' ? 'selected' : '' }}>Gagal
                                            </option>
                                            <option value="dibatalkan"
                                                {{ $order->status == 'dibatalkan' ? 'selected' : '' }}>Dibatalkan</option>
                                        </select>
                                    </form>
                                </td>

                                <!-- Status Produksi (Read-Only - dikelola via workflow teknisi & supervisor) -->
                                <td class="px-4 py-3.5 text-center align-middle whitespace-nowrap">
                                    @php
                                        $prodColors = [
                                            'menunggu' =>
                                                'bg-yellow-100 dark:bg-yellow-900/40 text-yellow-700 dark:text-yellow-300',
                                            'dalam_proses' =>
                                                'bg-blue-100 dark:bg-blue-900/40 text-blue-700 dark:text-blue-300',
                                            'selesai' =>
                                                'bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300',
                                            'dibatalkan' =>
                                                'bg-red-100 dark:bg-red-900/40 text-red-700 dark:text-red-300',
                                        ];
                                        $prodLabels = [
                                            'menunggu' => 'Menunggu',
                                            'dalam_proses' => 'Dalam Proses',
                                            'selesai' => 'Selesai',
                                            'dibatalkan' => 'Dibatalkan',
                                        ];
                                    @endphp
                                    <span
                                        class="inline-flex items-center px-2.5 py-1 text-xs font-medium rounded-md {{ $prodColors[$order->production_status] ?? 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200' }}">
                                        <i class="fas fa-industry mr-1 text-xs"></i>
                                        {{ $prodLabels[$order->production_status] ?? 'Menunggu' }}
                                    </span>
                                </td>

                                <!-- Pembayaran -->
                                <td class="px-4 py-3.5 align-middle text-center">
                                    <div class="flex flex-col items-center space-y-1">
                                        <span
                                            class="px-2.5 py-1 text-xs font-medium rounded-md bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200">
                                            {{ ucfirst($order->payment_method ?? '-') }}
                                        </span>
                                        @if ($order->payment_method === 'dp')
                                            <span
                                                class="px-2 py-0.5 text-xs font-medium rounded-md bg-orange-100 dark:bg-orange-900/40 text-orange-700 dark:text-orange-300">
                                                DP: {{ number_format($order->dp_percentage ?? 30.0, 0) }}%
                                            </span>
                                        @endif
                                    </div>
                                </td>

                                <!-- Piutang -->
                                <td class="px-4 py-3.5 align-middle text-center">
                                    @if ($order->payment_method === 'dp' && $order->receivable)
                                        <div class="flex flex-col items-center space-y-1">
                                            @if ($order->receivable->payment_status === 'pending')
                                                <span
                                                    class="px-2.5 py-1 text-xs font-medium rounded-md bg-yellow-100 dark:bg-yellow-900/40 text-yellow-700 dark:text-yellow-300">
                                                    Belum Bayar
                                                </span>
                                            @elseif($order->receivable->payment_status === 'partial')
                                                <span
                                                    class="px-2.5 py-1 text-xs font-medium rounded-md bg-blue-100 dark:bg-blue-900/40 text-blue-700 dark:text-blue-300">
                                                    Bayar Sebagian
                                                </span>
                                            @elseif($order->receivable->payment_status === 'paid')
                                                <span
                                                    class="px-2.5 py-1 text-xs font-medium rounded-md bg-purple-100 dark:bg-purple-900/40 text-purple-700 dark:text-purple-300">
                                                    Lunas
                                                </span>
                                            @elseif($order->receivable->payment_status === 'overdue')
                                                <span
                                                    class="px-2.5 py-1 text-xs font-medium rounded-md bg-red-100 dark:bg-red-900/40 text-red-700 dark:text-red-300">
                                                    Jatuh Tempo
                                                </span>
                                            @endif
                                            <a href="{{ route('admin.receivables.show', $order->receivable) }}"
                                                class="inline-flex items-center px-3 py-1 text-xs font-semibold rounded-full bg-orange-100 dark:bg-orange-900/40 text-orange-700 dark:text-orange-300 border border-orange-200 dark:border-orange-700 hover:bg-orange-200 dark:hover:bg-orange-800 hover:text-orange-900 dark:hover:text-orange-200 transition-colors duration-200 shadow-sm"
                                                title="Kelola pembayaran DP / piutang pesanan ini">
                                                <i class="fas fa-file-invoice-dollar mr-1 text-xs"></i>
                                                Kelola Piutang
                                            </a>
                                        </div>
                                    @else
                                        <span class="text-gray-400 dark:text-gray-500">-</span>
                                    @endif
                                </td>

                                <!-- Status Sale -->
                                <td class="px-4 py-3.5 align-middle text-center">
                                    @if ($order->sale)
                                        <div class="inline-flex items-center justify-center gap-2">
                                            <span
                                                class="inline-flex items-center px-2.5 py-1 text-xs font-medium rounded-md bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300">
                                                <i class="fas fa-check-circle mr-1 text-xs"></i>
                                                Sudah Jadi Sale
                                            </span>
                                            <a href="{{ route('admin.transaksi.sales.show', $order->sale) }}"
                                                class="inline-flex items-center px-2 py-0.5 text-xs font-medium text-emerald-600 dark:text-emerald-400 hover:text-emerald-800 dark:hover:text-emerald-300 hover:bg-emerald-50 dark:hover:bg-emerald-900/20 rounded transition-colors duration-200"
                                                title="Lihat detail penjualan">
                                                <i class="fas fa-chart-line mr-1 text-xs"></i>
                                                Penjualan
                                            </a>
                                        </div>
                                    @else
                                        <span
                                            class="inline-flex items-center px-2.5 py-1 text-xs font-medium rounded-md bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200">
                                            <i class="fas fa-clock mr-1 text-xs"></i>
                                            Belum Jadi Sale
                                        </span>
                                    @endif
                                </td>

                                <!-- Status (Dibaca/Belum Dibaca) -->
                                <td class="px-4 py-3.5 text-center align-middle">
                                    @php
                                        $isRead = $order->is_read ?? false;
                                    @endphp
                                    @if ($isRead)
                                        <span
                                            class="inline-flex items-center px-2.5 py-1 text-xs font-medium rounded-md bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300">
                                            <i class="fas fa-check-circle mr-1 text-xs"></i>
                                            Dibaca
                                        </span>
                                    @else
                                        <span
                                            class="inline-flex items-center px-2.5 py-1 text-xs font-medium rounded-md bg-red-100 dark:bg-red-900/40 text-red-700 dark:text-red-300">
                                            <i class="fas fa-eye-slash mr-1 text-xs"></i>
                                            Belum Dibaca
                                        </span>
                                    @endif
                                </td>

                                <!-- Tanggal -->
                                <td class="px-4 py-3.5 text-sm text-gray-500 dark:text-gray-400 align-middle text-center">
                                    {{ $order->created_at->format('d M Y H:i') }}
                                </td>

                                <!-- Aksi -->
                                <td class="px-4 py-3.5 text-center align-middle">
                                    <div class="flex items-center justify-center space-x-2">
                                        <a href="{{ route('admin.master.orders.show', $order->id) }}"
                                            class="inline-flex items-center justify-center w-9 h-9 text-indigo-600 dark:text-indigo-400 hover:text-indigo-700 dark:hover:text-indigo-300 hover:bg-indigo-50 dark:hover:bg-indigo-900/20 rounded transition-colors duration-200"
                                            title="Detail">
                                            <i class="fas fa-eye text-base"></i>
                                        </a>
                                        {{-- Tombol ke Menu Produksi: muncul jika order diterima & pembayaran selesai/diproses --}}
                                        @php
                                            $canGoToProduction =
                                                $order->order_status === 'diterima' &&
                                                (($order->payment_method === 'transfer' &&
                                                    $order->status === 'selesai') ||
                                                    ($order->payment_method === 'dp' &&
                                                        in_array($order->status, ['selesai', 'diproses'])) ||
                                                    ($order->payment_method === 'cod' && $order->status === 'selesai'));
                                        @endphp
                                        @if ($canGoToProduction)
                                            @if ($order->production)
                                                <a href="{{ route('admin.productions.show', $order->production->id) }}"
                                                    class="inline-flex items-center justify-center w-9 h-9 text-purple-600 dark:text-purple-400 hover:text-purple-700 dark:hover:text-purple-300 hover:bg-purple-50 dark:hover:bg-purple-900/20 rounded transition-colors duration-200"
                                                    title="Lihat Produksi">
                                                    <i class="fas fa-industry text-base"></i>
                                                </a>
                                            @else
                                                <span
                                                    class="inline-flex items-center justify-center w-9 h-9 text-gray-400 dark:text-gray-500"
                                                    title="Produksi akan dibuat otomatis saat order diterima">
                                                    <i class="fas fa-info-circle text-base"></i>
                                                </span>
                                            @endif
                                        @endif
                                        @if (!$order->is_read)
                                            <button type="button" data-id="{{ $order->id }}"
                                                onclick="markOrderAsRead(this)"
                                                class="inline-flex items-center justify-center w-9 h-9 text-orange-600 dark:text-orange-400 hover:text-orange-700 dark:hover:text-orange-300 hover:bg-orange-50 dark:hover:bg-orange-900/20 rounded transition-colors duration-200"
                                                title="Tandai Dibaca">
                                                <i class="fas fa-check text-base"></i>
                                            </button>
                                        @endif
                                        {{-- Tombol Hapus: hanya untuk pesanan yang ditolak --}}
                                        @if ($order->order_status === 'ditolak' && !$order->sale)
                                            <button type="button" data-id="{{ $order->id }}"
                                                data-customer="{{ $order->customer_name }}"
                                                onclick="openDeleteModal(this)"
                                                class="inline-flex items-center justify-center w-9 h-9 text-red-600 dark:text-red-400 hover:text-red-700 dark:hover:text-red-300 hover:bg-red-50 dark:hover:bg-red-900/20 rounded transition-colors duration-200"
                                                title="Hapus Pesanan">
                                                <i class="fas fa-trash text-base"></i>
                                            </button>
                                        @endif
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="17" class="px-6 py-16 text-center">
                                    <div class="flex flex-col items-center">
                                        <div
                                            class="w-16 h-16 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center mb-4">
                                            <i class="fas fa-shopping-cart text-2xl text-gray-400 dark:text-gray-500"></i>
                                        </div>
                                        <p class="text-gray-500 dark:text-gray-400 text-lg font-medium">Belum ada pesanan
                                        </p>
                                        <p class="text-gray-400 dark:text-gray-500 text-sm mt-1">Mulai dengan menambahkan
                                            pesanan pertama</p>
                                    </div>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
            {{-- Pagination (Inside Container, Tipis) --}}
            @if ($orders->hasPages())
                <div
                    class="flex items-center justify-between px-4 py-3 border-t border-gray-200 dark:border-gray-700 text-sm">

                    {{-- Info --}}
                    <div class="text-gray-600 dark:text-gray-400">
                        Menampilkan
                        <span class="font-medium">{{ $orders->firstItem() }}</span>
                        –
                        <span class="font-medium">{{ $orders->lastItem() }}</span>
                        dari
                        <span class="font-medium">{{ $orders->total() }}</span>
                        data
                    </div>

                    {{-- Pagination --}}
                    <div>
                        {{ $orders->onEachSide(1)->links() }}
                    </div>
                </div>
            @endif

        </div>

    </div>

    <!-- Create Tab -->
    <div id="createContent" class="tab-content hidden">
        <!-- Form Container -->
        <div
            class="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 overflow-hidden">
            <div class="p-6">
                <form action="{{ route('admin.master.orders.store') }}" method="POST" class="space-y-6">
                    @csrf

                    <!-- Customer Information -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label for="customer_name"
                                class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                Nama Customer <span class="text-red-500">*</span>
                            </label>
                            <input type="text" id="customer_name" name="customer_name"
                                value="{{ old('customer_name') }}"
                                class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white @error('customer_name') border-red-500 @enderror"
                                placeholder="Masukkan nama customer" required>
                            @error('customer_name')
                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                            @enderror
                        </div>

                        <div>
                            <label for="customer_email"
                                class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                Email Customer <span class="text-red-500">*</span>
                            </label>
                            <input type="email" id="customer_email" name="customer_email"
                                value="{{ old('customer_email') }}"
                                class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white @error('customer_email') border-red-500 @enderror"
                                placeholder="customer@example.com" required>
                            @error('customer_email')
                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label for="phone_number"
                                class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                Nomor Telepon <span class="text-red-500">*</span>
                            </label>
                            <input type="text" id="phone_number" name="phone_number"
                                value="{{ old('phone_number') }}"
                                class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white @error('phone_number') border-red-500 @enderror"
                                placeholder="081234567890" required>
                            @error('phone_number')
                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                            @enderror
                        </div>

                        <div>
                            <label for="payment_method"
                                class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                Metode Pembayaran <span class="text-red-500">*</span>
                            </label>
                            <select id="payment_method" name="payment_method"
                                class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white @error('payment_method') border-red-500 @enderror"
                                required>
                                <option value="">Pilih metode pembayaran</option>
                                <option value="transfer" {{ old('payment_method') == 'transfer' ? 'selected' : '' }}>
                                    Transfer Bank</option>
                                <option value="midtrans" {{ old('payment_method') == 'midtrans' ? 'selected' : '' }}>Midtrans (Online Payment)</option>
                                <option value="dp" {{ old('payment_method') == 'dp' ? 'selected' : '' }}>Pembayaran DP
                                    (Down Payment)</option>
                            </select>
                            @error('payment_method')
                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>

                    <div>
                        <label for="address" class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                            Alamat Lengkap <span class="text-red-500">*</span>
                        </label>
                        <textarea id="address" name="address" rows="3"
                            class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white @error('address') border-red-500 @enderror"
                            placeholder="Masukkan alamat lengkap customer" required>{{ old('address') }}</textarea>
                        @error('address')
                            <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- Product Information -->
                    <div class="border-t border-gray-200 dark:border-gray-600 pt-6">
                        <h3 class="text-lg font-semibold text-gray-800 dark:text-white mb-4">Informasi Produk</h3>

                        <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <div>
                                <label for="product_id"
                                    class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    Pilih Produk <span class="text-red-500">*</span>
                                </label>
                                <select id="product_id" name="product_id"
                                    class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white @error('product_id') border-red-500 @enderror"
                                    required>
                                    <option value="">Pilih produk</option>
                                    @foreach ($products as $product)
                                        <option value="{{ $product->id }}" data-price="{{ $product->product_prices }}"
                                            {{ old('product_id') == $product->id ? 'selected' : '' }}>
                                            {{ $product->product_title }} - Rp
                                            {{ number_format($product->product_prices, 0, ',', '.') }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('product_id')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>

                            <div>
                                <label for="quantity"
                                    class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    Jumlah <span class="text-red-500">*</span>
                                </label>
                                <input type="number" id="quantity" name="quantity" value="{{ old('quantity', 1) }}"
                                    min="1"
                                    class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white @error('quantity') border-red-500 @enderror"
                                    required>
                                @error('quantity')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>

                            <div>
                                <label for="order_status"
                                    class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    Status Pesanan <span class="text-red-500">*</span>
                                </label>
                                <select id="order_status" name="order_status"
                                    class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white @error('order_status') border-red-500 @enderror"
                                    required>
                                    <option value="">Pilih status pesanan</option>
                                    <option value="menunggu" {{ old('order_status') == 'menunggu' ? 'selected' : '' }}>
                                        Menunggu</option>
                                    <option value="diterima" {{ old('order_status') == 'diterima' ? 'selected' : '' }}>
                                        Diterima</option>
                                    <option value="ditolak" {{ old('order_status') == 'ditolak' ? 'selected' : '' }}>
                                        Ditolak</option>
                                </select>
                                @error('order_status')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>

                        <!-- Price Display -->
                        <div class="mt-4 p-4 bg-gray-50 dark:bg-gray-700 rounded-lg">
                            <div class="flex justify-between items-center">
                                <span class="text-sm font-medium text-gray-600 dark:text-gray-400">Total Harga:</span>
                                <span id="total-price" class="text-lg font-bold text-green-600 dark:text-green-400">
                                    Rp 0
                                </span>
                            </div>
                        </div>
                    </div>

                    <!-- Submit Button -->
                    <div class="flex justify-end space-x-3 pt-6 border-t border-gray-200 dark:border-gray-600">
                        <button type="button" onclick="toggleTab('list')"
                            class="px-6 py-3 bg-gray-500 hover:bg-gray-600 text-white font-semibold rounded-lg transition duration-200">
                            Batal
                        </button>
                        <button type="submit"
                            class="px-6 py-3 bg-blue-600 hover:bg-blue-700 text-white font-semibold rounded-lg transition duration-200">
                            <i class="fas fa-save mr-2"></i>
                            Simpan Pesanan
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- JavaScript untuk tab functionality dan price calculation -->
    <script>
        // Tab functionality untuk create form
        function toggleTab(tab) {
            const listContent = document.getElementById('listContent');
            const createContent = document.getElementById('createContent');

            if (tab === 'create') {
                // Show create form
                createContent.classList.remove('hidden');
                listContent.classList.add('hidden');
            } else {
                // Show list tab (default)
                listContent.classList.remove('hidden');
                createContent.classList.add('hidden');
            }
        }

        // Price calculation untuk form create
        document.addEventListener('DOMContentLoaded', function() {
            const productSelect = document.getElementById('product_id');
            const quantityInput = document.getElementById('quantity');
            const totalPriceElement = document.getElementById('total-price');

            if (productSelect && quantityInput && totalPriceElement) {
                function calculateTotal() {
                    const selectedOption = productSelect.options[productSelect.selectedIndex];
                    const price = selectedOption ? parseFloat(selectedOption.dataset.price) : 0;
                    const quantity = parseInt(quantityInput.value) || 0;
                    const total = price * quantity;

                    totalPriceElement.textContent = 'Rp ' + total.toLocaleString('id-ID');
                }

                productSelect.addEventListener('change', calculateTotal);
                quantityInput.addEventListener('input', calculateTotal);

                // Hitung total saat halaman dimuat
                calculateTotal();
            }
        });

        // Mark Order as Read
        function markOrderAsRead(button) {
            const orderId = button.getAttribute('data-id');

            if (!orderId) {
                alert('ID pesanan tidak ditemukan');
                return;
            }

            // Disable button saat proses
            button.disabled = true;
            button.innerHTML = '<i class="fas fa-spinner fa-spin text-sm"></i>';

            fetch(`/admin/master/orders/${orderId}/read`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Reload halaman untuk update tampilan
                        window.location.reload();
                    } else {
                        alert('Gagal menandai pesanan sebagai dibaca: ' + (data.message || 'Terjadi kesalahan'));
                        button.disabled = false;
                        button.innerHTML = '<i class="fas fa-check text-sm"></i>';
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Terjadi kesalahan saat menandai pesanan sebagai dibaca');
                    button.disabled = false;
                    button.innerHTML = '<i class="fas fa-check text-sm"></i>';
                });
        }

        // Submit filter dengan parameter yang bersih
        function submitFilter() {
            const params = new URLSearchParams();

            // Ambil nilai dari setiap input/select
            const search = document.getElementById('searchInput')?.value?.trim() || '';
            const orderStatus = document.getElementById('orderStatusFilter')?.value || '';
            const paymentStatus = document.getElementById('paymentStatusFilter')?.value || '';
            const productionStatus = document.getElementById('productionStatusFilter')?.value || '';
            const paymentMethod = document.getElementById('paymentMethodFilter')?.value || '';
            const orderType = document.getElementById('orderTypeFilter')?.value || '';
            const tab = new URLSearchParams(window.location.search).get('tab') || 'all';

            // Hanya tambahkan parameter yang memiliki nilai
            if (search) {
                params.append('search', search);
            }
            if (orderStatus) {
                params.append('order_status', orderStatus);
            }
            if (paymentStatus) {
                params.append('payment_status', paymentStatus);
            }
            if (productionStatus) {
                params.append('production_status', productionStatus);
            }
            if (paymentMethod) {
                params.append('payment_method', paymentMethod);
            }
            if (orderType) {
                params.append('order_type', orderType);
            }
            if (tab && tab !== 'all') {
                params.append('tab', tab);
            }

            // Redirect dengan parameter yang bersih
            const url = '{{ route('admin.master.orders.index') }}' + (params.toString() ? '?' + params.toString() : '');
            window.location.href = url;
        }

        // Clear search only
        function clearSearch() {
            const url = new URL(window.location);
            url.searchParams.delete('search');
            window.location.href = url.toString();
        }

        // Clear individual filter
        function clearFilter(filterName) {
            const url = new URL(window.location);
            url.searchParams.delete(filterName);
            window.location.href = url.toString();
        }

        // Clear all filters
        function clearAllFilters() {
            const tab = new URLSearchParams(window.location.search).get('tab');
            const url = '{{ route('admin.master.orders.index') }}' + (tab && tab !== 'all' ? '?tab=' + tab : '');
            window.location.href = url;
        }

        // Submit form search dengan Enter
        document.addEventListener('DOMContentLoaded', function() {
            const searchInput = document.getElementById('searchInput');
            if (searchInput) {
                searchInput.addEventListener('keypress', function(e) {
                    if (e.key === 'Enter') {
                        e.preventDefault();
                        submitFilter();
                    }
                });
            }
        });
    </script>

    <style>
        /* Konsisten Style untuk semua Index Pages */
        .form-input-search,
        .form-select {
            height: 40px !important;
            min-height: 40px !important;
            max-height: 40px !important;
            box-sizing: border-box !important;
            line-height: 1.5 !important;
            padding: 0 2.5rem 0 2.5rem !important;
            margin: 0 !important;
        }

        .form-input-search {
            width: 100%;
            padding-left: 2.5rem !important;
            padding-right: 2.5rem !important;
            border: 1px solid rgb(209 213 219);
            border-radius: 0.5rem;
            background-color: white;
            color: rgb(17 24 39);
            font-size: 0.875rem;
            transition: all 0.2s;
        }

        .dark .form-input-search {
            border-color: rgb(75 85 99);
            background-color: rgb(55 65 81);
            color: white;
        }

        .form-input-search:focus {
            outline: none;
            ring: 2px;
            ring-color: rgb(59 130 246);
        }

        .form-select {
            width: 100%;
            padding-left: 1rem !important;
            padding-right: 2.5rem !important;
            border: 1px solid rgb(209 213 219);
            border-radius: 0.5rem;
            background-color: white;
            color: rgb(17 24 39);
            font-size: 0.875rem;
            appearance: none;
            -webkit-appearance: none;
            -moz-appearance: none;
            transition: all 0.2s;
        }

        .dark .form-select {
            border-color: rgb(75 85 99);
            background-color: rgb(55 65 81);
            color: white;
        }

        .form-select:focus {
            outline: none;
            ring: 2px;
            ring-color: rgb(59 130 246);
        }

        .action-btn-primary {
            height: 40px !important;
            min-height: 40px !important;
            max-height: 40px !important;
            box-sizing: border-box !important;
            padding: 0 1rem !important;
            margin: 0 !important;
            display: inline-flex !important;
            align-items: center !important;
            justify-content: center !important;
            font-weight: 500;
            border-radius: 0.5rem;
            transition: all 0.2s;
            white-space: nowrap;
            background-color: rgb(59 130 246);
            color: white;
            text-decoration: none;
            border: none;
            cursor: pointer;
        }

        .action-btn-primary:hover {
            background-color: rgb(37 99 235);
        }

        .action-btn-primary:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        .dark .action-btn-primary {
            background-color: rgb(59 130 246);
        }

        .dark .action-btn-primary:hover {
            background-color: rgb(37 99 235);
        }
    </style>

<!-- Modal Penolakan Pesanan -->
<div id="rejectionModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50 items-center justify-center">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-md w-full">
            <div class="p-6">
                <div class="flex items-center mb-4">
                    <div class="flex-shrink-0 w-10 h-10 bg-red-100 dark:bg-red-900/40 rounded-full flex items-center justify-center mr-3">
                        <i class="fas fa-times-circle text-red-600 dark:text-red-400"></i>
                    </div>
                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white">Tolak Pesanan</h3>
                </div>
                <form id="rejectionForm" method="POST">
                    @csrf
                    @method('PATCH')
                    <input type="hidden" name="status" value="ditolak">
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Alasan Penolakan <span class="text-red-500">*</span>
                        </label>
                        <textarea name="rejection_reason" rows="4" required
                                class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-red-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm"
                                placeholder="Berikan alasan mengapa pesanan ditolak..."></textarea>
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Alasan ini akan dikirim ke customer melalui notifikasi.</p>
                    </div>
                    <div class="flex items-center justify-end space-x-3">
                        <button type="button" onclick="closeRejectionModal()"
                                class="px-4 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                            Batal
                        </button>
                        <button type="submit"
                                class="px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg transition-colors text-sm font-medium">
                            <i class="fas fa-times mr-2"></i>
                            Tolak Pesanan
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
let currentOrderId = null;
let currentOrderStatus = null;

function handleStatusChange(select, orderId, currentStatus) {
    const selectedStatus = select.value;
    
    if (selectedStatus === 'ditolak') {
        // Simpan order ID dan status sebelumnya
        currentOrderId = orderId;
        currentOrderStatus = currentStatus;
        
        // Buka modal rejection
        openRejectionModal(orderId);
        
        // Reset select ke status sebelumnya
        select.value = currentStatus;
    } else {
        // Submit langsung untuk status lain
        const form = select.closest('form');
        if (form) {
            form.submit();
        }
    }
}

function openRejectionModal(orderId) {
    const modal = document.getElementById('rejectionModal');
    const form = document.getElementById('rejectionForm');
    
    if (modal && form) {
        // Set form action
        form.action = '{{ route("admin.master.orders.updateStatus", ":id") }}'.replace(':id', orderId);
        
        // Show modal
        modal.classList.remove('hidden');
        modal.classList.add('flex');
        document.body.style.overflow = 'hidden';
    }
}

function closeRejectionModal() {
    const modal = document.getElementById('rejectionModal');
    const form = document.getElementById('rejectionForm');
    
    if (modal) {
        modal.classList.add('hidden');
        modal.classList.remove('flex');
        document.body.style.overflow = 'auto';
        
        // Reset form
        if (form) {
            form.reset();
        }
    }
}

// Close modal on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeRejectionModal();
        closeDeleteModal();
    }
});
</script>

<!-- Modal Konfirmasi Hapus Pesanan -->
<div id="deleteModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50 items-center justify-center">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-md w-full">
            <div class="p-6">
                <div class="flex items-center mb-4">
                    <div class="flex-shrink-0 w-10 h-10 bg-red-100 dark:bg-red-900/40 rounded-full flex items-center justify-center mr-3">
                        <i class="fas fa-exclamation-triangle text-red-600 dark:text-red-400"></i>
                    </div>
                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white">Hapus Pesanan</h3>
                </div>
                <p class="text-sm text-gray-600 dark:text-gray-300 mb-4">
                    Apakah Anda yakin ingin menghapus pesanan ini? Tindakan ini tidak dapat dibatalkan.
                </p>
                <div class="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-700 rounded-lg p-3 mb-4">
                    <p class="text-xs text-yellow-800 dark:text-yellow-200">
                        <i class="fas fa-info-circle mr-1"></i>
                        <strong>Pesanan:</strong> <span id="deleteCustomerName" class="font-medium"></span>
                    </p>
                </div>
                <div class="flex items-center justify-end space-x-3">
                    <button type="button" onclick="closeDeleteModal()"
                            class="px-4 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                        Batal
                    </button>
                    <button type="button" onclick="confirmDeleteOrder()"
                            class="px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg transition-colors text-sm font-medium">
                        <i class="fas fa-trash mr-2"></i>
                        Hapus Pesanan
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
let orderIdToDelete = null;

function openDeleteModal(button) {
    const orderId = button.getAttribute('data-id');
    const customerName = button.getAttribute('data-customer');
    const modal = document.getElementById('deleteModal');
    const customerNameElement = document.getElementById('deleteCustomerName');
    
    if (modal && customerNameElement) {
        orderIdToDelete = orderId;
        customerNameElement.textContent = customerName || `#${orderId}`;
        
        // Show modal
        modal.classList.remove('hidden');
        modal.classList.add('flex');
        document.body.style.overflow = 'hidden';
    }
}

function closeDeleteModal() {
    const modal = document.getElementById('deleteModal');
    
    if (modal) {
        modal.classList.add('hidden');
        modal.classList.remove('flex');
        document.body.style.overflow = 'auto';
        orderIdToDelete = null;
    }
}

function confirmDeleteOrder() {
    if (!orderIdToDelete) {
        alert('ID pesanan tidak ditemukan');
        return;
    }

    // Disable button saat proses
    const deleteButton = document.querySelector('#deleteModal button[onclick="confirmDeleteOrder()"]');
    const originalContent = deleteButton.innerHTML;
    deleteButton.disabled = true;
    deleteButton.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Menghapus...';

    fetch(`/admin/master/orders/${orderIdToDelete}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Tampilkan notifikasi sukses
                if (typeof showNotification === 'function') {
                    showNotification('success', data.message || 'Pesanan berhasil dihapus.');
                }
                
                // Reload halaman setelah 500ms untuk update tampilan
                setTimeout(() => {
                    window.location.reload();
                }, 500);
            } else {
                alert('Gagal menghapus pesanan: ' + (data.message || 'Terjadi kesalahan'));
                deleteButton.disabled = false;
                deleteButton.innerHTML = originalContent;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Terjadi kesalahan saat menghapus pesanan');
            deleteButton.disabled = false;
            deleteButton.innerHTML = originalContent;
        });
}

// Close modal saat klik di luar modal
document.addEventListener('click', function(e) {
    const deleteModal = document.getElementById('deleteModal');
    if (deleteModal && e.target === deleteModal) {
        closeDeleteModal();
    }
});
</script>
@endsection
